//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Apr 29, 2008
//---------------------

package org.cleversafe.config.evaluator;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.List;

import org.cleversafe.UnitTests;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.optimizedcauchy.CauchyInformationDispersalCodec;
import org.cleversafe.layer.grid.WriteControllerFactory;
import org.cleversafe.layer.grid.smartcontroller.SmartWriteControllerFactory;
import org.cleversafe.serialization.ProtocolMessageFactory;
import org.cleversafe.serialization.raw.RawGridProtocolMessageFactory;
import org.cleversafe.test.BaseTest;
import org.cleversafe.util.Tuple2;
import org.cleversafe.util.Version;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public class XMLEvaluatorTest extends BaseTest
{
   public static class MethodObject
   {
      private final String methodValue;

      public MethodObject(String methodValue)
      {
         this.methodValue = methodValue;
      }

      public String getMethodValue()
      {
         return this.methodValue;
      }

      public Version getVersion(String versionString)
      {
         return Version.fromString(versionString);
      }

      public void setThreshold(int t)
      {

      }

      public void setNumSlices(int n)
      {

      }

      public void setParam3(String methodValue)
      {

      }

      public void setParam4(InformationDispersalCodec ida)
      {

      }

      public void setParam5(InformationDispersalCodec ida)
      {

      }

      public void setParam6(ProtocolMessageFactory pmf)
      {

      }

      public void setParam7(WriteControllerFactory wcf)
      {

      }

      public void setParam8(Object obj)
      {

      }
   }

   private Literal createIntLiteral(int value)
   {
      Literal literal = null;
      literal = Literal.Factory.newInstance();
      literal.setType(SupportedTypes.INT);
      literal.setValue(String.valueOf(value));

      return literal;
   }

   private Parameter createIntParameter(String name, int value)
   {
      Literal literal = createIntLiteral(value);
      Parameter param = Parameter.Factory.newInstance();
      param.setLiteral(literal);
      param.setName(name);
      return param;
   }

   private Parameter createStringParameter(String name, String value)
   {
      Literal literal = createStringLiteral(value);
      Parameter param = Parameter.Factory.newInstance();
      param.setLiteral(literal);
      param.setName(name);
      return param;
   }

   private Literal createStringLiteral(String value)
   {
      Literal literal = null;
      literal = Literal.Factory.newInstance();
      literal.setType(SupportedTypes.STRING);
      literal.setValue(value);

      return literal;
   }

   private Reference createReference(String[] kind, String referral, Version version)
   {
      Reference ref = Reference.Factory.newInstance();

      String fullKindName = "";
      for (int i = 0; i < kind.length; i++)
      {
         fullKindName += kind[i];
         if (i != kind.length - 1)
            fullKindName += ".";
      }

      ref.setKind(fullKindName);

      if (referral != null)
         ref.setReferral(referral);

      if (version != null)
         ref.setVersion(version.toString());

      return ref;
   }

   private Reference createReference(Class<?> cls, String referral, Version version)
   {
      Reference ref = Reference.Factory.newInstance();

      ref.setInterface(cls.getName());

      if (referral != null)
         ref.setReferral(referral);

      if (version != null)
         ref.setVersion(version.toString());

      return ref;
   }

   private Tuple2<Reference, ExecutionContext> createReference(Object object)
   {
      Reference ref = Reference.Factory.newInstance();

      ExecutionContext ctx = new ExecutionContext();
      ctx.add("myObj1", object);

      ref.setObject("myObj1");

      return new Tuple2<Reference, ExecutionContext>(ref, ctx);
   }

   private Tuple2<Method, ExecutionContext> createStringMethod(String methodValue)
   {
      MethodObject methodObj = new MethodObject(methodValue);

      ExecutionContext ctx = new ExecutionContext();
      ctx.add("methodObj", methodObj);

      Method method = Method.Factory.newInstance();
      method.setObject("methodObj");
      method.setName("getMethodValue");

      return new Tuple2<Method, ExecutionContext>(method, ctx);
   }

   private Tuple2<Method, ExecutionContext> createVersionMethod(String versionString)
   {
      MethodObject methodObj = new MethodObject("doesn't matter");

      ExecutionContext ctx = new ExecutionContext();
      ctx.add("methodObj", methodObj);

      Method method = Method.Factory.newInstance();
      Literal versionStringLiteral = createStringLiteral(versionString);
      Argument versionStringArgument = method.addNewArg();
      versionStringArgument.setLiteral(versionStringLiteral);
      method.setObject("methodObj");
      method.setName("getVersion");

      return new Tuple2<Method, ExecutionContext>(method, ctx);
   }

   /**
    * @throws java.lang.Exception
    */
   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      UnitTests.resetConfiguration();
   }

   /**
    * @throws java.lang.Exception
    */
   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
      UnitTests.resetConfiguration();
   }

   /**
    * @throws java.lang.Exception
    */
   @Before
   public void setUp() throws Exception
   {
   }

   /**
    * @throws java.lang.Exception
    */
   @After
   public void tearDown() throws Exception
   {
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseSimpleArgument(org.cleversafe.config.evaluator.SimpleArgument)}.
    */
   @Test
   public void testParseSimpleArgument()
   {
      Literal literal = createIntLiteral(100);
      SimpleArgument arg = SimpleArgument.Factory.newInstance();
      arg.setLiteral(literal);

      try
      {
         assertEquals(XMLEvaluator.parseSimpleArgument(arg).evaluate(), 100);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseSimpleParameter(org.cleversafe.config.evaluator.SimpleParameter)}.
    */
   @Test
   public void testParseSimpleParameter()
   {
      Literal literal = createIntLiteral(100);
      SimpleParameter param = SimpleParameter.Factory.newInstance();
      param.setLiteral(literal);
      param.setName("my parameter");

      try
      {
         NamedEvaluator eval = XMLEvaluator.parseSimpleParameter(param);
         assertEquals(eval.getName(), "my parameter");
         assertEquals(eval.evaluate(), 100);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseArgument(org.cleversafe.config.evaluator.Argument, org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseLiteralArgument()
   {
      Literal literal = createStringLiteral("my literal argument");
      Argument arg = Argument.Factory.newInstance();
      arg.setLiteral(literal);

      try
      {
         assertEquals(XMLEvaluator.parseArgument(arg, null).evaluate(), "my literal argument");
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseArgument(org.cleversafe.config.evaluator.Argument, org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseReferenceArgument()
   {
      Reference ref = createReference(new String[]{
         "IDA"
      }, "cauchy", Version.fromString("1.0"));

      Parameter param1 = createIntParameter("threshold", 6);
      Parameter param2 = createIntParameter("num-slices", 8);
      Parameter[] params = new Parameter[]{
            param1, param2
      };
      ref.setParamArray(params);

      Argument arg = Argument.Factory.newInstance();
      arg.setReference(ref);

      try
      {
         Evaluatable eval = XMLEvaluator.parseArgument(arg, null);
         assertEquals(eval.evaluate().getClass(), CauchyInformationDispersalCodec.class);
         CauchyInformationDispersalCodec ida = (CauchyInformationDispersalCodec) eval.evaluate();
         assertEquals(ida.getNumSlices(), 8);
         assertEquals(ida.getThreshold(), 6);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseArgument(org.cleversafe.config.evaluator.Argument, org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseMethodArgument()
   {
      Tuple2<Method, ExecutionContext> tuple = createStringMethod("my method argument");
      Argument arg = Argument.Factory.newInstance();
      arg.setMethod(tuple.getFirst());

      try
      {
         Evaluatable eval = XMLEvaluator.parseArgument(arg, tuple.getSecond());
         assertEquals(eval.evaluate(), "my method argument");
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseLiteral(org.cleversafe.config.evaluator.Literal)}.
    */
   @Test
   public void testParseLiteral()
   {
      Literal literal = createIntLiteral(45);
      try
      {
         assertEquals(XMLEvaluator.parseLiteral(literal).evaluate(), 45);
      }
      catch (IllegalConfigurationContentException e)
      {
         fail(e.getMessage());
      }

      Literal literal2 = createStringLiteral("my value");

      try
      {
         assertEquals(XMLEvaluator.parseLiteral(literal2).evaluate(), "my value");
      }
      catch (IllegalConfigurationContentException e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseParameterArray(org.cleversafe.config.evaluator.Parameter[], org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseParameterArray()
   {
      Parameter param1 = createIntParameter("threshold", 6);
      Parameter param2 = createIntParameter("num-slices", 8);
      Parameter param3 = createStringParameter("my-new-param", "my new param value");
      Parameter[] params = new Parameter[]{
            param1, param2, param3
      };

      try
      {
         List<NamedEvaluator> evals = XMLEvaluator.parseParameterArray(params, null);
         assertEquals(evals.get(0).getName(), "threshold");
         assertEquals(evals.get(0).evaluate(), 6);

         assertEquals(evals.get(1).getName(), "num-slices");
         assertEquals(evals.get(1).evaluate(), 8);

         assertEquals(evals.get(2).getName(), "my-new-param");
         assertEquals(evals.get(2).evaluate(), "my new param value");
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseParameter(org.cleversafe.config.evaluator.Parameter, org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseLiteralParameter()
   {
      Parameter param1 = createStringParameter("my-new-param", "my new param value");

      try
      {
         NamedEvaluator eval = XMLEvaluator.parseParameter(param1, null);
         assertEquals(eval.getName(), "my-new-param");
         assertEquals(eval.evaluate(), "my new param value");
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseReferenceArray(org.cleversafe.config.evaluator.Reference[], org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseReferenceArray()
   {
      Parameter param1 = createIntParameter("threshold", 6);
      Parameter param2 = createIntParameter("num-slices", 8);
      Parameter[] params1 = new Parameter[]{
            param1, param2
      };

      Reference ref1 = createReference(InformationDispersalCodec.class, "cauchy", null);
      Reference ref2 = createReference(new String[]{
         "IDA"
      }, "cauchy", null);
      ref1.setParamArray(params1);
      ref2.setParamArray(params1);

      Reference ref3 = createReference(new String[]{
         "Serialization"
      }, null, null);

      Reference ref4 = createReference(WriteControllerFactory.class, null, null);

      Reference ref5 = Reference.Factory.newInstance();
      Parameter param3 = Parameter.Factory.newInstance();
      param3.setName("param3");
      Tuple2<Method, ExecutionContext> tuple1 = createStringMethod("ref5param1");
      param3.setMethod(tuple1.getFirst());
      ExecutionContext ctx = tuple1.getSecond();

      Parameter param4 = Parameter.Factory.newInstance();
      param4.setReference(ref1);
      param4.setName("param4");

      Parameter param5 = Parameter.Factory.newInstance();
      param5.setReference(ref2);
      param5.setName("param5");

      Parameter param6 = Parameter.Factory.newInstance();
      param6.setReference(ref3);
      param6.setName("param6");

      Parameter param7 = Parameter.Factory.newInstance();
      param7.setReference(ref4);
      param7.setName("param7");

      Parameter param8 = Parameter.Factory.newInstance();
      Reference p6ref = Reference.Factory.newInstance();
      Object p6refObj = new Object();
      ctx.add("p6refObj", p6refObj);
      p6ref.setObject("p6refObj");
      param8.setReference(p6ref);
      param8.setName("param8");

      Parameter[] ref5ParamArray = new Parameter[]{
            param1, param2, param3, param4, param5, param6, param7, param8
      };
      ref5.setParamArray(ref5ParamArray);
      MethodObject ref5Obj = new MethodObject("");
      ctx.add("ref5Obj", ref5Obj);
      ref5.setObject("ref5Obj");

      Reference[] refs = new Reference[]{
            ref1, ref2, ref3, ref4, ref5
      };

      try
      {
         List<ReferenceEvaluator> evals = XMLEvaluator.parseReferenceArray(refs, ctx);
         Object idaObj1 = evals.get(0).evaluate();
         Object idaObj2 = evals.get(1).evaluate();

         assertEquals(idaObj1.getClass(), CauchyInformationDispersalCodec.class);
         assertEquals(idaObj2.getClass(), CauchyInformationDispersalCodec.class);

         assertEquals(((CauchyInformationDispersalCodec) idaObj1).getNumSlices(), 8);
         assertEquals(((CauchyInformationDispersalCodec) idaObj1).getThreshold(), 6);
         assertEquals(((CauchyInformationDispersalCodec) idaObj2).getNumSlices(), 8);
         assertEquals(((CauchyInformationDispersalCodec) idaObj2).getThreshold(), 6);

         assertEquals(evals.get(2).evaluate().getClass(), RawGridProtocolMessageFactory.class);
         assertEquals(evals.get(3).evaluate().getClass(), SmartWriteControllerFactory.class);

         assertEquals(evals.get(4).evaluate().getClass(), MethodObject.class);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }

   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseReference(org.cleversafe.config.evaluator.Reference, org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseKindReference()
   {
      Reference ref = createReference(new String[]{
         "IDA"
      }, "cauchy", Version.fromString("1.0"));

      Parameter param1 = createIntParameter("threshold", 6);
      Parameter param2 = createIntParameter("num-slices", 8);
      Parameter[] params = new Parameter[]{
            param1, param2
      };
      ref.setParamArray(params);

      try
      {
         Evaluatable eval = XMLEvaluator.parseReference(ref, null);
         assertEquals(eval.evaluate().getClass(), CauchyInformationDispersalCodec.class);
         CauchyInformationDispersalCodec ida = (CauchyInformationDispersalCodec) eval.evaluate();
         assertEquals(ida.getNumSlices(), 8);
         assertEquals(ida.getThreshold(), 6);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseReference(org.cleversafe.config.evaluator.Reference, org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseObjectReference()
   {
      Object object = new Object();

      Tuple2<Reference, ExecutionContext> tuple = createReference(object);

      try
      {
         Evaluatable eval = XMLEvaluator.parseReference(tuple.getFirst(), tuple.getSecond());
         assertTrue(eval.evaluate() == object);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }

   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseReference(org.cleversafe.config.evaluator.Reference, org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseInterfaceReference()
   {
      Reference ref =
            createReference(InformationDispersalCodec.class, "cauchy", Version.fromString("1.0"));

      Parameter param1 = createIntParameter("threshold", 6);
      Parameter param2 = createIntParameter("num-slices", 8);
      Parameter[] params = new Parameter[]{
            param1, param2
      };
      ref.setParamArray(params);

      try
      {
         Evaluatable eval = XMLEvaluator.parseReference(ref, null);
         assertEquals(eval.evaluate().getClass(), CauchyInformationDispersalCodec.class);
         CauchyInformationDispersalCodec ida = (CauchyInformationDispersalCodec) eval.evaluate();
         assertEquals(ida.getNumSlices(), 8);
         assertEquals(ida.getThreshold(), 6);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * Test method for
    * {@link org.cleversafe.config.evaluator.XMLEvaluator#parseMethod(org.cleversafe.config.evaluator.Method, org.cleversafe.config.ExecutionContext)}.
    */
   @Test
   public void testParseMethod()
   {
      Tuple2<Method, ExecutionContext> tuple1 = createStringMethod("my method value");
      Tuple2<Method, ExecutionContext> tuple2 = createVersionMethod("4.0.3");
      try
      {
         Evaluatable eval = XMLEvaluator.parseMethod(tuple1.getFirst(), tuple1.getSecond());
         assertEquals(eval.evaluate(), "my method value");

         Evaluatable eval2 = XMLEvaluator.parseMethod(tuple2.getFirst(), tuple2.getSecond());
         assertEquals(eval2.evaluate(), Version.fromString("4.0.3"));

      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }

   }
}
