//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: Mar 7, 2008
//---------------------

package org.cleversafe.ida.compatibility;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Arrays;
import java.util.List;
import java.util.Random;

import org.cleversafe.codec.exceptions.CodecDecodeException;
import org.cleversafe.codec.exceptions.CodecEncodeException;
import org.cleversafe.codec.exceptions.CodecInvalidDataFormatException;
import org.cleversafe.codec.exceptions.CodecInvalidParametersException;
import org.cleversafe.codec.exceptions.CodecNotInitializedException;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.InformationDispersalDecoder;
import org.cleversafe.ida.InformationDispersalEncoder;
import org.cleversafe.ida.exceptions.IDADecodeException;
import org.cleversafe.ida.exceptions.IDAEncodeException;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;
import org.cleversafe.test.BaseTest;
import org.junit.Test;

public abstract class IDACompatibilityTestBase extends BaseTest
{
   public byte[] originalData;
   public List<byte[]> slices;
   
   public abstract InformationDispersalCodec getCodec() throws CodecInvalidParametersException;
   
   /**
    * This constructor generates and outputs code for declaring a byte array of encoded data
    * 
    * @param dataLength
    * @throws CodecInvalidParametersException 
    */
   public IDACompatibilityTestBase(int dataLength) throws CodecInvalidParametersException
   {
      // Intentional repeatability of output
      generateUnencodedData(dataLength);
      
      // Generate encoded Data
      generateEncodedData();
      
      // Display code
      printEncodedArray(slices);
   }
   
   /**
    * This constrctor takes a hard-coded encoded array, this
    * constructor should be the one called by base tests
    * 
    * @param encodedData
    */
   public IDACompatibilityTestBase(List<byte[]> encodedData, int dataLength)
   {
      this.slices = encodedData;
      generateUnencodedData(dataLength);
   }
   
   public void printEncodedArray(List<byte[]> arrayList)
   {
      for (int idx = 0; idx < arrayList.size(); idx++)
      {
         byte[] array = arrayList.get(idx);
      
         System.out.print("byte[] encoded" + idx + " = new byte[] { ");
         for (int itr = 0; itr < array.length; itr++)
         {
            String appendStr = (itr == array.length - 1) ? " };" :  ", ";
            System.out.print(array[itr] + appendStr);
         }
         System.out.println("");
      }
   }
   
   /**
    * This method generates the set of encoded data, which will be compared
    * against the output of the encoder byte for byte and will also be used
    * to test the decoder.  The first time this method is run an output file
    * is generated, subsequent runs will use the saved output as opposed to
    * the generated output.
    * @throws CodecInvalidParametersException 
    */
   private void generateEncodedData() throws CodecInvalidParametersException
   {
      // Create codec object
      InformationDispersalCodec ida = getCodec();
      assertNotNull(ida);
      
      try
      {
         // Encode data
         InformationDispersalEncoder encoder = ida.getEncoder();
         this.slices = encoder.finish(originalData);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail(e.getMessage());
      }
   }
   
   private void generateUnencodedData(int dataLength)
   {
      // Intentional repeatability of output
      Random rnd = new Random();
      rnd.setSeed(12345);
      this.originalData = new byte[dataLength];
      rnd.nextBytes(this.originalData);
   }
   
   @Test
   public void testEncodeCompatibility() throws CodecInvalidParametersException, CodecEncodeException, CodecInvalidDataFormatException, CodecNotInitializedException, IDAEncodeException, IDANotInitializedException, IDAInvalidParametersException
   {
      // Encode data
      InformationDispersalCodec codec = getCodec();
      assertNotNull(codec);
      InformationDispersalEncoder encoder = codec.getEncoder();
      List<byte[]> newEncodedData = encoder.finish(originalData);
      
      // Ensure they are identical
      assertTrue(slices.size() == newEncodedData.size());
      for (int idx = 0; idx < newEncodedData.size(); idx++)
      {
         assertTrue(Arrays.equals(slices.get(idx), newEncodedData.get(idx)));
      }
   }
   
   @Test
   public void testDecodeCompatibility() throws CodecInvalidParametersException, CodecDecodeException, CodecInvalidDataFormatException, CodecNotInitializedException, IDAInvalidParametersException, IDADecodeException, IDANotInitializedException
   {
      // Decode data
      InformationDispersalCodec ida = getCodec();
      assertNotNull(ida);
      InformationDispersalDecoder decoder = ida.getDecoder();
      byte[] newDecodedData = decoder.finish(slices);
      
      // Ensure they are identical
      assertTrue(Arrays.equals(originalData, newDecodedData));
   }
}
