//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Nov 1, 2007
//---------------------

package org.cleversafe.layer.communication.network.mina;

import static org.junit.Assert.fail;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.extras.DOMConfigurator;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.ConnectorManager;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.exceptions.NotConnectedException;
import org.cleversafe.layer.communication.policy.SimpleConnectorManager;
import org.cleversafe.layer.communication.policy.SmartConnectorManager;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.layer.protocol.NoopResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.serialization.ProtocolMessageFactory;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.ServerApplication;
import org.cleversafe.server.exceptions.ServerIOException;
import org.cleversafe.server.handlers.misc.NoopHandler;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.util.Log4jReloader;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.martiansoftware.jsap.FlaggedOption;
import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Parameter;
import com.martiansoftware.jsap.QualifiedSwitch;
import com.martiansoftware.jsap.SimpleJSAP;

public class MinaConnectorTest
{
   private static Logger _logger = Logger.getLogger(MinaConnectorTest.class);

   public static String HOST = "127.0.0.1";
   public static int PORT = 12345;
   public static int NUM_ITERATIONS = 100;
   public static int NUM_REQUESTS = 10000;
   public static int NUM_CONCURRENT_CONNECTIONS = 50;
   public static int SEND_WAIT_MS = 65000; // Time to wait for send thread to reconnect & send
   public static int NUM_THREADS = 20;
   public static int PAYLOAD_SIZE = 4096;

   private static ProtocolMessageFactory protocolMessageFactory;

   private MinaAcceptor server;
   private AtomicBoolean serverRunning = new AtomicBoolean();
   private Random random = new Random();
   private byte[] payload;

   public MinaConnectorTest()
   {
      this.payload = new byte[PAYLOAD_SIZE];
      this.random.nextBytes(payload);
      this.serverRunning.set(false);
   }

   @BeforeClass
   public static void initOnce() throws ConfigurationItemNotDefinedException,
         ObjectInstantiationException, ObjectInitializationException, ConfigurationLoadException
   {
      // Initialize logging
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));
      Log4jReloader.launch();
      
      protocolMessageFactory =
            ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                  ProtocolMessageFactory.class);
   }

   @Before
   public void setUp()
   {
      this.startServer();
   }

   @After
   public void tearDown()
   {
      this.stopServer();
   }

   private synchronized void startServer()
   {
      if (!this.serverRunning.get())
      {
         try
         {
            _logger.info("Starting server");

            // FIXME: In the future the MinaAcceptor may be reusable
            this.server = new NoopServer(PORT, false, protocolMessageFactory);

            this.server.start();
            this.serverRunning.set(true);
         }
         catch (ServerIOException ex)
         {
            fail("Failed to start server: " + ex.getMessage());
         }
      }
   }

   private synchronized void stopServer()
   {
      if (this.serverRunning.get())
      {
         _logger.info("Stopping server");
         this.server.stop();
         this.serverRunning.set(false);

         // FIXME: In the future the MinaAcceptor may be reusable
         this.server = null;
      }
   }

   /**
    * Test simple transmission w/payload echo
    */
   @Test
   public void simpleTransmission() throws CommunicationIOException, NotConnectedException,
         CommunicationInterruptedException, CommunicationResponseException, FileNotFoundException,
         CommunicationTransmissionException, CommunicationConnectionException
   {
      MinaConnector conn =
            new MinaConnector(HOST, PORT, new SimpleConnectorManager(), protocolMessageFactory);
      conn.ensureConnected();

      NoopRequest req = new NoopRequest();
      req.setPayload(this.payload);

      Response res = conn.exchange(req);
      _logger.info("Response type is: " + res.getClass());

      if (Arrays.equals(((NoopResponse) res).getPayload(), this.payload))
      {
         _logger.info("Payloads are equal");
      }
      else
      {
         _logger.warn("Payloads are not equal");
      }

      _logger.info("Payload size: " + this.payload.length);

      conn.disconnect();
   }

   /**
    * Tests transmission over a number of simultaneous connections
    */
   @Test
   public void multiConnectTransmission() throws CommunicationIOException, NotConnectedException,
         CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException, CommunicationConnectionException
   {
      LinkedList<MinaConnector> conns = new LinkedList<MinaConnector>();
      for (int i = 0; i < NUM_CONCURRENT_CONNECTIONS; i++)
      {
         MinaConnector conn =
               new MinaConnector(HOST, PORT, new SimpleConnectorManager(), protocolMessageFactory);
         conn.ensureConnected();
         conns.add(conn);
      }

      for (int i = 0; i < NUM_CONCURRENT_CONNECTIONS; i++)
      {
         NoopRequest req = new NoopRequest();
         req.setPayload(this.payload);

         Response res = conns.get(i).exchange(req);

         if (!Arrays.equals(((NoopResponse) res).getPayload(), this.payload))
         {
            _logger.warn("Payloads differ for transmission: " + i);
         }
      }

      for (int i = 0; i < NUM_CONCURRENT_CONNECTIONS; i++)
      {
         conns.get(i).disconnect();
      }
   }

   /**
    * Tests many concurrent transmissions
    */
   @Test
   public void bigAsynchronousTransmission() throws CommunicationIOException,
         NotConnectedException, CommunicationInterruptedException, CommunicationResponseException,
         FileNotFoundException, InterruptedException, ExecutionException,
         CommunicationConnectionException
   {
      class Transmission implements Callable<Void>
      {
         private int _sequence;
         private Request _req = null;
         private MinaConnector _conn = null;

         public Transmission(Request req, MinaConnector conn, int sequence)
         {
            this._req = req;
            this._conn = conn;
            this._sequence = sequence;
         }

         public Void call() throws CommunicationIOException, NotConnectedException,
               CommunicationInterruptedException, CommunicationResponseException,
               CommunicationTransmissionException
         {
            if ((this._sequence % 500) == 0)
            {
               _logger.info("Sending request: " + this._sequence);
               this._conn.printStats();
            }

            this._conn.exchange(this._req);
            return null;
         }
      }

      ExecutorService executor = new BoundedThreadPoolExecutor("mina-test", NUM_THREADS);
      //ExecutorService executor = Executors.newCachedThreadPool();

      MinaConnector conn =
            new MinaConnector(HOST, PORT, new SimpleConnectorManager(), protocolMessageFactory);
      conn.ensureConnected();

      List<Callable<Void>> tasks = new ArrayList<Callable<Void>>(NUM_REQUESTS);
      for (int i = 0; i < NUM_REQUESTS; i++)
      {
         NoopRequest req = new NoopRequest();
         req.setPayload(this.payload);
         tasks.add(i, new Transmission(req, conn, i));
      }

      long beginTime = System.currentTimeMillis();
      long bytesSent = 2 * this.payload.length * NUM_REQUESTS;
      List<Future<Void>> results = null;
      try
      {
         results = executor.invokeAll(tasks);
      }
      catch (final InterruptedException e)
      {
         fail(e.getMessage());
      }

      // When invokeAll returns, all tasks have been executed
      long duration = System.currentTimeMillis() - beginTime;

      for (Future<Void> result : results)
      {
         result.get();
      }

      _logger.info(String.format("Total time: %dm%.1fs, UDC: %.3fMB (%.1fMbps)", duration
            / (60 * 1000), (duration % (60 * 1000)) / 1000.0, bytesSent / (1024 * 1024.0),
            (bytesSent * 1000 * 8) / (duration * 1024 * 1024.0)));

      conn.disconnect();
   }

   /**
    * Tests connecting and disconnecting a single connection
    * @throws CommunicationConnectionException 
    * @throws CommunicationIOException 
    * @throws CommunicationInterruptedException 
    */
   @Test
   public void reusableConnection() throws CommunicationIOException,
         CommunicationConnectionException, CommunicationInterruptedException
   {
      MinaConnector conn =
            new MinaConnector(HOST, PORT, new SimpleConnectorManager(), protocolMessageFactory);

      for (int i = 0; i < NUM_ITERATIONS; i++)
      {
         conn.ensureConnected();

         _logger.info("Sending request: " + i);
         try
         {
            conn.exchange(new NoopRequest());
         }
         catch (Exception ex)
         {
            _logger.info("Failed", ex);
            fail(ex.getMessage());
         }
         _logger.info("Received response: " + i);

         conn.disconnect();
      }
   }

   /**
    * Tests idle connection disconnection
    */
   @Test
   public void idleConnectionDisconnect() throws CommunicationIOException,
         CommunicationConnectionException, InterruptedException, CommunicationInterruptedException,
         CommunicationResponseException, CommunicationTransmissionException
   {
      MinaConnector conn =
            new MinaConnector(HOST, PORT, new SimpleConnectorManager(), protocolMessageFactory);
      conn.setManager(new SmartConnectorManager());
      //conn.setConnectionIdleTimeout(5);
      conn.ensureConnected();

      Thread.sleep(10000);

      conn.ensureConnected();
      _logger.info("Connection successfully re-established");

      conn.exchange(new NoopRequest());
      _logger.info("Message successfully exchanged");

      conn.disconnect();
   }

   /**
    * Tests many synchronous transmissions
    */
   @Test
   public void bigSyncronousTransmission() throws CommunicationIOException, NotConnectedException,
         CommunicationInterruptedException, CommunicationResponseException, FileNotFoundException,
         CommunicationTransmissionException, CommunicationConnectionException
   {
      MinaConnector conn =
            new MinaConnector(HOST, PORT, new SimpleConnectorManager(), protocolMessageFactory);
      conn.ensureConnected();

      NoopRequest req = new NoopRequest();
      req.setPayload(this.payload);

      long beginTime = System.currentTimeMillis();
      long bytesSent = 2 * this.payload.length * NUM_REQUESTS;

      for (int i = 0; i < NUM_REQUESTS; i++)
      {
         if ((i % 500) == 0)
         {
            _logger.info("Sending request: " + i);
            conn.printStats();
         }

         conn.exchange(req);
      }

      long duration = System.currentTimeMillis() - beginTime;
      _logger.info(String.format("Total time: %dm%.1fs, UDC: %.3fMB (%.1fMbps)", duration
            / (60 * 1000), (duration % (60 * 1000)) / 1000.0, bytesSent / (1024 * 1024.0),
            (bytesSent * 1000 * 8) / (duration * 1024 * 1024.0)));

      conn.disconnect();
   }

   /**
    * Test sunny-day communication
    */
   @Test
   public void sunnyDayTest()
   {
      final MinaConnector connector =
            new MinaConnector(HOST, PORT, new SimpleConnectorManager(), protocolMessageFactory);

      try
      {
         for (int i = 0; i < NUM_ITERATIONS; ++i)
         {
            Request request = new NoopRequest();
            connector.ensureConnected();
            connector.exchange(request);
            _logger.info("Successfully exchanged message");
         }

         connector.disconnect();
      }
      catch (Exception ex)
      {
         _logger.info(ex);
         fail(ex.getMessage());
      }
   }

   /**
    * Test for race conditions during server start and stop
    * @throws InterruptedException 
    */
   @Test
   public void serverUpDownStressTest() throws InterruptedException
   {
      MinaConnector connector =
            new MinaConnector(HOST, PORT, new SimpleConnectorManager(), protocolMessageFactory);

      // One thread continues to send messages until stopped
      MessageThread sendThread = new MessageThread(connector);
      sendThread.start();

      // Toggle connectivity
      int MIN_WAIT = 500;
      int MAX_WAIT = 1000;
      for (int i = 0; i < NUM_ITERATIONS; ++i)
      {
         int wait = this.random.nextInt(MAX_WAIT - MIN_WAIT) + MIN_WAIT;

         // Stop server and wait some time
         this.stopServer();
         Thread.sleep(wait);

         // Reset the latch to test for reconnect
         CountDownLatch latch = new CountDownLatch(10);
         sendThread.setCounter(latch);
         this.startServer();

         // Wait for send thread to proceed
         if (!latch.await(SEND_WAIT_MS, TimeUnit.MILLISECONDS))
         {
            fail("The connector appears to be deadlocked");
         }
         else
         {
            _logger.info("CYCLE COMPLETE!");
         }
      }

      sendThread.stopAndJoin();
   }

   /**
    * Test for race conditions during calls to disconnect() and connect()
    * @throws InterruptedException 
    * @throws CommunicationConnectionException 
    * @throws CommunicationIOException 
    */
   @Test
   public void serverDisconnectStressTest() throws InterruptedException
   {
      ConnectorManager connectorManager = new SimpleConnectorManager();
      MinaConnector connector =
            new MinaConnector("127.0.0.1", PORT, connectorManager, protocolMessageFactory);

      // One thread continues to send messages until stopped
      MessageThread sendThread = new MessageThread(connector);
      sendThread.start();

      // Toggle connectivity
      for (int i = 0; i < NUM_ITERATIONS; ++i)
      {
         // Disconnect & wait
         try
         {
            connector.disconnect();
         }
         catch (Exception ex)
         {
            _logger.info(ex);
            fail("Disconnect failure");
         }

         // Reset latch and wait
         CountDownLatch latch = new CountDownLatch(10);
         sendThread.setCounter(latch);

         // Wait for send thread to proceed
         if (!latch.await(SEND_WAIT_MS, TimeUnit.MILLISECONDS))
         {
            fail("The connector appears to be deadlocked");
         }
         else
         {
            _logger.info("CYCLE COMPLETE!");
         }
      }

      sendThread.stopAndJoin();
   }

   private static class MessageThread extends Thread
   {
      private Connector connector;
      private CountDownLatch counter;
      private AtomicBoolean stop;

      public MessageThread(Connector connector)
      {
         this.connector = connector;
         this.stop = new AtomicBoolean(false);
         this.counter = null;
      }

      public void setCounter(CountDownLatch counter)
      {
         synchronized (this)
         {
            this.counter = counter;
         }
      }

      public void stopAndJoin()
      {
         this.stop.set(true);
         try
         {
            this.join();
         }
         catch (InterruptedException e)
         {
            // Do nothing
         }
      }

      public void run()
      {
         _logger.info("Send thread starting");
         while (!stop.get())
         {
            Request request = new NoopRequest();
            try
            {
               this.connector.ensureConnected();
               this.connector.exchange(request);
               //_logger.info("Successfully exchanged message");

               synchronized (this)
               {
                  if (this.counter != null)
                  {
                     counter.countDown();
                  }
               }
            }
            catch (CommunicationException e)
            {
               // These exceptions mean that we have gracefully handled the error
               _logger.info("Got CommunicationException (This is OK): " + e.getMessage());

               if (this.connector.isConnected())
               {
                  _logger.info("Operation failed on active connection");
               }
            }
         }
         _logger.info("Send thread stopping");
      }
   }

   private static class NoopServer extends MinaAcceptor
   {
      /**
       * Constructs a slice server additionally taking a SliceServerConfiguration object
       * 
       * @param bindHost
       * @param listenPort
       * @param enableSSL
       * @param enableNetworkLogging
       * @param config
       */
      public NoopServer(int listenPort, boolean enableSSL, ProtocolMessageFactory pmFactory)
      {
         //super(new EchoServerApplication(), pmFactory);

         setEnableSSL(enableSSL);
         setPort(listenPort);
         setHost("0.0.0.0");
         setEnableLogging(false);

         setProtocolMessageFactory(pmFactory);
         setServerApplication(new EchoServerApplication());

         this.isRunning = false;
      }

      private static class EchoServerApplication extends ServerApplication
      {
         /**
          * Default Constructor
          */
         public EchoServerApplication()
         {
            _addHandlers();
         }

         /**
          * Registers all handlers for the recognized protocol messages
          */
         private void _addHandlers()
         {
            // Echo handler
            addHandler(new NoopHandler());
         }

         /**
          * Verify that a protocol message is compatible with the Server as defined in its
          * configuration.
          * 
          * @see SliceServerApplicationImpl.verifyProtocolVersionInMessage
          */
         public boolean verifyProtocolVersionsInMessage(Request request, ClientSession session)
         {
            return (request instanceof NoopRequest);
         }
      }
   }

   /**
    * Command-line interface for mina testing
    */
   public static void main(String[] args) throws Exception
   {
      initOnce();
      Parameter[] jsapParameters =
            new Parameter[]{
                  new QualifiedSwitch("server", JSAP.BOOLEAN_PARSER, JSAP.NO_DEFAULT,
                        JSAP.NOT_REQUIRED, 's', "server", "Run in server mode"),
                  new FlaggedOption("client", JSAP.STRING_PARSER, JSAP.NO_DEFAULT,
                        JSAP.NOT_REQUIRED, 'c', "client", "Test to execute (sync/async)"),

                  new FlaggedOption("host", JSAP.STRING_PARSER, "0.0.0.0", JSAP.NOT_REQUIRED, 'h',
                        "host", "Interface to open/connect to"),
                  new FlaggedOption("port", JSAP.INTEGER_PARSER, "5099", JSAP.NOT_REQUIRED, 'p',
                        "port", "Port to open/connect to"),

                  new FlaggedOption("num", JSAP.INTEGER_PARSER, "1000", JSAP.NOT_REQUIRED, 'n',
                        "num", "Number of requests to send"),
                  new QualifiedSwitch("verbose", JSAP.BOOLEAN_PARSER, "false", JSAP.NOT_REQUIRED,
                        'v', "verbose", "Verbose output"),
                  new FlaggedOption("threads", JSAP.INTEGER_PARSER, "20", JSAP.NOT_REQUIRED, 't',
                        "threads", "Number of concurrent threads for async test"),
                  new FlaggedOption("payload-size", JSAP.INTEGER_PARSER, "4096", JSAP.NOT_REQUIRED, 'z',
                        "payload-size", "Payload size"),
            };

      SimpleJSAP jsap =
            new SimpleJSAP("Mina Tester", "Tests Mina and protocol serialization over any link",
                  jsapParameters);

      JSAPResult config = jsap.parse(args);
      if (config != null)
      {
         Logger.getRootLogger().setLevel(Level.ERROR);
         _logger.setLevel(Level.INFO);
         if (config.getBoolean("verbose"))
         {
            Logger.getLogger("org.cleversafe.layer.communication.network.mina").setLevel(Level.INFO);
         }

         int port = config.getInt("port");
         String host = config.getString("host");
         int numRequests = config.getInt("num");
         int numThreads = config.getInt("threads");
         int payloadSize = config.getInt("payload-size");
         
         // FIXME: Ghetto
         HOST = host;
         PORT = port;
         NUM_REQUESTS = numRequests;
         NUM_THREADS = numThreads;
         PAYLOAD_SIZE = payloadSize;

         if (config.getBoolean("server", false))
         {
            // Server mode
            System.out.println("Running in server mode on: " + host + "/" + port);

            // FIXME: Ignores host
            NoopServer server = new NoopServer(port, false, protocolMessageFactory);
            server.start();

            try
            {
               Thread.sleep(10000000); // Sleep for awhile =)
            }
            finally
            {
               server.stop();
            }
         }
         else
         {
            // Client mode
            String testToRun = config.getString("client");
            if (testToRun == null)
            {
               System.err.println("One of --server or --client=<test> must be specified");
               System.exit(1);
            }

            System.out.println("Running client test '" + testToRun + "' on: " + host + "/" + port);
            MinaConnectorTest test = new MinaConnectorTest();

            if (testToRun.equals("async"))
            {
               test.bigAsynchronousTransmission();
            }
            else if (testToRun.equals("sync"))
            {
               test.bigSyncronousTransmission();
            }
            else
            {
               System.err.println("Unknown test: " + testToRun);
               System.exit(1);
            }
         }
      }

      System.out.println("Done.");
   }
}
