//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Apr 25, 2008
//---------------------

package org.cleversafe.layer.grid;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;
import org.cleversafe.codec.Codec;
import org.cleversafe.config.BindingsProvider;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.optimizedcauchy.CauchyInformationDispersalCodec;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.exceptions.GridLayerException;
import org.cleversafe.layer.grid.smartcontroller.SmartReadController;
import org.cleversafe.layer.grid.smartcontroller.SmartWriteController;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.layer.slicestore.memory.MemorySliceStore;
import org.cleversafe.layer.slicestore.unreliable.UnreliableSliceStore;
import org.cleversafe.test.BaseTest;
import org.cleversafe.vault.MockVault;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultGenerator;
import org.junit.Test;

/**
 * Tests controller behavior under permutations of parameters and outage conditions 
 */
public class ControllerUnitTest extends BaseTest
{
   private static Logger _logger = Logger.getLogger(ControllerUnitTest.class);

   public static final int SOURCE_SIZE = 1024;
   public static final int NUM_SOURCES = 10;

   public static Random random = new Random();

   /**
    * Tests udc with no outages
    * 
    * @throws ObjectInitializationException 
    * @throws ObjectInstantiationException 
    * @throws ConfigurationItemNotDefinedException 
    * @throws ConfigurationLoadException 
    * @throws ControllerException 
    */
   @Test
   public void sunnyDayTest() throws ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException, ControllerException
   {
      doUDC(new VaultGenerator(3, 0, 0)); // 3-lose-k (a few seconds)
      doUDC(new VaultGenerator(4, 0, 0)); // 4-lose-k (a few seconds)
      doUDC(new VaultGenerator(8, 0, 0)); // 8-lose-k (less than a minute)
      doUDC(new VaultGenerator(16, 0, 0)); // 16-lose-k (a couple minutes)
   }

   /**
    * Tests udc with both recoverable and non-recoverable outages
    * 
    * @throws ControllerException 
    * @throws ObjectInitializationException 
    * @throws ObjectInstantiationException 
    * @throws ConfigurationItemNotDefinedException 
    * @throws ConfigurationLoadException 
    */
   @Test
   public void rainyDayTest() throws ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException, ControllerException
   {
      doUDC(new VaultGenerator(3, 1, 1)); // 3-lose-k w/{1,1} out (a few seconds)
      doUDC(new VaultGenerator(4, 1, 3)); // 4-lose-k, w/{1,3} out (a few seconds)
      //doUDC(new VaultGenerator(6, 1, 5)); // 6-lose-k, w/{1,4} out (a few hours)
      //doUDC(new VaultGenerator(8, 1, 7)); // 8-lose-k, w/{1,7} out (about a day)
      //doUDC(new VaultGenerator(16, 1, 15)); // 16-lose-k w/{1,15} out (heat death of the universe)
   }

   /**
    * Tests for race conditions between simultaneous readers/writers and consistency
    * 
    * @throws ConfigurationLoadException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    * @throws ControllerException
    */
   //@Test
   // Not currently working, likely a problem with the test -- will fix in 1.1
   public void concurrentTest() throws ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException, ControllerException
   {
      doConcurrentUD(new VaultGenerator(3, 0, 1), 10, 10, false);
      doConcurrentUD(new VaultGenerator(4, 0, 3), 10, 10, false);

      // Should work in v1.5
      //doConcurrentUD(new VaultGenerator(3, 0, 1), 10, 10, true);
      //doConcurrentUD(new VaultGenerator(4, 0, 3), 10, 10, true);
   }

   @Test
   public void testConstruction() throws IDAInvalidParametersException, ControllerException
   {
      MockVault vault = new VaultGenerator(8, 0, 0).next();
      WriteController wc = new SmartWriteController(vault);
      ReadController rc = new SmartReadController(vault);

      assert wc.isOperational();
      assert rc.isOperational();

      wc.shutdown();
      rc.shutdown();

      assert !wc.isOperational();
      assert !rc.isOperational();
   }

   @Test
   public void testRollback() throws ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException, IDAInvalidParametersException, ControllerException
   {
      MockVault vault = new VaultGenerator(8, 0, 0).next();

      // Construct controller
      GridController gridController = this.getGridController(vault);
      WriteController wc = gridController.getWriteController();
      ReadController rc = gridController.getReadController();

      DataSource source =
            new DataSource(new SourceName("0"), wc.getTransactionId(), "test1".getBytes());
      wc.write(source);
      wc.commit();

      wc = gridController.getWriteController();
      source = new DataSource(new SourceName("0"), wc.getTransactionId(), "test2".getBytes());
      wc.write(source);
      wc.rollback();

      source = rc.read(new SourceName("0"));

      assert source.getData() != null : "Unable to read data";
      assert new String(source.getData()).equals("test1");
   }

   @Test
   public void testCorruptedSliceNotification() throws IDAInvalidParametersException,
         ConfigurationLoadException, ConfigurationItemNotDefinedException,
         ObjectInstantiationException, ObjectInitializationException, ControllerException,
         SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException,
         IllegalSourceNameException, SliceStoreQuotaException
   {
      MockVault vault = new VaultGenerator(8, 2, 2).next();
      GridController gridController = this.getGridController(vault);
      WriteController wc = gridController.getWriteController();

      MockCorruptedSliceObserver observer = new MockCorruptedSliceObserver();
      wc.addCorruptedSliceObserver(observer);

      long transactionId = wc.getTransactionId();
      DataSource source = new DataSource(new SourceName("0"), transactionId, "test1".getBytes());
      wc.write(source);
      wc.commit();

      assert observer.getSliceNames(transactionId).size() > 0 : "Did not get notification of slice corruption for write";

      ReadController rc = gridController.getReadController();

      observer = new MockCorruptedSliceObserver();
      rc.addCorruptedSliceObserver(observer);

      source = rc.read(new SourceName("1"));

      assert observer.hasObservedCorruptedSlices() : "Did not get notification of slice corruption for read";
   }

   /**
    * Test revision fallback on reads
    * 
    * @throws IDAInvalidParametersException 
    * @throws NoSuchMethodException 
    * @throws ObjectInitializationException 
    * @throws ObjectInstantiationException 
    * @throws ConfigurationItemNotDefinedException 
    * @throws ConfigurationLoadException 
    * @throws ControllerException 
    */
   @Test
   public void revisionFallbackTest() throws IDAInvalidParametersException, NoSuchMethodException,
         ConfigurationLoadException, ConfigurationItemNotDefinedException,
         ObjectInstantiationException, ObjectInitializationException, ControllerException
   {
      class Params
      {
         public int width;
         public int threshold;
         public int readThreshold;
         public int writeThreshold;

         public Params(int width, int threshold, int readThreshold, int writeThreshold)
         {
            this.width = width;
            this.threshold = threshold;
            this.readThreshold = readThreshold;
            this.writeThreshold = writeThreshold;
         }

         public String toString()
         {
            return String.format("n=%d, t=%d, w=%d, r=%d", this.width, this.threshold,
                  this.writeThreshold, this.readThreshold);
         }
      }

      Params[] paramsToTest = new Params[]{
            new Params(8, 6, 6, 6), new Params(8, 4, 4, 5), new Params(8, 2, 5, 4)
      };

      for (Params params : paramsToTest)
      {
         InformationDispersalCodec ida =
               new CauchyInformationDispersalCodec(params.width, params.threshold, 1);
         List<Codec> codecs = new ArrayList<Codec>();
         List<Codec> sliceCodecs = new ArrayList<Codec>();

         for (int numFailures = 0; numFailures < params.width - params.writeThreshold; ++numFailures)
         {
            for (int order = 0; order < 2; ++order)
            {
               _logger.info(String.format("Beginning iteration: f=%d, o=%s [%s]", numFailures,
                     (order == 0) ? "low-high" : "high-low", params.toString()));
               List<SliceStore> stores = new ArrayList<SliceStore>(params.width);
               for (int i = 0; i < params.width; ++i)
               {
                  stores.add(new UnreliableSliceStore(new MemorySliceStore()));
               }

               MockVault vault =
                     new MockVault(ida, codecs, sliceCodecs, stores, params.writeThreshold,
                           params.readThreshold, 0, numFailures, numFailures, 0, 0,
                           new HashSet<Integer>());
               vault.startSessions();
               GridController gridController = this.getGridController(vault);

               // Try both low-high and high-low revisions
               WriteController wc1, wc2;
               if (order == 0)
               {
                  wc1 = gridController.getWriteController();
                  wc2 = gridController.getWriteController();
               }
               else
               {
                  wc2 = gridController.getWriteController();
                  wc1 = gridController.getWriteController();
               }

               // Generate source names
               List<SourceName> sourceNames = new ArrayList<SourceName>(NUM_SOURCES);
               for (int i = 0; i < NUM_SOURCES; ++i)
               {
                  sourceNames.add(new SourceName(Integer.toString(random.nextInt(1000000))));
               }

               // Perform first write - succeeds on all stores
               List<DataSource> sources1 = new ArrayList<DataSource>(NUM_SOURCES);
               for (SourceName name : sourceNames)
               {
                  byte[] data = new byte[SOURCE_SIZE];
                  random.nextBytes(data);
                  sources1.add(new DataSource(name, wc1.getTransactionId(), data));
               }
               wc1.write(sources1);
               wc1.commit();

               // Second write fails on zero or more stores
               for (int i = 0; i < numFailures; ++i)
               {
                  ((UnreliableSliceStore) stores.get(i)).setMethodExceptionRate("write",
                        SliceStoreIOException.class, 1.0);
               }

               // Perform second write
               List<DataSource> sources2 = new ArrayList<DataSource>(NUM_SOURCES);
               for (SourceName name : sourceNames)
               {
                  byte[] data = new byte[SOURCE_SIZE];
                  random.nextBytes(data);
                  sources2.add(new DataSource(name, wc2.getTransactionId(), data));
               }
               wc2.write(sources2);
               wc2.commit();

               // Perform read
               List<DataSource> readSources = null;
               ReadController rc = gridController.getReadController();
               try
               {
                  readSources = rc.read(sourceNames);
               }
               catch (Exception ex)
               {
                  _logger.fatal(ex);
                  fail("Read failure");
               }

               // Compare
               if (readSources.size() != sourceNames.size())
               {
                  fail("Wrong number of sources read");
               }
               for (int i = 0; i < readSources.size(); ++i)
               {
                  DataSource source = readSources.get(i);
                  byte[] cmp = source.getData();

                  byte[] origOk, origWrong;
                  if (order == 1 && numFailures >= params.threshold)
                  {
                     // Enough slices of the first source exist to be restored
                     origOk = sources1.get(i).getData();
                     origWrong = sources2.get(i).getData();
                  }
                  else
                  {
                     // Second source should be restored
                     origOk = sources2.get(i).getData();
                     origWrong = sources1.get(i).getData();
                  }

                  if (!Arrays.equals(origOk, cmp))
                  {
                     if (!Arrays.equals(origWrong, cmp))
                     {
                        fail("Compare failure -- ??");
                     }
                     else
                     {
                        fail("Compare failure -- wrong revision");
                     }
                  }
               }
            }
         }
      }
   }

   /**
    * Perform a random UDC and check for expected and unexpected errors.  One or more iterations
    * are performed for each Vault permutation.  On each iteration, the same Vault conditions
    * are used, but different outages are simulated.
    * 
    * @param generator
    * @throws ConfigurationLoadException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    * @throws ControllerException
    */
   private void doUDC(VaultGenerator generator) throws ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException, ControllerException
   {
      MockVault vault;

      long numVaults = 0;
      long totalVaults = generator.getNumVaults();

      while (true)
      {
         try
         {
            // Get next vault.  Impossible vaults will be skipped
            ++numVaults;
            vault = generator.next();
            if (vault == null)
            {
               break; // Done
            }

            _logger.info(String.format("%d/%d (%.1f%%): Trying %s", numVaults, totalVaults, 100.
                  * numVaults / (double) totalVaults, vault.toString()));

            // Generate sources
            List<SourceName> sourceNames = new ArrayList<SourceName>(NUM_SOURCES);
            byte[][] sourceData = new byte[NUM_SOURCES][SOURCE_SIZE];
            for (int i = 0; i < NUM_SOURCES; ++i)
            {
               sourceNames.add(new SourceName(Integer.toString(i)));
               random.nextBytes(sourceData[i]);
            }

            // Write
            try
            {
               boolean success = this.helperDoWrite(vault, sourceNames, sourceData);
               if (!success)
               {
                  // An expected failure occurred
                  continue;
               }
            }
            catch (GridLayerException ex)
            {
               _logger.error(ex);
               fail("Unexpected write failure");
            }

            // Read
            List<DataSource> readSources = null;
            try
            {
               readSources = this.helperDoRead(vault, sourceNames);
               if (readSources == null)
               {
                  // An expected failure occurred
                  continue;
               }
            }
            catch (GridLayerException ex)
            {
               _logger.error(ex);
               fail("Unexpected read failure");
            }

            // Compare
            boolean thresholdsOk =
                  (vault.getWriteThreshold() + vault.getReadThreshold() > vault.getWidth());
            if (readSources.size() != sourceNames.size())
            {
               fail("Wrong number of sources read");
            }
            for (int i = 0; i < readSources.size(); ++i)
            {
               DataSource source = readSources.get(i);
               byte[] cmp = source.getData();
               byte[] orig = sourceData[i];

               if (!Arrays.equals(orig, cmp))
               {
                  // Is this possible?
                  if (!thresholdsOk
                        && source.getTransactionId() == GridTransaction.NOT_FOUND_TRANSACTION_ID)
                  {
                     // This is acceptable with the given thresholds
                     continue;
                  }
                  else
                  {
                     fail("Compare failure");
                  }
               }
            }
         }
         catch (IDAInvalidParametersException e)
         {
            // Known invalid vault
            _logger.info(String.format("%d/%d (%.1f%%): skipping impossible vault", numVaults,
                  totalVaults, 100. * numVaults / (double) totalVaults));
         }
      }

      assertEquals("Counting error", totalVaults, numVaults - 1, totalVaults);
   }

   private boolean helperDoWrite(MockVault vault, List<SourceName> sourceNames, byte[][] sourceData)
         throws GridLayerException, ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      GridController gridController = this.getGridController(vault);
      WriteController wc = gridController.getWriteController();
      InformationDispersalCodec ida = vault.getInformationDispersalCodec();

      assert wc.isOperational() : "Write controller is not operational";

      // Set get/set vault methods
      assert wc.getVault() == vault : "Write controller did not have expected vault";

      wc.setVault(null);
      assert wc.getVault() == null : "Write controller did not have expected vault";

      wc.setVault(vault);
      assert wc.getVault() == vault : "Write controller did not have expected vault";

      // Generate sources
      List<DataSource> sources = new ArrayList<DataSource>(sourceNames.size());
      for (int i = 0; i < sourceNames.size(); ++i)
      {
         sources.add(new DataSource(sourceNames.get(i), wc.getTransactionId(), sourceData[i]));
      }

      try
      {
         wc.write(sources);
         wc.commit();

         // Check for expected exceptions
         if (ida.getNumSlices() - vault.getWriteFailures() < vault.getWriteThreshold())
         {
            fail("Write failure expected");
         }

         return true;
      }
      catch (GridLayerException ex)
      {
         // Check for unexpected exceptions
         if (ida.getNumSlices() - vault.getWriteFailures() >= vault.getWriteThreshold())
         {
            throw ex;
         }
         else
         {
            // Expected
            return false;
         }
      }
      finally
      {
         gridController.release(wc);
         assert !wc.isOperational() : "Write controller is still operational";
      }
   }

   public List<DataSource> helperDoRead(MockVault vault, List<SourceName> sourceNames)
         throws GridLayerException, ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      GridController gridController = this.getGridController(vault);
      ReadController rc = gridController.getReadController();
      InformationDispersalCodec ida = vault.getInformationDispersalCodec();

      assert rc.isOperational() : "Read controller is not operational";

      // Set get/set vault methods
      assert rc.getVault() == vault : "Read controller did not have expected vault";

      rc.setVault(null);
      assert rc.getVault() == null : "Read controller did not have expected vault";

      rc.setVault(vault);
      assert rc.getVault() == vault : "Read controller did not have expected vault";

      List<DataSource> readSources;
      try
      {
         readSources = rc.read(sourceNames);

         // Check for expected exceptions, excluding source not found
         if (ida.getNumSlices() - vault.getNumCumulativeErrors() - vault.getNumCumulativeCorrupt() < vault.getReadThreshold())
         {
            fail("Read failure expected (too many errors)");
         }
         else if (ida.getNumSlices() - vault.getNumCumulativeErrors()
               - vault.getNumCumulativeCorrupt() - vault.getNumCumulativeTxInvalid() < ida.getThreshold())
         {
            // Fail unless datasource not found
            if (readSources.get(0).getTransactionId() != GridTransaction.NOT_FOUND_TRANSACTION_ID)
            {
               fail("Read failure expected (no revision)");
            }
            else
            {
               // Expected -- fallback hit DS not found
               return null;
            }
         }

         return readSources;
      }
      catch (GridLayerException ex)
      {
         // Check for unexpected exceptions
         if (ida.getNumSlices() - vault.getNumCumulativeErrors() - vault.getNumCumulativeCorrupt() < vault.getReadThreshold())
         {
            // Expected -- does not meet read threshold
            return null;
         }
         else if (ida.getNumSlices() - vault.getNumCumulativeErrors()
               - vault.getNumCumulativeCorrupt() - vault.getNumCumulativeTxInvalid() < ida.getThreshold())
         {
            // Expected -- does not meet IDA threshold
            return null;
         }
         else
         {
            throw ex;
         }
      }
      finally
      {
         gridController.release(rc);
         assert !rc.isOperational() : "Read controller is still operational";
      }
   }

   /**
    * Performs concurrent upload/download with optional separation between reads and writes.  
    * Writes may fail due to transaction races, but reads should always succeed -- eg. there 
    * should never be an inconsistent view of data in the vault.
    * 
    * @param generator
    * @param numConcurrentWriters
    * @param numConcurrentReaders
    * @param rwConcurrent
    * @throws ConfigurationLoadException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    * @throws ControllerException
    */
   private void doConcurrentUD(
         VaultGenerator generator,
         int numConcurrentWriters,
         int numConcurrentReaders,
         boolean rwConcurrent) throws ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException, ControllerException
   {
      long numVaults = 0;
      long totalVaults = generator.getNumVaults();

      ExecutorService executor = Executors.newCachedThreadPool();
      try
      {
         while (true)
         {
            try
            {
               // Get next vault.  Impossible vaults will be skipped
               ++numVaults;
               final MockVault vault = generator.next();
               if (vault == null)
               {
                  break; // Done
               }

               _logger.info(String.format("%d/%d (%.1f%%): Trying %s", numVaults, totalVaults, 100.
                     * numVaults / (double) totalVaults, vault.toString()));

               // We will write sources with random names
               final List<SourceName> sourceNames = new ArrayList<SourceName>(NUM_SOURCES);
               for (int i = 0; i < NUM_SOURCES; ++i)
               {
                  sourceNames.add(new SourceName(Integer.toString(random.nextInt(1000000))));
               }

               // Generate writers and the data they will write
               class WriteTask implements Callable<Boolean>
               {
                  private byte[][] sourceData;

                  public WriteTask() throws ControllerException
                  {
                     // Generate source data
                     sourceData = new byte[NUM_SOURCES][SOURCE_SIZE];
                     for (int i = 0; i < NUM_SOURCES; ++i)
                     {
                        random.nextBytes(sourceData[i]);
                     }
                  }

                  public Boolean call() throws Exception
                  {
                     while (true)
                     {
                        try
                        {
                           return helperDoWrite(vault, sourceNames, sourceData);
                        }
                        catch (GridLayerException ex)
                        {
                           // Give it another try
                           _logger.debug("Write failed with '" + ex + "', retrying");
                        }
                        catch (AssertionError ex)
                        {
                           _logger.error("Async write assertion failure", ex);
                           throw ex;
                        }
                     }
                  }
               }

               class ReadTask implements Callable<List<DataSource>>
               {
                  public List<DataSource> call() throws Exception
                  {
                     return helperDoRead(vault, sourceNames);
                  }
               }

               List<WriteTask> writers = new ArrayList<WriteTask>(numConcurrentWriters);
               for (int i = 0; i < numConcurrentWriters; ++i)
               {
                  writers.add(new WriteTask());
               }

               // Generate readers
               List<ReadTask> readers = new ArrayList<ReadTask>(numConcurrentReaders);
               for (int i = 0; i < numConcurrentReaders; ++i)
               {
                  readers.add(new ReadTask());
               }

               // Begin writes in a random order to test revision fallback
               List<Future<Boolean>> writeFutures = new ArrayList<Future<Boolean>>(writers.size());
               for (int i = 0; i < writers.size(); ++i)
               {
                  WriteTask writer = writers.remove(random.nextInt(writers.size()));
                  writeFutures.add(executor.submit(writer));
               }

               // Wait for writers to complete only if flag is set
               boolean writeSuccess = true;
               if (!rwConcurrent)
               {
                  for (Future<Boolean> future : writeFutures)
                  {
                     try
                     {
                        writeSuccess &= future.get();
                     }
                     catch (Exception ex)
                     {
                        _logger.error(ex);
                        fail("Unexpected write failure");
                     }
                  }
               }

               // Begin reads
               List<Future<List<DataSource>>> readFutures =
                     new ArrayList<Future<List<DataSource>>>(readers.size());
               for (ReadTask reader : readers)
               {
                  readFutures.add(executor.submit(reader));
               }

               // Wait for writes if we haven't already
               if (rwConcurrent)
               {
                  for (Future<Boolean> future : writeFutures)
                  {
                     try
                     {
                        writeSuccess &= future.get();
                     }
                     catch (Exception ex)
                     {
                        _logger.error(ex);
                        fail("Unexpected write failure");
                     }
                  }
               }

               // Wait for readers to complete
               for (Future<List<DataSource>> future : readFutures)
               {
                  try
                  {
                     future.get();
                  }
                  catch (Throwable ex)
                  {
                     // Only fail on read error if writes succeeded
                     if (writeSuccess)
                     {
                        _logger.error(ex);
                        fail("Unexpected read failure");
                     }
                  }
               }
            }
            catch (IDAInvalidParametersException e)
            {
               // Known invalid vault
               _logger.info(String.format("%d/%d (%.1f%%): skipping impossible vault", numVaults,
                     totalVaults, 100. * numVaults / (double) totalVaults));
            }
         }

         assertEquals("Counting error", totalVaults, numVaults - 1, totalVaults);
      }
      finally
      {
         executor.shutdown();
      }
   }

   /**
    * Constructs a GridController from the current bindings configuration for the given vault
    * @param vault
    * @throws ConfigurationLoadException 
    * @throws ObjectInitializationException 
    * @throws ObjectInstantiationException 
    * @throws ConfigurationItemNotDefinedException 
    */
   private GridController getGridController(Vault vault) throws ConfigurationLoadException,
         ConfigurationItemNotDefinedException, ObjectInstantiationException,
         ObjectInitializationException
   {
      // Construct controller
      BindingsProvider bindingsProvider =
            ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);
      ReadControllerFactory rfc =
            bindingsProvider.getDefaultImplementation(ReadControllerFactory.class);
      WriteControllerFactory wfc =
            bindingsProvider.getDefaultImplementation(WriteControllerFactory.class);
      return new GridController(vault, wfc, rfc);
   }

   class MockCorruptedSliceObserver implements CorruptedSliceObserver
   {
      private Map<Long, List<SliceName>> transactionSliceNamesMap =
            new HashMap<Long, List<SliceName>>();

      public void notify(long transactionId, List<SliceName> sliceNames)
      {
         List<SliceName> sliceNameList = transactionSliceNamesMap.get(transactionId);
         if (sliceNameList == null)
         {
            sliceNameList = new ArrayList<SliceName>();
         }

         sliceNameList.addAll(sliceNames);
         transactionSliceNamesMap.put(transactionId, sliceNameList);
      }

      public List<SliceName> getSliceNames(long transactionId)
      {
         List<SliceName> slices = transactionSliceNamesMap.get(transactionId);
         return (slices != null) ? slices : new ArrayList<SliceName>();
      }

      public boolean hasObservedCorruptedSlices()
      {
         return transactionSliceNamesMap.size() > 0;
      }
   }
}
