
package org.cleversafe.layer.grid;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import org.cleversafe.UnitTests;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.codec.Codec;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.optimizedcauchy.CauchyInformationDispersalCodec;
import org.cleversafe.layer.grid.simplecontroller.SimpleReadControllerFactory;
import org.cleversafe.layer.grid.simplecontroller.SimpleWriteControllerFactory;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.block.BlockMultiFileSliceStore;
import org.cleversafe.layer.slicestore.block.BlockMultiFileSliceStore.FileStorageDefinition;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.unreliable.UnreliableSliceStore;
import org.cleversafe.test.BaseTest;
import org.cleversafe.util.math.CombinationGenerator;
import org.cleversafe.vault.BaseVault;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class SimpleSliceStoreControllerTest
{
   private static final int GRID_PILLARS = 16;
   private static final int IDA_THRESHOLD = 12;

   private static final String TEST_PATH =
         System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".") + "/"
               + GridController.class.getName().replace('.', '/');

   private static final String TEST_OUTPUT_PATH =
         System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".");

   private static final int DATA_SIZE = 4096;

   private static int nBlocksPerTest = 10;
   private static int nCombinationsPerTest = 100;

   static Class<? extends WriteControllerFactory> writeControllerImplementation = null;
   static Class<? extends ReadControllerFactory> readControllerImplementation = null;

   private Vault vault;
   private GridController gridController = null;

   private byte data1[];
   private byte data2[];

   @SuppressWarnings("unchecked")
   @BeforeClass
   static public void prepareTest() throws Exception
   {
      UnitTests.resetConfiguration();

      deleteDirectory(new File(TEST_OUTPUT_PATH));

      writeControllerImplementation = SimpleWriteControllerFactory.class;
      readControllerImplementation = SimpleReadControllerFactory.class;
   }

   @Before
   public void setUp()
   {
      UnitTests.resetConfiguration();
      String blocksString = System.getProperty("blocks.pertest");
      if (blocksString != null)
      {
         nBlocksPerTest = Integer.parseInt(blocksString);
      }
      String combinationString = System.getProperty("combinations.pertest");
      if (combinationString != null)
      {
         nCombinationsPerTest = Integer.parseInt(combinationString);
      }

      try
      {
         this.vault = getVault();
         /*
          * WriteControllerFactory cwf = new SimpleWriteControllerFactory(); ReadControllerFactory
          * crf = new SimpleReadControllerFactory();
          */
         // Any controller could be plugged
         WriteControllerFactory cwf = writeControllerImplementation.newInstance();
         ReadControllerFactory crf = readControllerImplementation.newInstance();
         this.gridController = new GridController(this.vault, cwf, crf);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail(e.getMessage());
      }

      this.data1 = new byte[DATA_SIZE];
      randomizeBuffer(this.data1);

      this.data2 = new byte[DATA_SIZE];
      randomizeBuffer(this.data2);
   }

   @After
   public void cleanUp() throws SliceStoreLayerException
   {
      UnitTests.resetConfiguration();
      if (this.vault != null)
      {
         for (int idx = 0; idx < this.vault.getSliceStores().size(); idx++)
         {
            if (this.vault.getSliceStores().get(idx) != null)
            {
               this.vault.getSliceStores().get(idx).endSession();
               this.vault.getSliceStores().get(idx).deleteStore();
            }
         }
         deleteDirectory(new File(TEST_OUTPUT_PATH));
      }
   }

   @Test
   public void testSingleReadWrite()
   {
      UnitTests.resetConfiguration();
      try
      {
         testReadWrite();
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail(e.getMessage());
      }
   }

   public void testReadWrite() throws Exception
   {
      UnitTests.resetConfiguration();
      List<SourceName> allBlocks = new ArrayList<SourceName>(nBlocksPerTest);
      for (int blockId = 0; blockId < nBlocksPerTest; blockId++)
      {
         testReadWrite(blockId);
         allBlocks.add(blockId, new SourceName(Integer.toString(blockId)));
      }
      testReadWrite(allBlocks);
   }

   private void testReadWrite(List<SourceName> allBlocks) throws Exception
   {
      UnitTests.resetConfiguration();
      ReadController readController = this.gridController.getReadController();
      WriteController writeController = this.gridController.getWriteController();

      List<DataSource> dataSources = new ArrayList<DataSource>(allBlocks.size());
      for (int i = 0; i < allBlocks.size(); i++)
      {
         DataSource ds =
               new DataSource(allBlocks.get(i), writeController.getTransactionId(), this.data1);
         dataSources.add(i, ds);
      }

      try
      {
         writeController.write(dataSources);
         writeController.commit();
      }
      catch (Exception ex)
      {
         writeController.rollback();
         throw ex;
      }

      List<DataSource> readDataSources = readController.read(allBlocks);
      for (int i = 0; i < allBlocks.size(); i++)
      {
         assertTrue(Arrays.equals(this.data1, readDataSources.get(i).getData()));
      }
   }

   public void testReadWrite(long blockId) throws Exception
   {
      UnitTests.resetConfiguration();
      ReadController readController = this.gridController.getReadController();
      WriteController writeController = this.gridController.getWriteController();

      SourceName name = new SourceName(String.valueOf(blockId));
      DataSource ds = new DataSource(name, writeController.getTransactionId(), this.data1);
      long writeTransactionId = -1;
      try
      {
         writeController.write(ds);
         writeTransactionId = writeController.getTransactionId();
         writeController.commit();
      }
      catch (Exception ex)
      {
         writeController.rollback();
         throw ex;
      }

      DataSource readDs = readController.read(name);
      assertEquals(name, readDs.getName());
      assertEquals(writeTransactionId, readDs.getTransactionId());
      assertTrue(Arrays.equals(this.data1, readDs.getData()));
   }

   @Test
   public void testReadUsingUnreliableSliceStores()
   {
      UnitTests.resetConfiguration();
      // Test read errors
      testReadWriteUsingUnreliableSliceStores(1, 0);
   }

   @Test
   public void testWriteUsingUnreliableSliceStores()
   {
      UnitTests.resetConfiguration();
      // Test write errors
      testReadWriteUsingUnreliableSliceStores(0, 1);
   }

   @Test
   public void testReadWriteUsingUnreliableSliceStores()
   {
      UnitTests.resetConfiguration();
      // Test read and write errors
      testReadWriteUsingUnreliableSliceStores(1, 1);
   }

   public void testReadWriteUsingUnreliableSliceStores(
         double readFailureRate,
         double writeFailureRate)
   {
      // TODO: Figure out why the performance is no longer sufficient to run an
      // exhaustive test

      // CombinationGenerator combinationGenerator =
      // new CombinationGenerator(vault.getWidth(), vault.getLoseCount() );

      CombinationGenerator combinationGenerator =
            new CombinationGenerator(this.vault.getWidth(), 2);

      SliceStore reliableStores[] = new SliceStore[this.vault.getSliceStores().size()];
      for (int idx = 0; idx < reliableStores.length; idx++)
      {
         reliableStores[idx] = this.vault.getSliceStores().get(idx);
      }

      int combinationCounter = 0;
      while (combinationGenerator.hasMore() && combinationCounter < nCombinationsPerTest)
      {
         combinationCounter++;
         List<SliceStore> unreliableStores = this.vault.getSliceStores();

         for (int idx = 0; idx < unreliableStores.size(); idx++)
         {
            unreliableStores.set(idx, reliableStores[idx]);
         }

         int combination[] = combinationGenerator.getNext();

         for (int idx = 0; idx < combination.length; idx++)
         {
            int storeIdx = combination[idx];

            try
            {
               unreliableStores.set(storeIdx, new UnreliableSliceStore(
                     unreliableStores.get(storeIdx), readFailureRate, writeFailureRate, 0, 0));
            }
            catch (NoSuchMethodException e)
            {
               e.printStackTrace();
               fail(e.getMessage());
            }
         }

         if (writeFailureRate > 0)
         {
            try
            {
               testReadWrite();
               fail("Write should have failed due to missing slice stores");
            }
            catch (Exception ex)
            {
            }
         }
         else
         {
            try
            {
               testReadWrite();
            }
            catch (Exception ex)
            {
               fail(ex.getMessage());
            }
         }
      }

   }

   /*
    * @Test public void testTransactions() {
    * 
    * try { SliceStoreController sliceStoreController = (SliceStoreController)
    * controllerPrototype.clone();
    * 
    * SourceName name = new SourceName("0"); // Attempt to write without starting a transaction try {
    * sliceStoreController.write(name, data1); fail("Was able to write outside of a transaction"); }
    * catch (Exception ex) { // This is expected behavior }
    * 
    * sliceStoreController.beginTransaction(); sliceStoreController.write(name, data1);
    * 
    * byte readData[] = null; // Attempt to read uncommitted transaction try { readData =
    * sliceStoreController.read(name); // Warning was able to read uncommitted transaction } catch
    * (Exception ex) { // This is expected behavior }
    * 
    * sliceStoreController.commitTransaction();
    * 
    * readData = sliceStoreController.read(name);
    * 
    * for (int idx = 0; idx < data1.length; idx++) { assertEquals(data1[idx], readData[idx]); } //
    * Test rollback of a transaction sliceStoreController.beginTransaction();
    * sliceStoreController.write(name, data2); sliceStoreController.rollbackTransaction();
    * 
    * readData = sliceStoreController.read(name);
    * 
    * for (int idx = 0; idx < data1.length; idx++) { assertEquals(data1[idx], readData[idx]); } }
    * catch (Exception ex) { ex.printStackTrace(); fail(ex.getMessage()); } } // @Test public void
    * testExistsDatasource() { try { SliceStoreController sliceStoreController =
    * (SliceStoreController) controllerPrototype.clone();
    * 
    * SourceName sourceName = new SourceName(String.valueOf(1)); if
    * (sliceStoreController.existGridSource(sourceName)) { fail("A non-existant source should not
    * exist"); }
    * 
    * sliceStoreController.beginTransaction(); sliceStoreController.write(sourceName, data1);
    * sliceStoreController.commitTransaction();
    * 
    * if (!sliceStoreController.existGridSource(sourceName)) { fail("The written source does not
    * exist"); } } catch (Exception ex) { ex.printStackTrace(); fail(ex.getMessage()); } }
    */
   private static void randomizeBuffer(byte data[])
   {
      for (int idx = 0; idx < data.length; idx++)
      {
         data[idx] = (byte) (Math.random() * 256);
      }
   }

   protected static class SimpleVault extends BaseVault
   {
      public SimpleVault(
            String type,
            InformationDispersalCodec idaCodec,
            List<Codec> datasourceCodecs,
            List<Codec> sliceCodecs,
            List<SliceStore> sliceStores)
      {
         super(type, idaCodec, datasourceCodecs, sliceCodecs, sliceStores);
      }

      public long getSize()
      {
         return -1;
      }

      public long getMaxSliceSize()
      {
         return -1;
      }

      public void optimizeVault()
      {
         // Does nothing
      }
   }

   protected Vault getVault() throws Exception
   {
      // Initialize slice storage
      File gridBase = new File(TEST_OUTPUT_PATH);
      gridBase.mkdirs();

      VaultACL acl = createACL();

      List<SliceStore> sliceStores = new ArrayList<SliceStore>();
      for (int i = 0; i < GRID_PILLARS; ++i)
      {
         File ssPath = getSliceStoreDirectory(gridBase, i);

         SliceStore ss = new BlockMultiFileSliceStore( new FileStorageDefinition(ssPath.getAbsolutePath(), 10*1024));
         ss.createStore("dummy", 10*1096, 100*4024, acl, null);
         sliceStores.add(ss);
         ss.startSession();
      }

      // assert sliceStores.size() == GRID_PILLARS;

      // Initialize IDA
      InformationDispersalCodec ida =
            new CauchyInformationDispersalCodec(GRID_PILLARS, IDA_THRESHOLD, 4096);

      return new SimpleVault("Dummy", ida, null, null, sliceStores);
   }

   public static boolean deleteDirectory(File dir)
   {
      if (dir.isDirectory())
      {
         String[] children = dir.list();
         for (int i = 0; i < children.length; i++)
         {
            boolean success = deleteDirectory(new File(dir, children[i]));

            if (!success)
            {
               return false;
            }
         }
      }

      // The directory is now empty so delete it
      return dir.delete();
   }

   private static VaultACL createACL() throws Exception
   {
      UUID vaultIdentifier = UUID.randomUUID();

      String vaultFileName = TEST_OUTPUT_PATH + vaultIdentifier.toString() + ".der";
      FileOutputStream out = new FileOutputStream(vaultFileName);

      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("vaultacltest");
      credentials.setPassword("password");

      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize(512);
      KeyPair keypair = keygen.generateKeyPair();

      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();

      VaultACLFactory fact = new VaultACLFactory();
      fact.create(out, vaultIdentifier, credentials, keypair.getPublic(), keyInfoList);
      out.close();

      FileInputStream in = new FileInputStream(vaultFileName);

      VaultACL vaultACL = fact.getInstance(in);
      in.close();

      return vaultACL;
   }

   private static File getSliceStoreDirectory(File gridBase, int sliceID)
   {
      return new File(gridBase, String.format("ss%d", sliceID));
   }
}
