//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason
//
// Date: Aug 9, 2007
//---------------------

package org.cleversafe.layer.slicestore;

import java.io.FileInputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import junit.framework.TestCase;

import org.cleversafe.UnitTests;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.layer.slicestore.block.BlockFileSliceStore;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.test.BaseTest;
import org.cleversafe.vault.FileVaultACL;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultPermission;
import org.cleversafe.vault.exceptions.VaultConfigurationException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

// Tests the slice store manager interface
public abstract class SliceStoreManagerTest extends TestCase
{
   private SliceStoreManager manager = null;
   private Map<String, String> options = null;
   private UUID vaultID = null;
   private VaultACL vaultACL = null;

   public abstract SliceStoreManager getSliceStoreManager();

   
   @Before
   public void setUp() throws Exception
   {
      this.manager = this.getSliceStoreManager();

      // For this test to support all implementations this options map
      // must be populated with every possible option required by all
      // slice store managers
      this.options = new HashMap<String, String>();
      this.options.put(BlockFileSliceStore.SYNC_PROPERTY_NAME, "false");

      this.vaultID = UUID.nameUUIDFromBytes("SliceStoreManagerTest".getBytes());

      this.vaultACL = loadACL();
   }

   @After
   public void tearDown() throws Exception
   {
      manager = null;
   }

   @Test
   public void testCreateSliceStore()
   {
      try
      {
         this.manager.createSliceStore("block", 512, SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED,
               vaultID, vaultACL, null, options);
      }
      catch (SliceStoreExistsException e)
      {
         fail("Slice Store Exists");
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice Store IO Exception");
      }
   }

   @Test
   public void testLoadSliceStore()
   {
      try
      {
         SliceStore sliceStore = this.manager.loadSliceStore(vaultID);
         junit.framework.Assert.assertNotNull(sliceStore);
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testLoadVaultPermissions()
   {
      UnitTests.resetConfiguration();
      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("test");
      credentials.setPassword("test");

      try
      {
         VaultPermission permission =
               this.manager.loadVaultPermissions(vaultID, credentials.getAccountIdentifier());
         assertNotNull(permission);
      }
      catch (SliceStoreIOException e)
      {
         e.printStackTrace();
         fail("slice store IO exception");
      }
      catch (SliceStoreNotFoundException e)
      {
         e.printStackTrace();
         fail("slice store not found exception");
      }
   }

   @Test
   public void testDeleteSliceStore()
   {
      try
      {
         this.manager.deleteSliceStore(vaultID, options);
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice Store IO exception: " + e.getMessage());
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice Store not found: " + e.getMessage());
      }
   }

   private FileVaultACL loadACL() throws Exception
   {
      final String aclFileName =
            System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
                  + "/org/cleversafe/layer/slicestore/SliceStoreManager.ACL.der";

      /*
       * FileOutputStream out = new FileOutputStream(aclFileName);
       * 
       * UUID vaultIdentifier = UUID.randomUUID(); this.credentials = new PasswordCredentials();
       * this.credentials.setUsername("test"); this.credentials.setPassword("test");
       * 
       * System.out.println( "Creating Vault ACL for account: " +
       * this.credentials.getAccountIdentifier().toString() );
       * 
       * KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA"); keygen.initialize( 2048 );
       * KeyPair keypair = keygen.generateKeyPair();
       * 
       * VaultACLFactory fact = new VaultACLFactory(); fact.create( out, vaultIdentifier,
       * this.credentials, keypair.getPublic() ); out.close();
       */
      FileInputStream in = new FileInputStream(aclFileName);
      return new FileVaultACL(in);
   }

}
