//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Nov 1, 2007
//---------------------

package org.cleversafe.layer.slicestore.block;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;
import org.cleversafe.UnitTests;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.layer.grid.GridBaseTest;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.test.BaseTest;
import org.cleversafe.vault.FileVaultACL;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultPermission;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class BlockMultiSliceStoreManagerTest 
{
   private BlockMultiFileSliceStoreManager manager = null;
   private Map<String, String> options = null;
   //private UUID vaultID = null;
   private VaultACL vaultACL = null;

   private final int maxRecords = 3 * 1000;
   private static final int RECORD_SIZE = 512;

   private final String basePath =
         System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, "output")
               + "/org/cleversafe/layer/slicestore/manager/block-multi-file";

   private Logger logger = Logger.getLogger(BlockMultiSliceStoreManagerTest.class.getName());
   
   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      UnitTests.resetConfiguration();
   }

   @Before
   public void setUp() throws Exception
   {
      //System.out.println(System.getProperty("log4j.configuration"));
      this.manager = createSliceStoreManager();

      logger.debug("Created manager " + this.manager.toString());
      
      // For this test to support all implementations this options map
      // must be populated with every possible option required by all
      // slice store managers
      this.options = new HashMap<String, String>();
      this.options.put(BlockMultiFileSliceStore.SYNC_PROPERTY_NAME, "false");

      //this.vaultID = UUID.nameUUIDFromBytes("SliceStoreManagerTest".getBytes());

      this.vaultACL = loadACL();
   }

   @After
   public void tearDown() throws Exception
   {
      this.manager = null;
      File root = new File(this.basePath);
      assert (root.isDirectory());
      deleteDir(root);
   }

   private void deleteDir(File deletedDir)
   {
      assert deletedDir.isDirectory();
      File[] files = deletedDir.listFiles();
      for (File f : files)
      {
         if (f.isDirectory())
         {
            deleteDir(f);
         }
         f.delete();
      }
   }

   public BlockMultiFileSliceStoreManager createSliceStoreManager()
   {
      BlockMultiFileSliceStoreManager.PartitionDefinition[] defs =
            new BlockMultiFileSliceStoreManager.PartitionDefinition[2];
      defs[0] =
            new BlockMultiFileSliceStoreManager.PartitionDefinition(this.basePath + File.separator
                  + "1", this.maxRecords / 3
                  * (RECORD_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead()));
      defs[1] =
            new BlockMultiFileSliceStoreManager.PartitionDefinition(this.basePath + File.separator
                  + "2", (2 * this.maxRecords / 3)
                  * (RECORD_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead()));
      return new BlockMultiFileSliceStoreManager(this.basePath + File.separator + "access", defs);
   }

   @Test
   public void simpleTest()
   {
      UUID vaultId1 = UUID.fromString("b3f66ec1-535d-3770-ac38-111111111111");
      testCreateSliceStore(vaultId1, this.maxRecords / 5);
      assertTrue(this.manager.getRemainingSize() == ((this.maxRecords - (this.maxRecords / 5)) * (RECORD_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead())));
      BlockMultiFileSliceStore store1 = testLoadSliceStore(vaultId1);
      assertNotNull(store1);

      assertTrue(store1.getAllPaths().size() == 1);

      testLoadVaultPermissions(vaultId1);

      UUID vaultId2 = UUID.fromString("b3f66ec1-535d-3770-ac38-222222222222");
      testCreateSliceStore(vaultId2, this.maxRecords / 3);

      assertTrue(this.manager.getRemainingSize() == ((this.maxRecords - (this.maxRecords / 5 + this.maxRecords / 3)) * (RECORD_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead())));
      BlockMultiFileSliceStore store2 = testLoadSliceStore(vaultId2);
      assertNotNull(store2);

      assertTrue(store2.getAllPaths().size() == 2);

      UUID vaultId3 = UUID.fromString("b3f66ec1-535d-3770-ac38-333333333333");
      testCreateSliceStore(vaultId3, this.maxRecords / 3);

      assertTrue(this.manager.getRemainingSize() == ((this.maxRecords - (this.maxRecords / 5
            + this.maxRecords / 3 + this.maxRecords / 3)) * (RECORD_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead())));
      BlockMultiFileSliceStore store3 = testLoadSliceStore(vaultId3);
      assertNotNull(store3);

      assertTrue(store3.getAllPaths().size() == 1);

      try
      {
         this.manager.createSliceStore("block", RECORD_SIZE, (this.maxRecords / 10) * RECORD_SIZE,
               vaultId3, this.vaultACL, null, this.options);
         fail("Created second store for the same id");
      }
      catch (SliceStoreExistsException ex)
      {
         // expected
      }
      catch (SliceStoreIOException e)
      {
         fail("Unexpected exception: " + e.getMessage());
      }

      UUID vaultId4 = UUID.fromString("b3f66ec1-535d-3770-ac38-444444444444");
      try
      {
         this.manager.createSliceStore("block", RECORD_SIZE, (this.maxRecords / 3) * RECORD_SIZE,
               vaultId4, this.vaultACL, null, this.options);
         fail("Created second store for the same id");
      }
      catch (SliceStoreExistsException e)
      {
         fail("Unexpected exception: " + e.getMessage());
      }
      catch (SliceStoreIOException e)
      {
         // expected
      }

      // Cleanup
      testDeleteSliceStore(vaultId1);
      assertTrue(this.manager.getRemainingSize() == ((this.maxRecords - (this.maxRecords / 3 + this.maxRecords / 3)) * (RECORD_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead())));

   }
   //final private int REPEAT_LOOP = 10;
   final int nWorkers = 4;
   final int repeatCount = 10;
   final private CountDownLatch countDown = new CountDownLatch(this.nWorkers);

   @Test
   public void stressTest()
   {
      for (int i = 0; i < this.nWorkers; i++)
      {
         // Total # of records less then maxRecords
         new Thread(new Worker(this.repeatCount, i, this.maxRecords / (2 << i))).start();
      }
      try
      {
         this.countDown.await(30, TimeUnit.SECONDS);
         assertTrue(this.manager.getRemainingSize() == this.maxRecords
               * (RECORD_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead()));
      }
      catch (InterruptedException e)
      {
         fail("Unexpected exception: " + e.getMessage());
         e.printStackTrace();
      }
   }
   
   @Test
   public void testIntegrityCheck()
   {

   }
   
   @Test
   public void testGetSizeInBytes() throws IllegalConfigurationContentException
   {
      assertEquals(1L, BlockMultiFileSliceStoreManager.getSizeInBytes("1"));
      assertEquals(1024L, BlockMultiFileSliceStoreManager.getSizeInBytes("1K"));
      assertEquals(1024L, BlockMultiFileSliceStoreManager.getSizeInBytes("1KB"));

      assertEquals(10240L, BlockMultiFileSliceStoreManager.getSizeInBytes("10KB"));

      assertEquals(1048576L, BlockMultiFileSliceStoreManager.getSizeInBytes("1M"));
      assertEquals(1048576L, BlockMultiFileSliceStoreManager.getSizeInBytes("1MB"));
      
      assertEquals(1073741824L, BlockMultiFileSliceStoreManager.getSizeInBytes("1G"));
      assertEquals(1073741824L, BlockMultiFileSliceStoreManager.getSizeInBytes("1GB"));
      
      assertEquals(1099511627776L, BlockMultiFileSliceStoreManager.getSizeInBytes("1T"));
      assertEquals(1099511627776L, BlockMultiFileSliceStoreManager.getSizeInBytes("1TB"));
      
      try
      {
         BlockMultiFileSliceStoreManager.getSizeInBytes("xTB");
         fail("Method did not fail as expected");
      } catch(IllegalConfigurationContentException ex)
      {
         // Expected failure
      }

      try
      {
         BlockMultiFileSliceStoreManager.getSizeInBytes("1PB");
         fail("Method did not fail as expected");
      } catch(IllegalConfigurationContentException ex)
      {
         // Expected failure
      }
      
      try
      {
         BlockMultiFileSliceStoreManager.getSizeInBytes("test");
         fail("Method did not fail as expected");         
      } catch(IllegalConfigurationContentException ex)
      {
         // Expected failure         
      }
   }
   
   /*--------------- Helpers -------------------*/
   public void testCreateSliceStore(UUID vaultId, int numRecords)
   {
      try
      {
         this.manager.createSliceStore("block", RECORD_SIZE, numRecords * RECORD_SIZE, vaultId,
               this.vaultACL, null, this.options);
      }
      catch (SliceStoreExistsException e)
      {
         fail("Slice Store Exists");
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice Store IO Exception");
      }
   }

   public BlockMultiFileSliceStore testLoadSliceStore(UUID vaultId)
   {
      try
      {
         SliceStore sliceStore = this.manager.loadSliceStore(vaultId);
         return (BlockMultiFileSliceStore) sliceStore;
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("slice store not found exception");
      }
      return null;
   }

   public void testLoadVaultPermissions(UUID vaultId)
   {
      UnitTests.resetConfiguration();
      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("test");
      credentials.setPassword("test");

      try
      {
         VaultPermission permission =
               this.manager.loadVaultPermissions(vaultId, credentials.getAccountIdentifier());
         assertNotNull(permission);
      }
      catch (SliceStoreLayerException e)
      {
         e.printStackTrace();
         fail("slice store exception");
      }
   }

   public void testDeleteSliceStore(UUID vaultId)
   {
      try
      {
         this.manager.deleteSliceStore(vaultId, this.options);
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice Store IO exception");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   private FileVaultACL loadACL() throws Exception
   {
      final String aclFileName =
            System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
                  + "/org/cleversafe/layer/slicestore/SliceStoreManager.ACL.der";

      /*
       * FileOutputStream out = new FileOutputStream(aclFileName);
       * 
       * UUID vaultIdentifier = UUID.randomUUID(); this.credentials = new PasswordCredentials();
       * this.credentials.setUsername("test"); this.credentials.setPassword("test");
       * 
       * System.out.println( "Creating Vault ACL for account: " +
       * this.credentials.getAccountIdentifier().toString() );
       * 
       * KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA"); keygen.initialize( 2048 );
       * KeyPair keypair = keygen.generateKeyPair();
       * 
       * VaultACLFactory fact = new VaultACLFactory(); fact.create( out, vaultIdentifier,
       * this.credentials, keypair.getPublic() ); out.close();
       */
      FileInputStream in = new FileInputStream(aclFileName);
      return new FileVaultACL(in);
   }

   class Worker implements Runnable
   {
      int repeatCount;
      int index;
      int maxRecords;

      Worker(int repeatCoiunt, int i, int maxRecords)
      {
         this.repeatCount = repeatCoiunt;
         this.index = i;
         this.maxRecords = maxRecords;
      }

      public void run()
      {
         // Make it the last part of UUID
         String indStrbase = Integer.toString(this.index);
         String indStr = "";
         while (indStr.length() < 12)
         {
            indStr += indStrbase;
         }
         indStr = indStr.substring(0, 12);

         while (this.repeatCount-- > 0)
         {
            UUID vaultId = UUID.fromString("b3f66ec1-535d-3770-ac38-" + indStr);
            testCreateSliceStore(vaultId, this.maxRecords);
            //System.out.println("" + maxRecords + " --> " + manager.getRemainingSize());

            testDeleteSliceStore(vaultId);
            //System.out.println("" + maxRecords + " <-- " + manager.getRemainingSize());
         }
         BlockMultiSliceStoreManagerTest.this.countDown.countDown();
         //System.out.println("---> Countdown=" + countDown.getCount());
      }

   }
}
