//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: Jason Resch
//
// Date: Aug 22, 2007
//---------------------

package org.cleversafe.layer.slicestore.unreliable;

import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.UnitTests;
import org.cleversafe.exceptions.BaseException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.block.BlockFileSliceStore;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.test.BaseTest;
import org.cleversafe.vault.FileVaultACL;
import org.cleversafe.vault.VaultACL;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class UnreliableSliceStoreTest
{
   private static Logger _logger = Logger.getLogger(UnreliableSliceStoreTest.class);

   private static String basePath;
   private UnreliableSliceStore unreliableStore = null;

   private static int blockLength = 512;

   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      UnitTests.resetConfiguration();
      basePath =
            System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
                  + "/org/cleversafe/layer/slicestore/unreliable/block";
      // Ensure directory is empty before running test
      File blockFilePath = new File(basePath);
      deleteDirectory(blockFilePath);
   }

   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
      // Clean up directory after running test
      File blockFilePath = new File(basePath);
      deleteDirectory(blockFilePath);
   }

   @Before
   public void setUp() throws Exception
   {
      _logger.debug("Creating BlockFileSliceStore to be wrapped with an unreliabe store");
      BlockFileSliceStore blockFileStore = new BlockFileSliceStore(new UUID(0, 0), basePath);

      unreliableStore = new UnreliableSliceStore(blockFileStore);
      unreliableStore.createStore("block", blockLength, SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED,
            this.createACL(), null);
      unreliableStore.startSession();
   }

   @After
   public void tearDown() throws Exception
   {
      unreliableStore.endSession();
      unreliableStore.deleteStore();
   }

   @Test
   public final void testWorksReliably()
   {
      try
      {
         long transactionId = 0;
         SliceStoreTransaction transaction = unreliableStore.createTransaction(transactionId);
         unreliableStore.beginTransaction(transaction);
         unreliableStore.write(new DataSlice(new SliceName("1", 0), transactionId,
               new byte[blockLength]));
         unreliableStore.commitTransaction(transaction);
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice store transaction exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreQuotaException e)
      {
         fail("Slice store quota exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public final void testSetUnreliable()
   {
      String unreliability =
            "write,org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException,1.0";
      unreliableStore.setUnreliability(unreliability);

      try
      {
         long transactionId = 0;
         SliceStoreTransaction transaction = unreliableStore.createTransaction(transactionId);
         unreliableStore.beginTransaction(transaction);
         unreliableStore.write(new DataSlice(new SliceName("1", 0), transactionId,
               new byte[blockLength]));
         unreliableStore.commitTransaction(transaction);
         fail("Expected slice store IO exception to be thrown");
      }
      catch (SliceStoreIOException e)
      {
         // Expected result
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice store transaction exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreQuotaException e)
      {
         fail("Slice store quota exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public final void testSetUnreliableMethodUnsupportedTransaction() throws NoSuchMethodException
   {
      try
      {
         unreliableStore.setMethodExceptionRate("write", BaseException.class, 0.5);
         fail("Expected illegal argument exception");
      }
      catch (IllegalArgumentException ex)
      {
         // Expected behavior
      }
   }

   @Test
   public final void testSetUnreliableNonexistantMethod()
   {
      try
      {
         unreliableStore.setMethodExceptionRate("noSuchMethod", BaseException.class, 0.5);
         fail("Expected no such method exception to be thrown");
      }
      catch (NoSuchMethodException e)
      {
         // Expected behavior
      }
   }

   private static boolean deleteDirectory(File dir)
   {
      if (dir.isDirectory())
      {
         String[] children = dir.list();
         for (int i = 0; i < children.length; i++)
         {
            boolean success = deleteDirectory(new File(dir, children[i]));

            if (!success)
            {
               return false;
            }
         }
      }

      // The directory is now empty so delete it
      return dir.delete();
   }

   private VaultACL createACL() throws Exception
   {
      UUID vaultIdentifier = UUID.nameUUIDFromBytes("RemoteSliceStore".getBytes());

      final String aclFileName =
            System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".") + "/org/cleversafe/vault/"
                  + vaultIdentifier.toString() + ".der";

      /*
       * File aclFile = new File(aclFileName);
       * aclFile.getParentFile().mkdirs();
       * 
       * FileOutputStream out = new FileOutputStream(aclFileName);
       * 
       * PasswordCredentials credentials = new PasswordCredentials();
       * credentials.setUsername("username");
       * credentials.setPassword("password");
       * 
       * System.out.println( "Creating Vault ACL for account: " +
       * credentials.getAccountIdentifier().toString() );
       * 
       * KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
       * keygen.initialize( 2048 ); KeyPair keypair =
       * keygen.generateKeyPair();
       * 
       * VaultACLFactory fact = new VaultACLFactory(); fact.create( out,
       * vaultIdentifier, credentials, keypair.getPublic() ); out.close();
       */

      FileInputStream in = new FileInputStream(aclFileName);
      return new FileVaultACL(in);
   }

}
