//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Dusty Hendrickson
//
// Date: Apr 17, 2007
//---------------------

package org.cleversafe.serialization.asn1;

import static org.junit.Assert.assertEquals;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.util.Arrays;
import java.util.Date;
import java.util.Vector;

import org.bouncycastle.asn1.DERBoolean;
import org.bouncycastle.asn1.DERGeneralString;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERSequenceGenerator;
import org.bouncycastle.asn1.DERUTCTime;
import org.cleversafe.serialization.GridSerializable;
import org.cleversafe.test.BaseTest;
import org.junit.Test;

/**
 * Tests the extendable ASN1 class.
 * 
 * @see ASN1
 */
public class ASN1Test
{
   static public class OtherClass implements GridSerializable
   {
      private int intN = 0;
      private boolean boolN = false;

      public OtherClass()
      {
      }

      public OtherClass(int intN, boolean boolN)
      {
         this.intN = intN;
         this.boolN = boolN;
      }

      public boolean equals(OtherClass o)
      {
         return this.intN == o.intN && this.boolN == o.boolN;
      }

      @Override
      public String toString()
      {
         return "" + this.intN + "," + this.boolN;
      }

      public boolean isBoolN()
      {
         return this.boolN;
      }

      public void setBoolN(boolean boolN)
      {
         this.boolN = boolN;
      }

      public int getIntN()
      {
         return this.intN;
      }

      public void setIntN(int intN)
      {
         this.intN = intN;
      }
   }

   /**
    * A sample class that extends ASN1 that contains all supported encodable data types as well as
    * one unsupported type.
    */
   public static class TestClass extends OtherClass
   {
      @GridSerializable.OwnSequence()
      private OtherClass nestedC;

      @GridSerializable.ElementOrder(order = 1)
      private byte byteP;
      private Byte byteO;
      private byte[] byteA;

      @GridSerializable.ElementOrder(order = 2)
      private int intP;
      private int[] intA;
      private Integer intO;

      @GridSerializable.ElementOrder(order = 3)
      private boolean boolP;
      private Boolean boolO;
      private boolean[] boolA;

      private Date date;
      private String string;
      private String[] stringA;

      @GridSerializable.ElementOrder(order = 4)
      private short shortP;
      private Short shortO;
      private short[] shortA;

      @GridSerializable.ElementOrder(order = 5)
      private long longP;
      private Long longO;
      private Long[] longA;

      @GridSerializable.ElementOrder(order = 6)
      private char charP;
      private Character charO;
      private char[] charA;

      @GridSerializable.NotSerializable
      private Vector<?> unhandledType;

      public TestClass(
            OtherClass base,
            OtherClass nested,
            int intP,
            Integer intO,
            int[] intA,
            boolean boolP,
            Boolean boolO,
            boolean[] boolA,
            Date date,
            String string,
            String[] stringA,
            byte byteP,
            Byte byteO,
            byte[] byteA,
            short shortP,
            Short shortO,
            short[] shortA,
            long longP,
            Long longO,
            Long[] longA,
            char charP,
            Character charO,
            char[] charA)
      {
         super(base.intN, base.boolN);

         this.nestedC = nested;
         this.intP = intP;
         this.intO = intO;
         this.intA = intA;
         this.boolP = boolP;
         this.boolO = boolO;
         this.boolA = boolA;
         this.date = date;

         this.string = string;
         this.stringA = stringA;

         this.byteP = byteP;
         this.byteO = byteO;
         this.byteA = byteA;

         this.shortP = shortP;
         this.shortO = shortO;
         this.shortA = shortA;
         this.unhandledType = null;

         this.longP = longP;
         this.longO = longO;
         this.longA = longA;

         this.charP = charP;
         this.charO = charO;
         this.charA = charA;

      }

      @Override
      public boolean equals(Object obj)
      {
         TestClass testClass = (TestClass) obj;

         if (super.equals(testClass) && this.nestedC.equals(testClass.nestedC)
               && this.intP == testClass.intP && this.intO.equals(testClass.intO)
               && Arrays.equals(this.intA, testClass.intA) && this.boolP == testClass.boolP
               && this.boolO.equals(testClass.boolO) && Arrays.equals(this.boolA, testClass.boolA)
               && this.date.equals(testClass.date) && this.string.equals(testClass.string)
               && Arrays.equals(this.stringA, testClass.stringA) && this.byteP == testClass.byteP
               && this.byteO.equals(testClass.byteO) && Arrays.equals(this.byteA, testClass.byteA)
               && this.shortP == testClass.shortP && this.shortO.equals(testClass.shortO)
               && Arrays.equals(this.shortA, testClass.shortA) && this.longP == testClass.longP
               && this.longO.equals(testClass.longO) && Arrays.equals(this.longA, testClass.longA)
               && this.charP == testClass.charP && this.charO.equals(testClass.charO)
               && Arrays.equals(this.charA, testClass.charA)

         )
         {
            return true;
         }

         return false;
      }

      @Override
      public String toString()
      {
         // don't print array
         return "{{" + super.toString() + "}}" + "{" + this.nestedC + "}" + this.intP + ","
               + this.intO + "," + Arrays.toString(this.intA) + "," + this.boolP + ", "
               + this.boolO + ", " + Arrays.toString(this.boolA) + "," + this.date + ", "
               + this.string + ", " + Arrays.toString(this.stringA) + "," + this.byteP + ", "
               + this.byteO + ", " + Arrays.toString(this.byteA) + "," + this.shortP + ", "
               + this.shortO + Arrays.toString(this.shortA) + this.longP + ", " + this.longO
               + Arrays.toString(this.longA) + this.charP + ", " + this.charO
               + Arrays.toString(this.charA);
      }

      public boolean[] getBoolA()
      {
         return this.boolA;
      }

      public void setBoolA(boolean[] boolA)
      {
         this.boolA = boolA;
      }

      public Boolean getBoolO()
      {
         return this.boolO;
      }

      public void setBoolO(Boolean boolO)
      {
         this.boolO = boolO;
      }

      public boolean isBoolP()
      {
         return this.boolP;
      }

      public void setBoolP(boolean boolP)
      {
         this.boolP = boolP;
      }

      public byte[] getByteA()
      {
         return this.byteA;
      }

      public void setByteA(byte[] byteA)
      {
         this.byteA = byteA;
      }

      public Byte getByteO()
      {
         return this.byteO;
      }

      public void setByteO(Byte byteO)
      {
         this.byteO = byteO;
      }

      public byte getByteP()
      {
         return this.byteP;
      }

      public void setByteP(byte byteP)
      {
         this.byteP = byteP;
      }

      public Date getDate()
      {
         return this.date;
      }

      public void setDate(Date date)
      {
         this.date = date;
      }

      public int[] getIntA()
      {
         return this.intA;
      }

      public void setIntA(int[] intA)
      {
         this.intA = intA;
      }

      public Integer getIntO()
      {
         return this.intO;
      }

      public void setIntO(Integer intO)
      {
         this.intO = intO;
      }

      public int getIntP()
      {
         return this.intP;
      }

      public void setIntP(int intP)
      {
         this.intP = intP;
      }

      public OtherClass getNestedC()
      {
         return this.nestedC;
      }

      public void setNestedC(OtherClass nestedC)
      {
         this.nestedC = nestedC;
      }

      public short[] getShortA()
      {
         return this.shortA;
      }

      public void setShortA(short[] shortA)
      {
         this.shortA = shortA;
      }

      public Short getShortO()
      {
         return this.shortO;
      }

      public void setShortO(Short shortO)
      {
         this.shortO = shortO;
      }

      public short getShortP()
      {
         return this.shortP;
      }

      public void setShortP(short shortP)
      {
         this.shortP = shortP;
      }

      public String getString()
      {
         return this.string;
      }

      public void setString(String string)
      {
         this.string = string;
      }

      public String[] getStringA()
      {
         return this.stringA;
      }

      public void setStringA(String[] stringA)
      {
         this.stringA = stringA;
      }

      public Vector<?> getUnhandledType()
      {
         return this.unhandledType;
      }

      public void setUnhandledType(Vector<?> unhandledType)
      {
         this.unhandledType = unhandledType;
      }

      public long getLongP()
      {
         return this.longP;
      }

      public void setLongP(long longP)
      {
         this.longP = longP;
      }

      public Long[] getLongA()
      {
         return this.longA;
      }

      public void setLongA(Long[] longA)
      {
         this.longA = longA;
      }

      public Long getLongO()
      {
         return this.longO;
      }

      public void setLongO(Long longO)
      {
         this.longO = longO;
      }

      public char[] getCharA()
      {
         return this.charA;
      }

      public void setCharA(char[] charA)
      {
         this.charA = charA;
      }

      public Character getCharO()
      {
         return this.charO;
      }

      public void setCharO(Character charO)
      {
         this.charO = charO;
      }

      public char getCharP()
      {
         return this.charP;
      }

      public void setCharP(char charP)
      {
         this.charP = charP;
      }

   }

   /**
    * Tests the encode()/decode() methods of ASN1. It instantiates two classes, each with different
    * values. The first class is encoded into an ASN.1 byte stream. The second class then decodes
    * the byte stream produced by the first class. The test will pass if the first class and the
    * second class are equal in the end.
    * 
    * @throws Exception
    */
   @Test
   public void testEncodeDecode()
   {
      try
      {
         // Instantiate the first test class
         TestClass testClass1 =
               new TestClass(new OtherClass(999, true), new OtherClass(10, true), 1, 2, null, true,
                     false, new boolean[]{
                           false, true, false
                     }, new Date(0), "hello world", new String[]{
                           "Hello 1", "Hello 2"
                     }, (byte) 3, new Byte((byte) 4), new byte[]{
                           10, 20, 30, 40
                     }, (short) 55, new Short((short) 56), new short[]{
                           57, 58, 59
                     }, 25L << 34, new Long(25L << 34 + 10), new Long[]{
                           25L << 34 + 55, 25L << 34 + 66
                     }, 'a', 'b', new char[]{
                           'd', 'e', 'f', 'g'
                     });
         System.out.println(testClass1);
         // Produce ASN.1 encoded byte stream
         byte[] encodedBytes = ASN1.DERConverter.encode(testClass1);
         // System.out.println(ASN1.DERConverter.serializationMap(TestClass.class));

         File f =
               new File(System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".") + "/ASN1Test.der");
         f.getParentFile().mkdirs();
         f.createNewFile();
         FileOutputStream fo = new FileOutputStream(f);
         fo.write(encodedBytes);
         fo.close();

         // Instantiate the second test class
         TestClass testClass2 =
               new TestClass(new OtherClass(888, true), new OtherClass(11, false), 3, 4, new int[]{
                     5, 6, 7, 8
               }, false, true, new boolean[]{
                     true, false, true
               }, new Date(1), "world hello", new String[]{}, (byte) 5, new Byte((byte) 5),
                     new byte[]{
                           50, 60, 70, 80
                     }, (short) 61, new Short((short) 62), new short[]{
                           63, 64
                     }, 25L << 33, new Long(25L << 33 + 10), new Long[]{
                           25L << 33 + 55, 25L << 33 + 66
                     }, 'A', 'B', new char[]{
                           'D', 'E'
                     });
         System.out.println(testClass2);
         // Decode the ASN.1 encoded byte stream
         ASN1.DERConverter.decode(testClass2, encodedBytes);

         // Check if the first class is equal to the second class
         System.out.println(testClass2);
         assertEquals(testClass1, testClass2);
      }
      catch (Exception e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
   }

   public static void main(String[] args) throws Exception
   {
      // Instantiate the first test class
      TestClass testClass1 =
            new TestClass(new OtherClass(777, true), new OtherClass(11, false), 3, 4, new int[]{
                  5, 6, 7, 8
            }, false, true, new boolean[]{
                  true, false, true
            }, new Date(1), "world hello", new String[]{}, (byte) 5, new Byte((byte) 5),
                  new byte[]{
                        50, 60, 70, 80
                  }, (short) 61, new Short((short) 62), new short[]{
                        63, 64
                  }, 25L << 39, new Long(25L << 39 + 10), new Long[]{
                        25L << 39 + 55, 25L << 39 + 66
                  }, 'A', 'B', new char[]{
                        'D', 'E'
                  }

            );
      // Class registration speed test
      int numRegisterationIterations =
            args.length > 0 && args[0] != null ? Integer.parseInt(args[0]) : 100;
      long timeBefore = System.currentTimeMillis();
      for (int i = 0; i < numRegisterationIterations; i++)
      {
         ASN1.DERConverter.registerASN1Order(testClass1.getClass());
         ASN1.DERConverter.clearCachedOrders();
      }
      long timeAfter = System.currentTimeMillis();
      long timeTaken = timeAfter - timeBefore;
      double timeTakenPerIteration = ((double) timeTaken) / (double) numRegisterationIterations;

      System.out.println("Registration:");
      System.out.println("  Number of iterations: " + numRegisterationIterations);
      System.out.println("  Total time taken: " + timeTaken + " ms");
      System.out.println("  Time taken per iteration: " + timeTakenPerIteration + " ms");

      // Encode/Decode speed test
      ASN1.DERConverter.registerASN1Order(testClass1.getClass());

      int numEncodeDecodeIterations =
            args.length > 1 && args[1] != null ? Integer.parseInt(args[1]) : 100;
      timeBefore = System.currentTimeMillis();

      for (int i = 0; i < numEncodeDecodeIterations; i++)
      {
         TestClass testClass =
               new TestClass(new OtherClass(666, true), new OtherClass(11, false), 3, 4, new int[]{
                     5, 6, 7, 8
               }, false, true, new boolean[]{
                     true, false, true
               }, new Date(1), "world hello", new String[]{}, (byte) 5, new Byte((byte) 5),
                     new byte[]{
                           50, 60, 70, 80
                     }, (short) 61, new Short((short) 62), new short[]{
                           63, 64
                     }, 25L << 41, new Long(25L << 41 + 10), new Long[]{
                           25L << 41 + 55, 25L << 41 + 66
                     }, 'a', 'b', new char[]{
                           'd', 'e', 'f', 'g'
                     });
         byte[] encodedBytes = ASN1.DERConverter.encode(testClass);
         encodedBytes[0] = 0;
         // ASN1.DERConverter.decode(testClass1, encodedBytes);
      }

      timeAfter = System.currentTimeMillis();
      timeTaken = timeAfter - timeBefore;
      double timeTakenPerIterationReflect =
            ((double) timeTaken) / (double) numEncodeDecodeIterations;

      System.out.println("\nEncode (reflect):");
      System.out.println("  Number of iterations: " + numEncodeDecodeIterations);
      System.out.println("  Total time taken: " + timeTaken + " ms");
      System.out.println("  Time taken per iteration: " + timeTakenPerIterationReflect + " ms");

      timeBefore = System.currentTimeMillis();
      for (int i = 0; i < numEncodeDecodeIterations; i++)
      {
         TestClass t =
               new TestClass(new OtherClass(555, true), new OtherClass(11, false), 3, 4, new int[]{
                     5, 6, 7, 8
               }, false, true, new boolean[]{
                     true, false, true
               }, new Date(1), "world hello", new String[]{}, (byte) 5, new Byte((byte) 5),
                     new byte[]{
                           50, 60, 70, 80
                     }, (short) 61, new Short((short) 62), new short[]{
                           63, 64
                     }, 25L << 45, new Long(25L << 45 + 10), new Long[]{
                           25L << 45 + 55, 25L << 45 + 66
                     }, 'a', 'b', new char[]{
                           'd', 'e', 'f', 'g'
                     });

         ByteArrayOutputStream out = new ByteArrayOutputStream();
         DERSequenceGenerator derGenerator = new DERSequenceGenerator(out);

         derGenerator.addObject(new DERInteger(t.getNestedC().getIntN()));
         derGenerator.addObject(new DERBoolean(t.getNestedC().isBoolN()));

         derGenerator.addObject(new DERInteger(t.getIntP()));
         derGenerator.addObject(new DERInteger(t.getIntO()));
         int[] v1 = t.getIntA();
         DERInteger[] derIntegers = new DERInteger[v1.length];
         for (int a = 0; a < v1.length; a++)
         {
            derIntegers[a] = new DERInteger(v1[a]);
         }
         derGenerator.addObject(new DERSequence(derIntegers));

         derGenerator.addObject(new DERBoolean(t.isBoolP()));
         derGenerator.addObject(new DERBoolean(t.getBoolO()));
         boolean[] v2 = t.getBoolA();
         DERBoolean[] derBoolean = new DERBoolean[v2.length];
         for (int a = 0; a < v2.length; a++)
         {
            derBoolean[a] = new DERBoolean(v2[a]);
         }
         derGenerator.addObject(new DERSequence(derBoolean));

         derGenerator.addObject(new DERUTCTime(t.getDate()));

         derGenerator.addObject(new DERGeneralString(t.getString()));
         String[] vs = t.getStringA();
         DERGeneralString[] derString = new DERGeneralString[vs.length];
         for (int a = 0; a < vs.length; a++)
         {
            derString[a] = new DERGeneralString(vs[a]);
         }
         derGenerator.addObject(new DERSequence(derString));

         derGenerator.addObject(new DERInteger(t.getByteP()));
         derGenerator.addObject(new DERInteger(t.getByteO()));
         byte[] v3 = t.getByteA();
         DERInteger[] derByte = new DERInteger[v3.length];
         for (int a = 0; a < v3.length; a++)
         {
            derByte[a] = new DERInteger(v3[a]);
         }
         derGenerator.addObject(new DERSequence(derByte));

         derGenerator.addObject(new DERInteger(t.getShortP()));
         derGenerator.addObject(new DERInteger(t.getShortO()));
         short[] v4 = t.getShortA();
         DERInteger[] derShort = new DERInteger[v4.length];
         for (int a = 0; a < v4.length; a++)
         {
            derByte[a] = new DERInteger(v4[a]);
         }
         derGenerator.addObject(new DERSequence(derShort));

         derGenerator.close();
         out.close();
      }
      timeAfter = System.currentTimeMillis();
      timeTaken = timeAfter - timeBefore;
      double timeTakenPerIterationDirect =
            ((double) timeTaken) / (double) numEncodeDecodeIterations;

      System.out.println("\nEncode (direct):");
      System.out.println("  Number of iterations: " + numEncodeDecodeIterations);
      System.out.println("  Total time taken: " + timeTaken + " ms");
      System.out.println("  Time taken per iteration: " + timeTakenPerIterationDirect + " ms");

      System.out.println("\nRatio:" + timeTakenPerIterationReflect / timeTakenPerIterationDirect);

   }

}
