//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Jul 20, 2007
//---------------------

package org.cleversafe.serialization.asn1;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileOutputStream;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.List;

import org.cleversafe.exceptions.BaseException;
import org.cleversafe.layer.protocol.BeginSessionRequest;
import org.cleversafe.layer.protocol.GridProtocolOperation;
import org.cleversafe.layer.protocol.ProtocolOperation;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.exceptions.OperationException;
import org.cleversafe.layer.protocol.exceptions.OperationNotDefinedException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedSliceStoreTypeException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedVaultTypeException;
import org.cleversafe.server.exceptions.ServerRequestException;
import org.cleversafe.test.BaseTest;
import org.junit.Before;
import org.junit.Test;

// TODO: Describe class or interface

public class ExceptionRegistratorTest2
{
   private final Class<?>[] stringBasedExceptionClasses =
         {
               org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException.class,
               org.cleversafe.layer.protocol.exceptions.ProtocolDeserializationException.class,

               // Slice store exceptions
               org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException.class,
               org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException.class,
               org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException.class,
               org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException.class,
               org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException.class,
               org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException.class,

               // Server exceptions
               org.cleversafe.server.exceptions.ServerConfigurationLoadException.class,
               org.cleversafe.server.exceptions.IllegalServerConfigurationContentException.class,
         };

   private final Class<?>[] serverRequestBasedExceptionClasses =
         {
               org.cleversafe.server.exceptions.ServerRequestException.class,
               org.cleversafe.server.exceptions.UnauthorizedRequestException.class,
               org.cleversafe.server.exceptions.UnrecognizedRequestException.class,
               org.cleversafe.server.exceptions.IllegalSessionStateException.class,
         };

   private final Class<?>[] protocolOperationBasedExceptionClasses =
         {
               org.cleversafe.layer.protocol.exceptions.OperationNotRegisteredException.class,
               org.cleversafe.layer.protocol.exceptions.OperationAlreadyRegisteredException.class,
         };

   @Before
   public void setup()
   {
      System.clearProperty(ExceptionRegistrator.ERROR_DEFINITION_PROPERTY);
      ASN1ExceptionWrapper.clearRegistry();
      ExceptionRegistrator.loadAndRegister();
   }

   public BaseException testException(BaseException originalException) throws Throwable
   {
      ASN1ExceptionWrapper wrapperExc = new ASN1ExceptionWrapper(originalException);
      byte[] serializedOriginalException = ASN1.DERConverter.encode(wrapperExc);

      File f =
            new File(System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".") + "/"
                  + originalException.getClass().getName() + ".der");
      f.getParentFile().mkdirs();
      f.createNewFile();
      FileOutputStream fo = new FileOutputStream(f);
      fo.write(serializedOriginalException);
      fo.close();

      ASN1ExceptionWrapper restoredWrapperExc = new ASN1ExceptionWrapper();
      ASN1.DERConverter.decode(restoredWrapperExc, serializedOriginalException);

      assertTrue(restoredWrapperExc.getOriginalException() instanceof BaseException);
      BaseException restoredException = (BaseException) restoredWrapperExc.getOriginalException();

      // System.out.println("Original class: " + originalException.getClass().toString());
      // System.out.println("Restored class: " + restoreException.getClass().toString());
      assertEquals(originalException.getClass(), restoredException.getClass());

      // System.out.println("Original message: " + originalException.getMessage());
      // System.out.println("Restored message: " + restoreException.getMessage());
      assertEquals(originalException.getMessage(), restoredException.getMessage());

      // System.out.println("Original string: " + originalException.toString());
      // System.out.println("Restored string: " + restoreException.toString());
      assertEquals(originalException.toString(), restoredException.toString());

      // System.out.println();
      return restoredException;
   }

   public BaseException testStringBasedExceptionClass(Class<?> exceptionClass) throws Throwable
   {
      Constructor<?> exceptionConstructor = exceptionClass.getConstructor(new Class[]{
         String.class
      });

      Object exceptionObj = exceptionConstructor.newInstance("Testing " + exceptionClass.getName());

      assertTrue(exceptionObj instanceof BaseException);

      BaseException originalException = (BaseException) exceptionObj;
      return testException(originalException);
   }

   @Test
   public void testStringBasedExceptions()
   {
      List<String> errorList = new ArrayList<String>();
      for (Class<?> testClass : this.stringBasedExceptionClasses)
      {
         try
         {
            testStringBasedExceptionClass(testClass);
         }
         catch (Throwable e)
         {
            errorList.add("ERROR in " + testClass.getName() + " [" + e.getMessage() + "]");
         }
      }

      if (errorList.size() > 0)
      {
         for (String error : errorList)
         {
            System.out.println(error);
         }
         fail("Encoding/decoding failed for the following exceptions: " + errorList);
      }
   }

   public ServerRequestException testServerRequestBasedExceptionClass(Class<?> exceptionClass)
         throws Throwable
   {
      Constructor<?> exceptionConstructor = exceptionClass.getConstructor(new Class[]{
            String.class, Request.class
      });

      Object exceptionObj =
            exceptionConstructor.newInstance("Testing " + exceptionClass.getName(),
                  new BeginSessionRequest());

      assertTrue(exceptionObj instanceof ServerRequestException);

      ServerRequestException originalException = (ServerRequestException) exceptionObj;

      BaseException restoredBaseException = testException(originalException);
      assertTrue(restoredBaseException instanceof ServerRequestException);

      ServerRequestException restoredException = (ServerRequestException) restoredBaseException;
      assertEquals(restoredException.getOperation(), originalException.getOperation());
      return restoredException;
   }

   @Test
   public void testServerRequestBasedExceptions()
   {
      List<String> errorList = new ArrayList<String>();
      for (Class<?> testClass : this.serverRequestBasedExceptionClasses)
      {
         try
         {
            testServerRequestBasedExceptionClass(testClass);
         }

         catch (Throwable t)
         {
            errorList.add("ERROR in " + testClass.getName() + " [" + t.getMessage() + "]");
         }
      }

      if (errorList.size() > 0)
      {
         for (String error : errorList)
         {
            System.out.println(error);
         }
         fail("Encoding/decoding failed for the following exceptions: " + errorList);
      }
   }

   public OperationException testProtocolOperationBasedExceptionClass(Class<?> exceptionClass)
         throws Throwable
   {
      Constructor<?> exceptionConstructor = exceptionClass.getConstructor(new Class[]{
            String.class, ProtocolOperation.class
      });

      Object exceptionObj =
            exceptionConstructor.newInstance("Testing " + exceptionClass.getName(),
                  GridProtocolOperation.CREATE_STORE_REQUEST);

      assertTrue(exceptionObj instanceof OperationException);

      OperationException originalException = (OperationException) exceptionObj;

      BaseException restoredBaseException = testException(originalException);
      assertTrue(restoredBaseException instanceof OperationException);

      OperationException restoredException = (OperationException) restoredBaseException;
      assertEquals(restoredException.getOperation(), originalException.getOperation());
      return restoredException;
   }

   @Test
   public void testProtocolOperationBasedExceptions()
   {
      List<String> errorList = new ArrayList<String>();
      for (Class<?> testClass : this.protocolOperationBasedExceptionClasses)
      {
         try
         {
            testProtocolOperationBasedExceptionClass(testClass);
         }
         catch (Throwable t)
         {
            errorList.add("ERROR in " + testClass.getName() + " [" + t.getMessage() + "]");
         }
      }

      if (errorList.size() > 0)
      {
         for (String error : errorList)
         {
            System.out.println(error);
         }
         fail("Encoding/decoding failed for the following exceptions: " + errorList);
      }
   }

   @Test
   public void testOperationNotDefinedException()
   {
      String className = OperationNotDefinedException.class.getName();
      OperationNotDefinedException exception =
            new OperationNotDefinedException("Testing " + className, 1044);
      try
      {
         OperationNotDefinedException e = (OperationNotDefinedException) testException(exception);
         assertEquals(e.getOperationCode(), exception.getOperationCode());
      }
      catch (Throwable t)
      {
         System.out.println("ERROR in " + className + " [" + t.getMessage() + "]");
         fail(t.getMessage());
      }
   }

   @Test
   public void testUnsupportedSliceStoreTypeException()
   {
      String className = UnsupportedSliceStoreTypeException.class.getName();
      UnsupportedSliceStoreTypeException exception =
            new UnsupportedSliceStoreTypeException("Testing " + className, "block",
                  "block-multi-file");
      try
      {
         UnsupportedSliceStoreTypeException e =
               (UnsupportedSliceStoreTypeException) testException(exception);
         assertEquals(e.getVaultType(), exception.getVaultType());
         assertEquals(e.getSliceStoreType(), exception.getSliceStoreType());
      }
      catch (Throwable t)
      {
         System.out.println("ERROR in " + className + " [" + t.getMessage() + "]");
         fail(t.getMessage());
      }
   }

   @Test
   public void testUnsupportedVaultTypeException()
   {
      String className = UnsupportedVaultTypeException.class.getName();
      UnsupportedVaultTypeException exception =
            new UnsupportedVaultTypeException("Testing " + className, "block");
      try
      {
         UnsupportedVaultTypeException e = (UnsupportedVaultTypeException) testException(exception);
         assertEquals(e.getVaultType(), exception.getVaultType());
      }
      catch (Throwable t)
      {
         System.out.println("ERROR in " + className + " [" + t.getMessage() + "]");
         fail(t.getMessage());
      }
   }
}
