//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: gdhuse
//
// Date: Jul 18, 2007
//---------------------

package org.cleversafe.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.fail;

import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Logger;
import org.junit.Test;

/**
 * Unit test of the BoundedThreadPoolExecutor
 */
public class BoundedThreadPoolExcecutorTest
{
   private static Logger _logger = Logger.getLogger(BoundedThreadPoolExcecutorTest.class);
   
   Random rng = new Random();

   /**
    * Test class parameters
    */
   @Test
   public void parameterTest()
   {
      int numRuns = this.rng.nextInt(15);
      for (int i = 0; i < numRuns; ++i)
      {
         int numThreads = this.rng.nextInt(20)+1;
         BoundedThreadPoolExecutor executor = new BoundedThreadPoolExecutor("Test", numThreads);
         assertEquals(numThreads, executor.getBound());
      }
   }

   /**
    * Test a batch of jobs less than the bound
    */
   @Test
   public void testUnderBoundBatch()
   {
      testBatch(1000, 20);
   }
   
   /**
    * Test a batch of jobs larger than the bound
    */
   @Test
   public void testOverBoundBatch()
   {
      testBatch(25, 250);
   }
   
   public void testBatch(int numThreads, int numJobs)
   {
      _logger.info("Testing " + numJobs + " jobs with " + numThreads + " threads");
      final AtomicInteger jobsRun = new AtomicInteger(0);
      
      class TestCallable implements Callable<Void>
      {
         boolean throwException;
         
         public TestCallable(boolean throwException)
         {
            this.throwException = throwException;
         }
         
         public Void call() throws Exception
         {
            //_logger.info("Task executing");
            Thread.sleep(rng.nextInt(100));
            jobsRun.incrementAndGet();
            
            if(this.throwException)
            {
               throw new Exception();
            }
            return null;
         }
      }
      
      class TestRunnable implements Runnable
      {
         public void run()
         {
            //_logger.info("Task executing");
            try { Thread.sleep(rng.nextInt(100)); }
            catch(final InterruptedException e) {}
            jobsRun.incrementAndGet();
         }
      }
      
      BoundedThreadPoolExecutor executor = new BoundedThreadPoolExecutor("Test", numThreads);
      
      // Callable w/o exception
      jobsRun.set(0);
      for(int i = 0; i < numJobs; ++i)
      {
         Future<Void> future = executor.submit(new TestCallable(false));
         try{ future.get(); }
         catch(final Exception e) {}
      }
      assertEquals(jobsRun.get(), numJobs);
      
      // Callable w/exception
      jobsRun.set(0);
      for(int i = 0; i < numJobs; ++i)
      {
         Future<Void> future = executor.submit(new TestCallable(true));
         try
         { 
            future.get(); 
            assertFalse("This should not be reached", true);
         }
         catch(final Exception e) {}
      }
      assertEquals(jobsRun.get(), numJobs);
      
      // Runnable
      jobsRun.set(0);
      for(int i = 0; i < numJobs; ++i)
      {
         Future<?> future = executor.submit(new TestRunnable());
         try{ future.get(); }
         catch(final Exception e) {}
      }
      assertEquals(jobsRun.get(), numJobs);
      
      // Group execution
      jobsRun.set(0);
      List<Callable<Void>> jobList = new LinkedList<Callable<Void>>();
      for(int i = 0; i < numJobs; ++i)
      {
         jobList.add(new TestCallable(false));
      }
      try { executor.invokeAll(jobList); }
      catch(final Exception e) 
      {
         fail("Unexpected exception: " + e);
      }
      
      assertEquals(jobsRun.get(), numJobs);
   }   
}
