//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Jul 24, 2007
//---------------------

package org.cleversafe.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.IOException;

import org.cleversafe.test.BaseTest;
import org.junit.BeforeClass;
import org.junit.Test;

import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Switch;
import com.thoughtworks.xstream.io.StreamException;

// TODO: Describe class or interface
public class JSAPCommandLineParserTest
{
   @BeforeClass
   static public void setup()
   {
      String classPath = System.getProperty("java.class.path");
      classPath = "./bin/org/cleversafe/util/test-manifest.jar;" + classPath;
      System.setProperty("java.class.path", classPath);
      //      System.out.println(classPath);
   }

   @Test
   public void simpleCreateNoResource()
   {
      try
      {
         JSAPCommandLineParser parser =
               new JSAPCommandLineParser("Test format", "Test description");
         assertNull(parser.parse(new String[]{
            "--unknowFlag"
         }));
         assertNull(parser.parse(new String[]{
            "-u"
         }));
         assertNull(parser.parse(new String[]{
            "-h"
         }));
         assertNull(parser.parse(new String[]{
            "--help"
         }));
         assertNull(parser.parse(new String[]{
            "-v"
         }));
         assertNull(parser.parse(new String[]{
            "--verbose"
         }));
      }
      catch (Exception ex)
      {
         fail("unexpected exception" + ex.getMessage());
      }
   }

   @Test
   public void createNoResource()
   {
      try
      {
         JSAPCommandLineParser parser =
               new JSAPCommandLineParser("Test format", "Test description");
         parser.registerParameter(new Switch("TEST", 't', "test"));

         assertNull(parser.parse(new String[]{
            "-h"
         }));
         assertNull(parser.parse(new String[]{
            "--help"
         }));
         assertNull(parser.parse(new String[]{
            "-v"
         }));
         assertNull(parser.parse(new String[]{
            "--verbose"
         }));

         assertNotNull(parser.parse(new String[]{
            "--test"
         }));
         assertNotNull(parser.parse(new String[]{
            "-t"
         }));

      }
      catch (Exception ex)
      {
         fail("unexpected exception" + ex.getMessage());
      }
   }

   @Test(expected = IOException.class)
   public void createWithNotExistingResource() throws IOException
   {
      try
      {
         JSAPCommandLineParser parser =
               new JSAPCommandLineParser(JSAPCommandLineParser.class, "notExistingJSAP.xml",
                     "Test format", "Test description");
         assertNull(parser.parse(new String[]{
            "-h"
         }));

      }
      catch (JSAPException e)
      {
         fail("Unexpected JSAPException");
      }
   }

   @Test(expected = JSAPException.class)
   public void createWithBadResource() throws JSAPException
   {
      try
      {
         JSAPCommandLineParser parser =
               new JSAPCommandLineParser(JSAPCommandLineParser.class, "badJSAP.xml", "Test format",
                     "Test description");
         assertNotNull(parser);
      }
      catch (IOException e)
      {
         fail("Unexpected IOException");
      }
      catch (StreamException e)
      {
         throw new JSAPException("config problem", e);
      }
   }

   @Test
   public void createWithResource()
   {
      try
      {
         JSAPCommandLineParser parser =
               new JSAPCommandLineParser(JSAPCommandLineParser.class, "goodJSAP.xml",
                     "Test format", "Test description");
         assertNull(parser.parse(new String[]{
            "-h"
         }));
         assertNotNull(parser.parse(new String[]{
            "--verbose"
         }));

      }
      catch (Exception e)
      {
         fail("Unexpected Exception: " + e.getMessage());
      }
   }

   public void createWithResourceAndConf()
   {
      try
      {
         JSAPCommandLineParser parser =
               new JSAPCommandLineParser(JSAPCommandLineParser.class, "confJSAP.xml",
                     "Test format", "Test descripton");
         assertNull(parser.parse(new String[]{
            "-h"
         }));
         assertNotNull(parser.parse(new String[]{
            "--verbose"
         }));

         JSAPResult result = parser.parse(new String[]{});
         assertEquals("configuration-set-value", result.getString("data_string"));

         parser =
               new JSAPCommandLineParser(JSAPCommandLineParser.class, "confJSAP.xml",
                     "Test format", "Test descripton");
         result = parser.parse(new String[]{
            "--data-string=cmdline-set-value"
         });
         assertEquals("cmdline-set-value", result.getString("data_string"));

         parser =
               new JSAPCommandLineParser(JSAPCommandLineParser.class, "confJSAP.xml",
                     "Test format", "Test descripton");
         result =
               parser.parse(new String[]{
                     "-D",
                     System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
                           + "/org/cleversafe/util/conf-jsap-test2.conf"
               });
         System.out.println("configuration_file: " + result.getString("configuration_file"));
         assertEquals("configuration-set-value2", result.getString("data_string"));
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail("Unexpected Exception: " + e.getMessage());
      }
   }

   @Test
   public void testAttributes()
   {
      String versionInfo = JSAPCommandLineParser.getVersionInfo();
      int start = versionInfo.indexOf("TestJar");
      assertTrue(start != -1);
      int end = versionInfo.indexOf('\n', start);
      assertTrue(end != -1);
      String testVersion = versionInfo.substring(start, end);
      assertEquals("TestJar 0.6 built 6/27/07 11:32 AM", testVersion);
   }
}
