//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: May 5, 2008
//---------------------

package org.cleversafe.util.rangelock;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Logger;
import org.apache.log4j.extras.DOMConfigurator;
import org.cleversafe.util.rangelock.RangeReadWriteLock.Lock;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class RangeLockTest
{
   private static Logger _logger = Logger.getLogger(RangeLockTest.class);
   private static AtomicInteger threadCounter = new AtomicInteger(0);

   @BeforeClass
   static public void setup()
   {
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));
   }

   public interface RangeReadWriteFactory
   {
      RangeReadWriteLock getInstance();
   }

   public class RangeReadWriteFactory_WritePreference implements RangeReadWriteFactory
   {
      public RangeReadWriteLock getInstance()
      {
         return new RangeReadWriteLockWritePreference();
      }
   }

   public class RangeReadWriteFactory_Fair implements RangeReadWriteFactory
   {
      public RangeReadWriteLock getInstance()
      {
         return new RangeReadWriteLockFair();
      }
   }

   class SimpleRunnerWithSleeps extends Thread
   {
      private int sleepBeforeLoop;
      private int sleepInAquire;
      private int sleepAfter;
      private RangeReadWriteLock lock;
      private CountDownLatch latch;
      private int repeat = 1;

      private RangeReadWriteLock.Lock acquirer = null;

      SimpleRunnerWithSleeps(
            RangeReadWriteLock lock,
            CountDownLatch latch,
            int sleepBefore,
            int sleepInAcquire)
      {
         this(lock, latch, sleepBefore, sleepInAcquire, 1, 0);
      }

      SimpleRunnerWithSleeps(
            RangeReadWriteLock lock,
            CountDownLatch latch,
            int sleepBeforeLoop,
            int sleepInAquire,
            int repeat,
            int sleepAfter)
      {
         super("Runner-" + threadCounter.incrementAndGet());
         this.sleepBeforeLoop = sleepBeforeLoop;
         this.sleepInAquire = sleepInAquire;
         this.sleepAfter = sleepAfter;
         this.lock = lock;
         this.latch = latch;
         this.repeat = repeat;
      }

      public void setReadRangeLock(int start, int amount)
      {
         acquirer = this.lock.getReadRangeLock(start, amount);
      }

      public void setWriteRangeLock(int start, int amount)
      {
         acquirer = this.lock.getWriteRangeLock(start, amount);
      }

      public void run()
      {
         try
         {
            if (latch != null)
            {
               latch.countDown();
            }
            _logger.debug(this.getName() + " started");
            assert acquirer != null;

            Thread.sleep(this.sleepBeforeLoop);
            while (this.repeat-- > 0)
            {
               _logger.debug(this.getName() + " before lock acquire");
               acquirer.acquire();
               _logger.debug(this.getName() + " after lock acquired");
               Thread.sleep(this.sleepInAquire);
               _logger.debug(this.getName() + " before lock released");
               acquirer.release();
               Thread.sleep(this.sleepAfter);
            }
         }
         catch (Throwable e)
         {
            e.printStackTrace();
            fail("Unexpected excaption" + e);
         }
         _logger.debug(this.getName() + " exiting");
      }
   }

   @Before
   public void setTest()
   {
      threadCounter.set(0);
   }

   @After
   public void clearTest()
   {
      _logger.debug("Test complete");
   }

   @Test
   public void simple1ReadTest_WritePreference() throws InterruptedException
   {
      simple1ReadTest(new RangeReadWriteFactory_WritePreference());
   }

   @Test
   public void simple1ReadTest_Fair() throws InterruptedException
   {
      simple1ReadTest(new RangeReadWriteFactory_Fair());
   }

   public void simple1ReadTest(RangeReadWriteFactory factory) throws InterruptedException
   {
      final RangeReadWriteLock l = factory.getInstance();
      Thread thr = new Thread("Test-1")
      {
         public void run()
         {
            RangeReadWriteLock.Lock r = l.getReadRangeLock(1, 10);
            r.acquire();
            r.release();
         }
      };
      thr.start();
      Thread.sleep(100);
      if (thr.isAlive())
      {
         fail("Single reader lock failed");
      }
   }

   @Test
   public void simple1WriteTest_WritePreference() throws InterruptedException
   {
      simple1WriteTest(new RangeReadWriteFactory_WritePreference());
   }

   @Test
   public void simple1WriteTest_Fair() throws InterruptedException
   {
      simple1WriteTest(new RangeReadWriteFactory_Fair());
   }

   public void simple1WriteTest(RangeReadWriteFactory factory) throws InterruptedException
   {
      final RangeReadWriteLock l = factory.getInstance();
      Thread thr = new Thread("Test-1")
      {
         public void run()
         {
            RangeReadWriteLock.Lock w = l.getWriteRangeLock(3, 10);
            w.acquire();
            w.release();
         }
      };
      thr.start();
      Thread.sleep(100);
      if (thr.isAlive())
      {
         fail("Single writer lock failed");
      }
   }

   @Test
   public void manyReadersTest_WritePreference() throws InterruptedException
   {
      manyReadersTest(new RangeReadWriteFactory_WritePreference());
   }

   @Test
   public void manyReadersTest_Fair() throws InterruptedException
   {
      manyReadersTest(new RangeReadWriteFactory_Fair());
   }

   public void manyReadersTest(RangeReadWriteFactory factory) throws InterruptedException
   {
      final RangeReadWriteLock l = factory.getInstance();
      int readers = 10;
      final Thread[] thr = new Thread[readers];
      final CountDownLatch latch = new CountDownLatch(readers);

      for (int i = 0; i < readers; i++)
      {
         final int index = i;
         thr[i] = new Thread("Test-" + index)
         {
            public void run()
            {
               RangeReadWriteLock.Lock r = l.getReadRangeLock(1, index);
               latch.countDown();
               r.acquire();
               try
               {
                  Thread.sleep(100);
               }
               catch (InterruptedException e)
               {
                  // TODO Auto-generated catch block
                  e.printStackTrace();
               }
               r.release();
            }
         };
         thr[i].start();
      }

      latch.await();
      Thread.sleep(200);
      for (int i = 0; i < readers; i++)
      {
         if (thr[i].isAlive())
         {
            fail("Multiple reader locks took too long");
         }
      }
   }

   @Test
   public void manyWritersTest_WritePreference() throws InterruptedException
   {
      manyWritersTest(new RangeReadWriteFactory_WritePreference());
   }

   @Test
   public void manyWritersTest_Fair() throws InterruptedException
   {
      manyWritersTest(new RangeReadWriteFactory_Fair());
   }

   public void manyWritersTest(RangeReadWriteFactory factory) throws InterruptedException
   {
      final RangeReadWriteLock l = factory.getInstance();
      int writers = 10;
      Thread[] thr = new Thread[writers];
      final CountDownLatch latch = new CountDownLatch(writers);

      for (int i = 0; i < writers; i++)
      {
         final int index = i;
         thr[i] = new Thread("Test-" + index)
         {
            public void run()
            {
               RangeReadWriteLock.Lock r = l.getWriteRangeLock(1, index);
               latch.countDown();
               _logger.debug("Acquiring: " + index);
               r.acquire();
               _logger.debug("Acquired: " + index);
               try
               {
                  Thread.sleep(100);
               }
               catch (InterruptedException e)
               {
                  // TODO Auto-generated catch block
                  e.printStackTrace();
               }
               r.release();
               _logger.debug("Exiting: " + index);
            }
         };
         thr[i].start();
      }

      // Make sure that all threads started
      latch.await();
      Thread.sleep(50);

      if (!thr[9].isAlive())
      {
         fail("Multiple writers didn't wait for each other");

      }
      _logger.debug("Before sleep");
      Thread.sleep(1200);
      for (int i = 0; i < writers; i++)
      {
         if (thr[i].isAlive())
         {
            fail("Multiple writer locks took too long");
         }
      }
   }

   @Test
   public void writerPreemptReader_WritePreference() throws InterruptedException
   {
      writerPreemptReaderTest(new RangeReadWriteFactory_WritePreference());
   }

   @Test
   public void writerPreemptReader_Fair() throws InterruptedException
   {
      writerPreemptReaderTest(new RangeReadWriteFactory_Fair());
   }

   public void writerPreemptReaderTest(RangeReadWriteFactory factory) throws InterruptedException
   {
      final RangeReadWriteLock lock = factory.getInstance();
      CountDownLatch latch = new CountDownLatch(4);

      SimpleRunnerWithSleeps r1 = new SimpleRunnerWithSleeps(lock, latch, 0, 100);
      r1.setReadRangeLock(1, 10);
      r1.start();

      SimpleRunnerWithSleeps w2 = new SimpleRunnerWithSleeps(lock, latch, 50, 100);
      w2.setWriteRangeLock(1, 10);
      w2.start();

      SimpleRunnerWithSleeps r3 = new SimpleRunnerWithSleeps(lock, latch, 70, 40);
      r3.setReadRangeLock(1, 10);
      r3.start();

      SimpleRunnerWithSleeps r4 = new SimpleRunnerWithSleeps(lock, latch, 75, 100);
      r4.setReadRangeLock(1, 10);
      r4.start();

      latch.await();

      Thread.sleep(25);
      assertTrue(r1.isAlive());

      Thread.sleep(100);

      assertTrue(!r1.isAlive());
      assertTrue(w2.isAlive());
      assertTrue(r3.isAlive());
      assertTrue(r4.isAlive());

      Thread.sleep(100);
      assertTrue(!w2.isAlive());
      assertTrue(r3.isAlive());
      assertTrue(r4.isAlive());

      Thread.sleep(30);
      assertTrue(!r3.isAlive());
      assertTrue(r4.isAlive());

      Thread.sleep(100);
      assertTrue(!r4.isAlive());
   }

   @Test
   public void starvedReaderTest_writePreference() throws InterruptedException
   {
      int nrepeat = 10;
      CountDownLatch latch = new CountDownLatch(3);
      RangeReadWriteLock lock = new RangeReadWriteLockWritePreference();

      SimpleRunnerWithSleeps w1 = new SimpleRunnerWithSleeps(lock, latch, 0, 100, nrepeat, 50);
      w1.setWriteRangeLock(1, 10);
      w1.start();

      SimpleRunnerWithSleeps w2 = new SimpleRunnerWithSleeps(lock, latch, 50, 100, nrepeat, 50);
      w2.setWriteRangeLock(1, 10);
      w2.start();

      // Should be staved by writers
      SimpleRunnerWithSleeps r1 = new SimpleRunnerWithSleeps(lock, latch, 25, 100);
      r1.setReadRangeLock(1, 10);
      r1.start();

      latch.await();

      for (int i = 0; i < 2 * nrepeat; i++)
      {
         Thread.sleep(100);
         _logger.debug("Attempt #" + i);
         assertTrue(r1.isAlive());
      }
      // Extra time
      Thread.sleep(30 * nrepeat);
      assertTrue(!r1.isAlive());
   }

   @Test
   public void doesntStarveReaderTest_Fair() throws InterruptedException
   {
      int nrepeat = 3;
      CountDownLatch latch = new CountDownLatch(3);
      RangeReadWriteLock lock = new RangeReadWriteLockFair();

      SimpleRunnerWithSleeps w1 = new SimpleRunnerWithSleeps(lock, latch, 0, 100, nrepeat, 50);
      w1.setWriteRangeLock(1, 10);
      w1.start();

      SimpleRunnerWithSleeps w2 = new SimpleRunnerWithSleeps(lock, latch, 50, 100, nrepeat, 50);
      w2.setWriteRangeLock(1, 10);
      w2.start();

      // Should be staved by writers
      SimpleRunnerWithSleeps r3 = new SimpleRunnerWithSleeps(lock, latch, 25, 100);
      r3.setReadRangeLock(1, 10);
      r3.start();

      latch.await();

      // After 300 msec r3 should be gone
      Thread.sleep(300);
      assertTrue(!r3.isAlive());

      // Extra time
      Thread.sleep(200 * nrepeat);
      assertTrue(!w1.isAlive());
      assertTrue(!w2.isAlive());
   }
   
   @Test
   public void writeOverlap_fair() throws InterruptedException
   {
      CountDownLatch latch = new CountDownLatch(4);
      RangeReadWriteLock lock = new RangeReadWriteLockFair();

      SimpleRunnerWithSleeps w1 = new SimpleRunnerWithSleeps(lock, latch, 0, 200);
      w1.setWriteRangeLock(1, 10);
      w1.start();

      SimpleRunnerWithSleeps w2 = new SimpleRunnerWithSleeps(lock, latch, 0, 50);
      w2.setWriteRangeLock(20, 10);
      w2.start();

      SimpleRunnerWithSleeps w3 = new SimpleRunnerWithSleeps(lock, latch, 10, 100);
      w3.setWriteRangeLock(1, 10);
      w3.start();


      SimpleRunnerWithSleeps w4 = new SimpleRunnerWithSleeps(lock, latch, 20, 20);
      w4.setWriteRangeLock(20, 10);
      w4.start();

      latch.await();      
      
      Thread.sleep(100);
      assertTrue (!w2.isAlive());
      assertTrue (!w4.isAlive());
      
      Thread.sleep(100);
      assertTrue (w3.isAlive());

      Thread.sleep(10);
      assertTrue (!w1.isAlive());

      Thread.sleep(20);
      assertTrue (!w4.isAlive());
   }

   @Test
   public void intersectionTest()
   {
      Lock l1, l2;
      RangeReadWriteLock lock = new NullRangeReadWriteLock();

      l1 = lock.getReadRangeLock(0, 10);
      l2 = lock.getReadRangeLock(10, 10);
      assertTrue(!l1.intersect(l2));
      assertTrue(!l2.intersect(l1));
      assertTrue(l1.intersect(l1));

      l1 = lock.getReadRangeLock(0, 11);
      l2 = lock.getReadRangeLock(10, 1);
      assertTrue(l1.intersect(l2));
      assertTrue(l2.intersect(l1));

      l1 = lock.getReadRangeLock(5, 10);
      l2 = lock.getReadRangeLock(10, 10);
      assertTrue(l1.intersect(l2));
      assertTrue(l2.intersect(l1));
   }
}
