//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Wesley Leggette
//
// Date: May 3, 2007
//---------------------

package org.cleversafe.vault;
//import org.apache.log4j.Logger;

import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.Date;
import java.util.UUID;

import org.bouncycastle.asn1.x509.X509Name;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.x509.X509V3CertificateGenerator;


/**
 * Generates sample data for testing purposes.
 *
 */
public class ModelTestUtil
{ 
   private static final long VALID_PERIOD = 1000L * 60 * 60 * 24 * 30 * 12;
   private static final long OFFSET_PERIOD = 1000L *60 * 60 * 24 * 30;
   
   private static final UUID GRID_ACCOUNT_IDENTIFIER = 
      UUID.fromString( "ceedfc07-821d-4af2-95b3-71badb79a908" );
   
   private static final String ISSUER  = 
      "C=US,ST=Illinois,L=Chicago,O=Cleversafe,OU=Development,CN=Dev Root CA";
   
   private static final String BAD_ISSUER =
      "C=US,ST=Illinois,L=Chicago,O=Densesafe,OU=Quality Assurance,CN=Busy Leaf CA";
   
   private static final String RA_SUBJECT =
      "C=US,ST=Illinois,L=Chicago,O=Cleversafe,OU=Development," +
      "OU=Accounts,CN=ra.cleversafelabs.com";
   
   private static final String ACCOUNT_SUBJECT =
      "C=US,ST=Illinois,L=Chicago,O=Cleversafe,OU=Development," +
      "OU=Accounts,UID=ceedfc07-821d-4af2-95b3-71badb79a908";
   
   private static final String ACCOUNT_SUBJECT_PREFIX =
      "C=US,ST=Illinois,L=Chicago,O=Cleversafe,OU=Development," +
      "OU=Accounts,UID=";
   
   private static final String SIGNATURE_ALGORITHM = "SHA1WithRSAEncryption";
   
   private static final RSAPublicKeySpec CA_PUBLIC_KEY_SPEC = new RSAPublicKeySpec(
         new BigInteger("b4a7e46170574f16a97082b22be58b6a2a629798419be12872a4bdba626cfae9900f76abfb12139dce5de56564fab2b6543165a040c606887420e33d91ed7ed7", 16),
         new BigInteger("11", 16));

   private static final RSAPrivateCrtKeySpec CA_PRIVATE_KEY_SPEC = new RSAPrivateCrtKeySpec(
         new BigInteger("b4a7e46170574f16a97082b22be58b6a2a629798419be12872a4bdba626cfae9900f76abfb12139dce5de56564fab2b6543165a040c606887420e33d91ed7ed7", 16),
         new BigInteger("11", 16),
         new BigInteger("9f66f6b05410cd503b2709e88115d55daced94d1a34d4e32bf824d0dde6028ae79c5f07b580f5dce240d7111f7ddb130a7945cd7d957d1920994da389f490c89", 16),
         new BigInteger("c0a0758cdf14256f78d4708c86becdead1b50ad4ad6c5c703e2168fbf37884cb", 16),
         new BigInteger("f01734d7960ea60070f1b06f2bb81bfac48ff192ae18451d5e56c734a5aab8a5", 16),
         new BigInteger("b54bb9edff22051d9ee60f9351a48591b6500a319429c069a3e335a1d6171391", 16),
         new BigInteger("d3d83daf2a0cecd3367ae6f8ae1aeb82e9ac2f816c6fc483533d8297dd7884cd", 16),
         new BigInteger("b8f52fc6f38593dabb661d3f50f8897f8106eee68b1bce78a95b132b4e5b5d19", 16));
   
   private static final RSAPublicKeySpec RA_PUBLIC_KEY_SPEC = new RSAPublicKeySpec(
         new BigInteger("b259d2d6e627a768c94be36164c2d9fc79d97aab9253140e5bf17751197731d6f7540d2509e7b9ffee0a70a6e26d56e92d2edd7f85aba85600b69089f35f6bdbf3c298e05842535d9f064e6b0391cb7d306e0a2d20c4dfb4e7b49a9640bdea26c10ad69c3f05007ce2513cee44cfe01998e62b6c3637d3fc0391079b26ee36d5", 16),
         new BigInteger("11", 16));
   
   private static final RSAPrivateCrtKeySpec RA_PRIVATE_KEY_SPEC = new RSAPrivateCrtKeySpec(
         new BigInteger("b259d2d6e627a768c94be36164c2d9fc79d97aab9253140e5bf17751197731d6f7540d2509e7b9ffee0a70a6e26d56e92d2edd7f85aba85600b69089f35f6bdbf3c298e05842535d9f064e6b0391cb7d306e0a2d20c4dfb4e7b49a9640bdea26c10ad69c3f05007ce2513cee44cfe01998e62b6c3637d3fc0391079b26ee36d5", 16),
         new BigInteger("11", 16),
         new BigInteger("92e08f83cc9920746989ca5034dcb384a094fb9c5a6288fcc4304424ab8f56388f72652d8fafc65a4b9020896f2cde297080f2a540e7b7ce5af0b3446e1258d1dd7f245cf54124b4c6e17da21b90a0ebd22605e6f45c9f136d7a13eaac1c0f7487de8bd6d924972408ebb58af71e76fd7b012a8d0e165f3ae2e5077a8648e619", 16),
         new BigInteger("f75e80839b9b9379f1cf1128f321639757dba514642c206bbbd99f9a4846208b3e93fbbe5e0527cc59b1d4b929d9555853004c7c8b30ee6a213c3d1bb7415d03", 16),
         new BigInteger("b892d9ebdbfc37e397256dd8a5d3123534d1f03726284743ddc6be3a709edb696fc40c7d902ed804c6eee730eee3d5b20bf6bd8d87a296813c87d3b3cc9d7947", 16),
         new BigInteger("1d1a2d3ca8e52068b3094d501c9a842fec37f54db16e9a67070a8b3f53cc03d4257ad252a1a640eadd603724d7bf3737914b544ae332eedf4f34436cac25ceb5", 16),
         new BigInteger("6c929e4e81672fef49d9c825163fec97c4b7ba7acb26c0824638ac22605d7201c94625770984f78a56e6e25904fe7db407099cad9b14588841b94f5ab498dded", 16),
         new BigInteger("dae7651ee69ad1d081ec5e7188ae126f6004ff39556bde90e0b870962fa7b926d070686d8244fe5a9aa709a95686a104614834b0ada4b10f53197a5cb4c97339", 16));

   private static final RSAPublicKeySpec ACCOUNT_PUBLIC_KEY_SPEC = new RSAPublicKeySpec(
         new BigInteger("a9b299fc5ad5493f35f5d7848f08504ff8333b900ebda8a6139185e3be134bd3aae20ef2285bb2d1c2e69c13495a9c66228f5e7d4521d3e05d0499bc158f1162ed9e78d9de51d9575fe38b23efb4d442d6e3825ed6e5211cc6184b117b2acda8cdf7f43074f44400e15cca231fbfc3839f86390c1ee710130d7ad4dec955c663", 16),
         new BigInteger("65537", 10));
   
   private static final RSAPrivateCrtKeySpec ACCOUNT_PRIVATE_KEY_SPEC = new RSAPrivateCrtKeySpec(
         new BigInteger("a9b299fc5ad5493f35f5d7848f08504ff8333b900ebda8a6139185e3be134bd3aae20ef2285bb2d1c2e69c13495a9c66228f5e7d4521d3e05d0499bc158f1162ed9e78d9de51d9575fe38b23efb4d442d6e3825ed6e5211cc6184b117b2acda8cdf7f43074f44400e15cca231fbfc3839f86390c1ee710130d7ad4dec955c663", 16),
         new BigInteger("65537", 10),
         new BigInteger("53010140e64d1d1ec60e11f4cce18c2711883dc220f3b3e248000e356f6b90d35fea62d51fd87a870f862d2161e7e8482cef639072983142b53ce2d594d5fdf2517fada9997038d01da098917581c971095531b08056bbc20a7c7a477ccb3ca654a5396cd23dd71e966d1a3479ec9321e1b8abb388848cb35fd3a30a1c462f81", 16), 
         new BigInteger("dc70af560c362237f6494b5d56f0b70edfedf33fdd8b5ff5218d1851270dd2cfe41033b5bd110fd6229094c88e0de321783c6d01a2e9117d0cf759794af90329", 16),
         new BigInteger("c5127320bcc14ba8f9eb97dc763d92ed2787438f45f56ad5f19ad01384b5da50d24d5979aab48df55ab82479859c8fb47772940a9413cd74df2bff062d8a9aab", 16),
         new BigInteger("ba69fc8c3ac56888849894c7f92694d75a5cc210e70bdef8d0079e61ba32016d5682b107e63bd075984345de97ea7c81a8490931920fa07f6c809bc46bb74a89", 16),
         new BigInteger("78f2efd1a1c7084676cc4ef1b167eceabc1dda2aee86c6c0f37d4e430c6a07c91ea8f64723ab988c82d70d1bc012ec6797ffac3829d5557d6b12b77bf52b61ed", 16),
         new BigInteger("25543cef2ff6730eb8dbd827b682fa95bea34f7bb967f9ca2723d12f8f254b12bca20cf188d5e83698d2f2474a3521e5853680b1936c744e49ba16eae2259ad0", 16));


   ////////////////////////////////////////////////////////////////////////////

   private static int serial; 
   
   static
   {
      Security.addProvider( new BouncyCastleProvider() );
      ModelTestUtil.serial = 1;
   }
   
   protected static KeyPair generateKeyPair(
         RSAPublicKeySpec pubspec,
         RSAPrivateCrtKeySpec privspec )
   {
      try
      {
         KeyFactory fact = KeyFactory.getInstance("RSA", "BC");
         PrivateKey priv = fact.generatePrivate( privspec );
         PublicKey pub = fact.generatePublic( pubspec );
         return new KeyPair( pub, priv );
      }
      catch (NoSuchProviderException e)
      {
         throw new RuntimeException("Bouncy Castle JCE Provider not properly registered");
      }
      catch (InvalidKeySpecException e)
      {
         throw new RuntimeException("CA key specifications not correct: " + e);
      }
      catch (NoSuchAlgorithmException e)
      {
         throw new RuntimeException("RSA algorithm not available: " + e);
      }
   }
   
   protected static KeyPair getFixedCaKeyPair()
   {
      return generateKeyPair( CA_PUBLIC_KEY_SPEC, CA_PRIVATE_KEY_SPEC );
   }
   
   protected static KeyPair getFixedRaKeyPair()
   {
      return generateKeyPair( RA_PUBLIC_KEY_SPEC, RA_PRIVATE_KEY_SPEC );
   }
   
   
   protected static X509Certificate signCertificateWithIssuer(
         X509V3CertificateGenerator certgen,
         String issuer,
         PrivateKey issuerKey )
   {
      certgen.setSerialNumber( BigInteger.valueOf(ModelTestUtil.serial) );
      certgen.setIssuerDN( new X509Name(issuer) );
      certgen.setSignatureAlgorithm( SIGNATURE_ALGORITHM );
      
      try
      {
         X509Certificate cert = certgen.generate( issuerKey, "BC" );
         ModelTestUtil.serial++;
         return cert;
      }
      catch (Exception e)
      {
         // Throwing a runtime exception because under normal circumstances this code
         // should work; inputs are fixed and code should be unit tested.
         throw new RuntimeException("Could not generate fixed certificate: " + e);
      }
   }
   
   protected static X509Certificate signCertificate( X509V3CertificateGenerator certgen )
   {
      return signCertificateWithIssuer( certgen, ISSUER, getFixedCaKeyPair().getPrivate() );
   }
   
   
   
   
   public static KeyPair generateFixedKeyPair()
   {
      return generateKeyPair( ACCOUNT_PUBLIC_KEY_SPEC, ACCOUNT_PRIVATE_KEY_SPEC );
   }
   
   public static KeyPair generateRandomKeyPair()
   {
      try
      {
         return KeyPairGenerator.getInstance("RSA").generateKeyPair();
      }
      catch (NoSuchAlgorithmException e)
      {
         throw new RuntimeException("RSA algorithm not available: " + e);
      }
   }
   
   public static X509Certificate generateFixedValidCertificate()
   {
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( new X509Name(ACCOUNT_SUBJECT) );
      certgen.setPublicKey( ModelTestUtil.generateFixedKeyPair().getPublic() );
      certgen.setNotBefore(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD) );
      certgen.setNotAfter(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD + VALID_PERIOD) );
      
      // TODO: Add specified extensions
      
      return ModelTestUtil.signCertificate( certgen );
   }
   
   public static X509Certificate generateFixedExpiredCertificate()
   {
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( new X509Name(ACCOUNT_SUBJECT) );
      certgen.setPublicKey( ModelTestUtil.generateFixedKeyPair().getPublic() );
      certgen.setNotBefore(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD - VALID_PERIOD) );
      certgen.setNotAfter(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD) );
      
      // TODO: Add specified extensions
      
      return ModelTestUtil.signCertificate( certgen );
   }
   
   public static X509Certificate generateFixedBadIssuerCertificate()
   {  
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( new X509Name(ACCOUNT_SUBJECT) );
      certgen.setPublicKey( ModelTestUtil.generateFixedKeyPair().getPublic() );
      certgen.setNotBefore(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD) );
      certgen.setNotAfter(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD + VALID_PERIOD) );
      
      // TODO: Add specified extensions
      
      return ModelTestUtil.signCertificateWithIssuer(
            certgen,
            BAD_ISSUER,
            ModelTestUtil.generateFixedKeyPair().getPrivate() );
   }
   
   public static X509Certificate generateFixedCaCertificate()
   {
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( new X509Name(ISSUER) );
      certgen.setPublicKey( ModelTestUtil.getFixedCaKeyPair().getPublic() );
      certgen.setNotBefore(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD) );
      certgen.setNotAfter(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD + VALID_PERIOD) );
      
      // TODO: Add specified extensions
      
      return ModelTestUtil.signCertificate( certgen );
   }
   
   public static X509Certificate generateFixedRaCertificate()
   {
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( new X509Name(RA_SUBJECT) );
      certgen.setPublicKey( ModelTestUtil.getFixedRaKeyPair().getPublic() );
      certgen.setNotBefore(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD) );
      certgen.setNotAfter(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD + VALID_PERIOD) );
      
      // TODO: Add specified extensions
      
      return ModelTestUtil.signCertificate( certgen );
   }
   
   public static X509Certificate generateRandomValidCertificate( UUID gridAccountIdentifier )
   {
      return generateRandomValidCertificate(
            gridAccountIdentifier,
            ModelTestUtil.generateRandomKeyPair() );
   }
   
   public static X509Certificate generateRandomValidCertificate( 
         UUID gridAccountIdentifier,
         KeyPair keyPair )
   {
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( 
            new X509Name(ACCOUNT_SUBJECT_PREFIX + gridAccountIdentifier.toString()) );
      certgen.setPublicKey( keyPair.getPublic() );
      certgen.setNotBefore(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD) );
      certgen.setNotAfter(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD + VALID_PERIOD) );
      
      // TODO: Add specified extensions
      
      return ModelTestUtil.signCertificate( certgen );
   }
         
   
   
   public static X509Certificate generateRandomExpiredCertificate( UUID gridAccountIdentifier )
   {
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( 
            new X509Name(ACCOUNT_SUBJECT_PREFIX + gridAccountIdentifier.toString()) );
      certgen.setPublicKey( ModelTestUtil.generateRandomKeyPair().getPublic() );
      certgen.setNotBefore(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD - VALID_PERIOD) );
      certgen.setNotAfter(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD) );
      
      // TODO: Add specified extensions
      
      return ModelTestUtil.signCertificate( certgen );
   }
   
   public static X509Certificate generateRandomBadIssuerCertificate( UUID gridAccountIdentifier )
   {
      X509V3CertificateGenerator certgen = new X509V3CertificateGenerator();
      certgen.reset();
      certgen.setSubjectDN( 
            new X509Name(ACCOUNT_SUBJECT_PREFIX + gridAccountIdentifier.toString()) );
      certgen.setPublicKey( ModelTestUtil.generateRandomKeyPair().getPublic() );
      certgen.setNotBefore(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD) );
      certgen.setNotAfter(
            new Date(System.currentTimeMillis() - OFFSET_PERIOD + VALID_PERIOD) );
      
      // TODO: Add specified extensions
      
      return ModelTestUtil.signCertificateWithIssuer(
            certgen,
            BAD_ISSUER,
            ModelTestUtil.generateFixedKeyPair().getPrivate() );
   }
   
   
   public static UUID generateFixedUUID()
   {
      return ModelTestUtil.GRID_ACCOUNT_IDENTIFIER;
   }
   
  
   
   
   
   
   
   


}


