//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: Zach
//
// Date: Sep 6, 2007
//---------------------

package org.cleversafe.vault;

import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import org.cleversafe.config.evaluator.SupportedTypes;
import org.cleversafe.test.BaseTest;
import org.cleversafe.util.Version;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.junit.Test;

public class XMLVaultDescriptorTest
{
   private static final File BASE_FILE_PATH =
         new File(System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
               + "/org/cleversafe/vault/descriptor-generated-orig.xml");
   private static final File GENERATED_FILE_PATH =
         new File(System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
               + "/org/cleversafe/vault/descriptor-generated.xml");

   /**
    * Helper method which generates a vault descriptor which becomes the "gold standard" to use in
    * tests.
    */
   private XMLVaultDescriptor createTestDescriptor() throws VaultDescriptorException
   {
      XMLVaultDescriptor desc = new XMLVaultDescriptor();

      desc.setVaultType("block");
      desc.addLiteralVaultParam("blog", SupportedTypes.INT, "166");

      desc.setIDA("IDA", "cauchy", Version.fromString("1.0"));
      desc.addLiteralIDAParam("num-slices", SupportedTypes.INT, "16");
      desc.addLiteralIDAParam("threshold", SupportedTypes.INT, "12");

      desc.addSliceStore("SliceStore", "block-multi-file", null);

      return desc;
   }

   /**
    * Helper method which does a binary diff on two InputStreams. Makes the assumption that the
    * streams have an end.
    * 
    * @param left
    *           First stream
    * @param right
    *           Second stream
    * @return True if streams are identical, false otherwise.
    * @throws IOException
    *            If a read error occurs.
    */
   private boolean diff(InputStream left, InputStream right) throws IOException
   {
      while (true)
      {
         int first = left.read();
         int second = right.read();
         if ((first == -1) && (second == -1))
         {
            return true;
         }
         else if (first != second)
         {
            return false;
         }
      }
   }

   // This code is purposely left commented in case a new vault descriptor is required to be
   // generated to be used as a base for this test. If this becomes the case, uncomment the
   // code and run the test.
   // --zmark 2007-09-10
   //
   // @Test
   // public void generateBaseDescriptorOnce() throws VaultDescriptorException, VaultIOException
   // {
   // this.writeBaseDescriptor();
   // }

   /**
    * This test generates a vault descriptor, saves it, and does a binary diff against the original.
    * If the generated descriptor is not byte-identical to the original, the test will fail.
    * 
    */
   @Test
   public void testSave() throws VaultDescriptorException, VaultIOException, FileNotFoundException,
         IOException
   {
      XMLVaultDescriptor generatedDesc = createTestDescriptor();
      generatedDesc.writeDescriptor(GENERATED_FILE_PATH);

      InputStream baseFile = new FileInputStream(BASE_FILE_PATH);
      InputStream generatedFile = new FileInputStream(GENERATED_FILE_PATH);

      boolean identical = diff(baseFile, generatedFile);

      baseFile.close();
      generatedFile.close();

      if (!identical)
      {
         fail("Generated vault descriptor not identical to base vault descriptor");
      }
   }

   /**
    * This test attempts to generate and save an invalid vault descriptor. A descriptor is defined
    * to require at least one IDA and one slice store.
    */
   @Test
   public void testSaveInvalidDescriptor()
   {
      File loc = GENERATED_FILE_PATH;

      XMLVaultDescriptor desc = new XMLVaultDescriptor();

      try
      {
         desc.writeDescriptor(loc);
         fail("Vault descriptor unexpectedly written, expected XmlException");
      }
      catch (VaultIOException e)
      {
         fail("VaultIOException thrown, expected XmlException");
      }
      catch (VaultDescriptorException e)
      {
      }
   }

   @Test
   public void testLoadVaultDescriptor()
   {
      File loc = BASE_FILE_PATH;

      XMLVaultDescriptor desc = null;
      try
      {
         desc = new XMLVaultDescriptor(loc.getAbsolutePath());

         System.out.println(desc.getDatasourceCodecs().toString());
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }
}
