//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 18, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.security.KeyPair;
import java.security.KeyPairGenerator;

import org.apache.log4j.Logger;
import org.bouncycastle.asn1.DERObject;
import org.cleversafe.vault.storage.asn1.EncryptedKeyInfo;
import org.cleversafe.vault.storage.asn1.KeyUsage;
import org.cleversafe.vault.storage.asn1.PlainKeyInfo;
import org.cleversafe.vault.storage.asn1.KeyInfo.KeyInfoPair;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;


// TODO: Describe class or interface
public class EncryptedKeyInfoTest
{
   private static Logger _logger = Logger.getLogger(EncryptedKeyInfoTest.class);

   private static KeyPair encryptionKeyPair;

   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize(512);
      EncryptedKeyInfoTest.encryptionKeyPair = keygen.generateKeyPair();
   }

   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
   }

   @Before
   public void setUp() throws Exception
   {
   }

   @After
   public void tearDown() throws Exception
   {
   }

   private static String datasourceEncryptionAlgorithm = "AES";
   private static String wrapTransformation = "RSA/ECB/PKCS1Padding";

   private PlainKeyInfo getPlainKeyInfo() throws Exception
   {
      PlainKeyInfo plainKeyInfo = PlainKeyInfo.generateKey(
            datasourceEncryptionAlgorithm, 128, KeyUsage.DATASOURCE_ENCRYPTION);
      return plainKeyInfo;
   }

   private EncryptedKeyInfo getEncryptedKeyInfo(PlainKeyInfo pinfo)
         throws Exception
   {
      EncryptedKeyInfo encryptedKeyInfo = EncryptedKeyInfo.wrap(pinfo,
            encryptionKeyPair.getPublic(), wrapTransformation);
      return encryptedKeyInfo;
   }

   @Test
   public void testGetTransformationGetAlgorithm()
   {
      try
      {
         PlainKeyInfo pinfo = getPlainKeyInfo();
         EncryptedKeyInfo einfo = getEncryptedKeyInfo(pinfo);
         assertEquals(einfo.getTransformation(), wrapTransformation);
         assertEquals(einfo.getAlgorithm(), datasourceEncryptionAlgorithm);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         _logger.error(e);
         fail(e.getMessage());
      }
   }

   @Test
   public void testSetKey()
   {
      try
      {
         PlainKeyInfo pinfo = getPlainKeyInfo();
         EncryptedKeyInfo einfo = getEncryptedKeyInfo(pinfo);
         einfo.setKey(pinfo.getKey());
         fail("setKey() on an EncryptedKeyInfo should have thrown a SecurityException");
      }
      catch (SecurityException e)
      {
         // Expected.
      }
      catch (Exception e)
      {
         e.printStackTrace();
         _logger.error(e);
         fail(e.getMessage());
      }
   }

   @Test
   public void testGetKey()
   {
      try
      {
         EncryptedKeyInfo einfo = getEncryptedKeyInfo(getPlainKeyInfo());
         einfo.getKey();
         fail("getKey() on an EncryptedKeyInfo should have thrown a SecurityException");
      }
      catch (SecurityException e)
      {
         // Expected.
      }
      catch (Exception e)
      {
         e.printStackTrace();
         _logger.error(e);
         fail(e.getMessage());
      }
   }

   @Test
   public void testGetInstanceASN1Object()
   {
      try
      {
         PlainKeyInfo pinfo = getPlainKeyInfo();
         EncryptedKeyInfo einfo = getEncryptedKeyInfo(pinfo);

         DERObject derObject = einfo.toASN1Object();
         EncryptedKeyInfo returned = EncryptedKeyInfo.getInstance(derObject);

         assertEquals(pinfo.getAlgorithm(), returned.getAlgorithm());
         assertEquals(pinfo.getUsage(), returned.getUsage());
         assertEquals(pinfo.getType(), returned.getType());
      }
      catch (Exception e)
      {
         e.printStackTrace();
         _logger.error(e);
         fail(e.getMessage());
      }
   }

   @Test
   public void testWrapUnwrap()
   {
      try
      {
         PlainKeyInfo pinfo = getPlainKeyInfo();
         EncryptedKeyInfo einfo = getEncryptedKeyInfo(pinfo);

         PlainKeyInfo unwrapped = EncryptedKeyInfo.unwrap(einfo,
               encryptionKeyPair.getPrivate());

         assertEquals(pinfo.getAlgorithm(), unwrapped.getAlgorithm());
         assertEquals(pinfo.getKey(), unwrapped.getKey());
         assertEquals(pinfo.getUsage(), unwrapped.getUsage());
         assertEquals(pinfo.getType(), unwrapped.getType());
      }
      catch (Exception e)
      {
         e.printStackTrace();
         _logger.error(e);
         fail(e.getMessage());
      }
   }
   
   @Test
   public void testWrapUnwrapAsymetricKey() throws Exception
   {
      KeyInfoPair pinfopair = PlainKeyInfo.generateKeyPair( 
            "RSA", 512, KeyUsage.SLICE_VERIFICATION );
      PlainKeyInfo pinfo = (PlainKeyInfo)pinfopair.getPrivate();
      EncryptedKeyInfo einfo = getEncryptedKeyInfo(pinfo);
      
      PlainKeyInfo unwrapped = EncryptedKeyInfo.unwrap(
            einfo, encryptionKeyPair.getPrivate() );
      
      assertEquals(pinfo.getAlgorithm(), unwrapped.getAlgorithm());
      assertEquals(pinfo.getKey(), unwrapped.getKey());
      assertEquals(pinfo.getUsage(), unwrapped.getUsage());
      assertEquals(pinfo.getType(), unwrapped.getType());
   }
}
