//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 18, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.util.Arrays;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.util.ASN1Dump;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.cleversafe.test.BaseTest;
import org.cleversafe.vault.storage.asn1.KeyInfo.KeyInfoPair;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public class PlainKeyInfoTest
{
   private static File _outputdir =
         new File(System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".")
               + "/org/cleversafe/vault/storage/asn1");

   static
   {
      Security.addProvider(new BouncyCastleProvider());
   }

   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      _outputdir.mkdirs();
   }

   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
   }

   @Before
   public void setUp() throws Exception
   {
   }

   @After
   public void tearDown() throws Exception
   {
   }

   @Test
   public void testGenerateKey() throws Exception
   {
      try
      {
         PlainKeyInfo.generateKey("RSA", 1024, KeyUsage.DATASOURCE_ENCRYPTION);
         fail("Successfully created symmetric key with asymmetric algorithm; shouldn't have");
      }
      catch (NoSuchAlgorithmException e)
      {
      }

      PlainKeyInfo info = PlainKeyInfo.generateKey("DES", 56, KeyUsage.DATASOURCE_ENCRYPTION);
      assertEquals("DES", info.getAlgorithm());
      assertEquals("DES", info.getKey().getAlgorithm());

      assertEquals(KeyUsage.DATASOURCE_ENCRYPTION, info.getUsage());

      assertTrue(info.getKey() instanceof SecretKey);
   }

   @Test
   public void testGenerateKeyPair() throws Exception
   {
      try
      {
         PlainKeyInfo.generateKeyPair("DES", 56, KeyUsage.DATASOURCE_ENCRYPTION);
         fail("Successfully created asymmetric key with symmetric algorithm; shouldn't have");
      }
      catch (NoSuchAlgorithmException e)
      {
      }

      KeyInfoPair pair = PlainKeyInfo.generateKeyPair("RSA", 1024, KeyUsage.DATASOURCE_ENCRYPTION);
      assertEquals("RSA", pair.getPublic().getAlgorithm());
      assertEquals("RSA", pair.getPublic().getKey().getAlgorithm());
      assertEquals("RSA", pair.getPrivate().getAlgorithm());
      assertEquals("RSA", pair.getPrivate().getKey().getAlgorithm());

      assertEquals(KeyUsage.DATASOURCE_ENCRYPTION, pair.getPublic().getUsage());
      assertEquals(KeyUsage.DATASOURCE_ENCRYPTION, pair.getPrivate().getUsage());

      assertTrue(pair.getPublic().getKey() instanceof PublicKey);
      assertTrue(pair.getPrivate().getKey() instanceof PrivateKey);

      assertEquals(Cipher.PUBLIC_KEY, pair.getPublic().getType());
      assertEquals(Cipher.PRIVATE_KEY, pair.getPrivate().getType());

   }

   @Test
   public void testToASN1ObjectSecretKey() throws Exception
   {
      PlainKeyInfo info = PlainKeyInfo.generateKey("AES", 256, KeyUsage.DATASOURCE_ENCRYPTION);

      ASN1Encodable asnobj = info.toASN1Object();

      FileOutputStream out =
            new FileOutputStream(System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".")
                  + "/org/cleversafe/vault/storage/asn1/KeyInfoTest01.der");
      out.write(asnobj.getDEREncoded());
      System.out.println(ASN1Dump.dumpAsString(asnobj));

      ASN1InputStream in =
            new ASN1InputStream(new FileInputStream(System.getProperty(
                  BaseTest.TEST_OUTPUT_PROPERTY, ".")
                  + "/org/cleversafe/vault/storage/asn1/KeyInfoTest01.der"));

      PlainKeyInfo dest = PlainKeyInfo.getInstance(in.readObject());

      assertEquals(info.getAlgorithm(), dest.getAlgorithm());
      assertEquals(info.getType(), dest.getType());
      assertEquals(info.getUsage(), dest.getUsage());
      assertTrue(Arrays.equals(info.getKey().getEncoded(), dest.getKey().getEncoded()));
      assertTrue(dest.getKey() instanceof SecretKey);
   }

   @Test
   public void testASN1ObjectPrivateKey() throws Exception
   {
      KeyInfo info =
            PlainKeyInfo.generateKeyPair("RSA", 512, KeyUsage.DATASOURCE_ENCRYPTION).getPrivate();

      ASN1Encodable asnobj = info.toASN1Object();

      FileOutputStream out =
            new FileOutputStream(System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".")
                  + "/org/cleversafe/vault/storage/asn1/KeyInfoTest02.der");
      out.write(asnobj.getDEREncoded());
      System.out.println(ASN1Dump.dumpAsString(asnobj));

      ASN1InputStream in =
            new ASN1InputStream(new FileInputStream(System.getProperty(
                  BaseTest.TEST_OUTPUT_PROPERTY, ".")
                  + "/org/cleversafe/vault/storage/asn1/KeyInfoTest02.der"));

      PlainKeyInfo dest = PlainKeyInfo.getInstance(in.readObject());

      assertEquals(info.getAlgorithm(), dest.getAlgorithm());
      assertEquals(info.getType(), dest.getType());
      assertEquals(info.getUsage(), dest.getUsage());
      assertTrue(Arrays.equals(info.getKey().getEncoded(), dest.getKey().getEncoded()));
      assertTrue(dest.getKey() instanceof PrivateKey);
   }

   @Test
   public void testASN1ObjectPublicKey() throws Exception
   {
      KeyInfo info =
            PlainKeyInfo.generateKeyPair("RSA", 512, KeyUsage.DATASOURCE_ENCRYPTION).getPublic();

      ASN1Encodable asnobj = info.toASN1Object();

      FileOutputStream out =
            new FileOutputStream(System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".")
                  + "/org/cleversafe/vault/storage/asn1/KeyInfoTest03.der");
      out.write(asnobj.getDEREncoded());
      System.out.println(ASN1Dump.dumpAsString(asnobj));

      ASN1InputStream in =
            new ASN1InputStream(new FileInputStream(System.getProperty(
                  BaseTest.TEST_OUTPUT_PROPERTY, ".")
                  + "/org/cleversafe/vault/storage/asn1/KeyInfoTest03.der"));

      PlainKeyInfo dest = PlainKeyInfo.getInstance(in.readObject());

      assertEquals(info.getAlgorithm(), dest.getAlgorithm());
      assertEquals(info.getType(), dest.getType());
      assertEquals(info.getUsage(), dest.getUsage());
      assertTrue(Arrays.equals(info.getKey().getEncoded(), dest.getKey().getEncoded()));
      assertTrue(dest.getKey() instanceof PublicKey);
   }

}
