//Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
//Cleversafe Dispersed Storage(TM) is software for secure, private and
//reliable storage of the world's data using information dispersal.
//
//Copyright (C) 2005-2007 Cleversafe, Inc.
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
//USA.
//
//Contact Information: 
// Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email: licensing@cleversafe.org
//
//END-OF-HEADER
//-----------------------
//@author: John Quigley <jquigley@cleversafe.com>
//@date: January 1, 2008
//---------------------

package org.jscsi.scsi.tasks.management;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Logger;
import org.jscsi.scsi.tasks.Status;
import org.jscsi.scsi.protocol.Command;
import org.jscsi.scsi.protocol.sense.exceptions.OverlappedCommandsAttemptedException;
import org.jscsi.scsi.tasks.Task;
import org.jscsi.scsi.tasks.TaskAttribute;
import org.jscsi.scsi.tasks.Task.TaskStatus;
import org.jscsi.scsi.transport.Nexus;
import org.jscsi.scsi.transport.TargetTransportPort;

/**
 * A SAM-2 task set implementation providing a single task set for all I_T nexuses.
 * <p>
 * Because this implementation tracks outstanding tasks solely by task tag it will not provide
 * reliable service for multiple initiator connections. Further, the ABORT TASK SET task management
 * function has the same results as the CLEAR TASK SET function.
 */
public class DefaultTaskSet implements TaskSet
{
   private static Logger _logger = Logger.getLogger(DefaultTaskSet.class);

   /*
    * The "tasks" map contains a map of task tags to currently live tasks. The map contains all
    * enabled and dormant tasks.
    * 
    * All tasks are wrapped in a "task container" which takes care of notifying this task set when
    * the task is completed. This notification (implemented using the finished() method) is
    * synchronous. The container object also provides a poll() method which the task set uses to
    * determine if the task is ready to be enabled. The task set will not return a task by its
    * poll() or take() until the task is unblocked.
    * 
    * The "queue" contains all dormant tasks. Tasks are removed from the queue before being returned
    * by poll() or take(). Tasks are removed from the task map once they are finished executing.
    * 
    * Note that tagged and untagged tasks can be differentiated by an invalid task tag (-1) on the
    * nexus, indicating an I_T_L Nexus instead of an I_T_L_Q nexus. Untagged tasks are always
    * treated as SIMPLE tasks.
    */

   // Treated as a decrementing counter
   private int capacity;

   private final Lock lock = new ReentrantLock();
   private final Condition notEmpty = lock.newCondition();
   private final Condition notFull = lock.newCondition();
   private final Condition unblocked = lock.newCondition();

   // Task set members
   private final Map<Long, TaskProxy> tasks; // Tag-to-Task map with 'null' key as the untagged task
   private final List<TaskProxy> enabled; // Enabled tasks
   private final List<TaskProxy> dormant; // Dormant task queue
   
   private static final Long UNTAGGED_TASK_KEY = new Long(-1);
   
   private static final long TASK_COMPLETION_WAIT_NS = 5000;

   /**
    * Used to encapsulate tasks. Notifies the task set when task execution is complete.
    * <p>
    * The task container is also used to determine when a task is ready to be enabled. The
    * constructor will examine the task set's tasks map and remember which tasks must be cleared
    * before this task can execute.
    */
   private class TaskProxy implements Task
   {
      private Task task;

      /**
       * Creates a task container for the given task. Uses the current enabled list and task queue
       * to determine which tasks will block this task.
       * 
       * @param task
       *           The task this container will encapsulate.
       */
      public TaskProxy(Task task) throws InterruptedException
      {
         this.task = task;
      }

      public Command getCommand()
      {
         return this.task.getCommand();
      }

      public TargetTransportPort getTargetTransportPort()
      {
         return this.task.getTargetTransportPort();
      }
      
      
      public TaskStatus waitForFinish()
      {
         return this.task.waitForFinish();
      }
      
      public TaskStatus getStatus()
      {
    	  return this.task.getStatus();
      }

      public void run()
      {
         if (_logger.isDebugEnabled())
            _logger.debug("Command now being run: " + this.task.getCommand());

         try
         {
            this.task.run();
         }
         catch (RuntimeException ex)
         {
            _logger.error("SCSI task execution threw an invalid exception", ex);
         }

         if (_logger.isTraceEnabled())
            _logger.trace("Task finished: " + this.task);

         long taskTag = this.task.getCommand().getNexus().getTaskTag();
         if (this.task.waitForFinish() == TaskStatus.COMPLETED)
         {
            isFinished(taskTag > -1 ? taskTag : UNTAGGED_TASK_KEY); // untagged tasks have a Q value of -1 (invalid)
         }
         

         if (_logger.isTraceEnabled())
            _logger.trace("Marked task as finished in task set: " + this.task);
      }

      public boolean abort()
      {
         boolean success = this.task.abort();
         if (success)
         {
            long taskTag = this.task.getCommand().getNexus().getTaskTag();
            isFinished(taskTag > -1 ? taskTag : UNTAGGED_TASK_KEY);   
         }
         return success;
      }

      @Override
      public String toString()
      {
         return "TaskContainer(" + this.task.toString() + ")";
      }

   }

   public static String toCommandReferenceNumberList(Collection<TaskProxy> tasks)
   {
      List<String> references = new LinkedList<String>();
      for (Task task : tasks)
      {
         references.add(Long.toHexString(task.getCommand().getCommandReferenceNumber()));
      }
      return references.toString();
   }

   public static String toNexusList(Collection<TaskProxy> tasks)
   {
      List<String> references = new LinkedList<String>();
      for (Task task : tasks)
      {
         references.add(task.getCommand().getNexus().toString());
      }
      return references.toString();
   }

   /////////////////////////////////////////////////////////////////////////////

   /**
    * Constructs a task set capable of enqueuing the indicated number of tasks.
    */
   public DefaultTaskSet(int capacity)
   {
      this.capacity = capacity;
      this.tasks = new HashMap<Long, TaskProxy>(capacity);
      this.enabled = new LinkedList<TaskProxy>();
      this.dormant = new LinkedList<TaskProxy>();
   }

   /////////////////////////////////////////////////////////////////////////////////////////////////


   /*
    * @see TaskSet#poll(Task, long, TimeUnit)
    */
   public boolean addTask(Task task, long timeout, TimeUnit unit) throws InterruptedException
   {
      if (task == null)
         throw new NullPointerException("task set does not take null objects");

      lock.lockInterruptibly();

      if (_logger.isTraceEnabled())
      {
         _logger.trace("--------------------------------------------------------------");
         _logger.trace("Printing task set BEFORE offer() execution");
         _logger.trace("Offering new command number: "
               + Long.toHexString(task.getCommand().getCommandReferenceNumber()));
         _logger.trace(this.toString());
         _logger.trace("offering to taskset command: " + task.getCommand());
         _logger.trace("--------------------------------------------------------------");
      }

      try
      {
         // check capacity
         timeout = unit.toNanos(timeout);
         while (this.capacity <= 0)
         {
            if (timeout > 0)
            {
               timeout = this.notFull.awaitNanos(timeout);
            }
            else
            {
               // on timeout, we write TASK SET FULL to the transport port
               lock.unlock(); // we don't want to block on transport port operations
               Command command = task.getCommand();
               task.getTargetTransportPort().writeResponse(command.getNexus(),
                     command.getCommandReferenceNumber(), Status.TASK_SET_FULL, null);
               _logger.debug("task set is full, rejecting task: " + task);
               return false;
            }
         }
         
         // Check that untagged tasks have the SIMPLE task attribute
         long taskTag = task.getCommand().getNexus().getTaskTag();
         if (taskTag < 0 && task.getCommand().getTaskAttribute() != TaskAttribute.SIMPLE)
         {
            throw new RuntimeException("Transport layer should have set untagged task as SIMPLE");
         }
         
         // There is a potential race condition where a task can still be in the
         // task map but also completed.  This loop checks to see if a task is still in the map
         // and blocks if a task is in the map and is completed (it waits for the task
         // thread to clean the task out of the map).
         boolean taskInMap = true;
         while(taskInMap)
         {
        	 TaskProxy existingTask = this.tasks.get(taskTag < 0 ? UNTAGGED_TASK_KEY : taskTag);
        	 if(existingTask != null)
        	 {
        		 TaskStatus status = existingTask.getStatus();
        		 if(status == TaskStatus.IN_PROGRESS)
        		 {
        			 // This is treated as an error because the initiator should not be sending
        			 // tasks with duplicate tags before receiving a response, and this cannot
        			 // happen if the task is in the state IN_PROGRESS.
        			 lock.unlock(); // we don't want to block on transport port operations
        			 _logger.warn("Overlapped command condition, task still in task map: " + existingTask.toString());
        			 Command command = task.getCommand();
        			 task.getTargetTransportPort().writeResponse(command.getNexus(),
        					 command.getCommandReferenceNumber(), Status.CHECK_CONDITION,
        					 ByteBuffer.wrap((new OverlappedCommandsAttemptedException(true)).encode()));
        			 return false;
        		 }
        		 else
        		 {
        			 _logger.trace("New task with duplicate tag received during task completion, waiting until existing task is cleaned up");
        			 this.unblocked.awaitNanos(TASK_COMPLETION_WAIT_NS);
        		 }
        	 }
        	 else
        	 {
        		 taskInMap = false;
        	 }
         }

         // wrap task in a task container
         TaskProxy container = new TaskProxy(task);

         // add task to the queue and map
         this.tasks.put(taskTag < 0 ? UNTAGGED_TASK_KEY : taskTag, container); // -1 Q value is 'untagged'

         if (task.getCommand().getTaskAttribute() == TaskAttribute.HEAD_OF_QUEUE)
         {
            this.dormant.add(0, container);
         }
         else
         {
            this.dormant.add(container);
         }

         if (_logger.isTraceEnabled())
         {
            _logger.trace("--------------------------------------------------------------");
            _logger.trace("Printing task set AFTER addTask() execution");
            _logger.trace("Offering new command number: "
                  + Long.toHexString(task.getCommand().getCommandReferenceNumber()));
            _logger.trace(this.toString());
            _logger.trace("offering to taskset command: " + task.getCommand());
            _logger.trace("--------------------------------------------------------------");
         }

         this.capacity--;
         this.notEmpty.signalAll();
         this.unblocked.signalAll();

         if (_logger.isTraceEnabled())
         {   
        	 _logger.trace("Successfully offered command: " + task.getCommand());
             _logger.trace("Outstanding SCSI task set tasks: " + this.tasks.size());
         }
         return true;

      }
      finally
      {
         if (((ReentrantLock) lock).isHeldByCurrentThread())
            lock.unlock();
      }
   }

   /*
    * @see TaskSet#addTask(Task)
    */
   public boolean addTask(Task task)
   {
      try
      {
         return this.addTask(task, 0, TimeUnit.SECONDS);
      }
      catch (InterruptedException e)
      {
         return false;
      }
   }

   /*
    * @see TaskSet#poll(long, TimeUnit)
    */
   public Task poll(long timeout, TimeUnit unit) throws InterruptedException
   {
      lock.lockInterruptibly();

      try
      {
         // wait for the set to be not empty
         timeout = unit.toNanos(timeout);
         while (this.dormant.size() == 0)
         {
            _logger.trace("Task set empty; waiting for new task to be added");
            if (timeout > 0)
            {
               // "notEmpty" is notified whenever a task is added to the set
               timeout = notEmpty.awaitNanos(timeout);
            }
            else
            {
               return null;
            }
         }

         // wait until the next task is not blocked
         while (this.isBlocked(this.dormant.get(0)))
         {
            _logger.trace("Next task blocked; waiting for other tasks to finish");
            if (timeout > 0)
            {
               // "unblocked" is notified whenever a task is finished or a new task is
               // added to the set. We wait on that before checking if this task is still
               // blocked.
               timeout = unblocked.awaitNanos(timeout);
            }
            else
            {
               return null; // a timeout occurred
            }
         }

         TaskProxy container = this.dormant.remove(0);
         this.enabled.add(container);

         if (_logger.isDebugEnabled())
         {
            _logger.debug("Enabling command: " + container.getCommand());
            _logger.debug("Dormant task set: " + this.dormant);
         }

         return container;
      }
      finally
      {
         lock.unlock();
      }
   }

   /**
    * Adds the specified element to this queue. This method deviates from the interface
    * specification in that it does not wait if the task set is full. This is because such waiting
    * would not be ideal.
    * <p>
    * Normally insertion failures can happen for a variety of reasons (see
    * {@link #addTask(Task, long, TimeUnit)}). This method will not communicate a failure with the
    * caller. However, this is not always bad because the error will be written to the target
    * transport port in any case.
    */
   public void addTaskNoWait(Task task) throws InterruptedException
   {
      this.addTask(task);
   }

   /**
    * Called by {@link TaskProxy#run()} when execution of the task is complete.
    */
   private void isFinished(Long taskTag)
   {
      lock.lock(); // task execution thread is finished now, so we don't check interrupts
      try
      {
         Task task = this.tasks.remove(taskTag); // 'null' task tag is the untagged task
         this.enabled.remove(task);
         this.capacity++;
         this.notFull.signalAll();
         this.unblocked.signalAll();
         _logger.debug("Task finished or aborted; outstanding SCSI task set tasks "
               + this.tasks.size());
      }
      finally
      {
         lock.unlock();
      }
   }

   /*
    * Checks if the given task is currently blocked.
    */
   private boolean isBlocked(Task task) throws InterruptedException
   {
      lock.lockInterruptibly();

      try
      {
         TaskAttribute executing =
               enabled.size() > 0
                     ? enabled.get(enabled.size() - 1).getCommand().getTaskAttribute()
                     : null;
         {
            switch (task.getCommand().getTaskAttribute())
            {
               case SIMPLE :
                  if (executing == null || executing == TaskAttribute.SIMPLE)
                     return false;
                  else
                     return true;
               case HEAD_OF_QUEUE :
                  return false;
               case ORDERED :
                  if (executing == null)
                     return false;
                  else
                     return true;
               default :
                  throw new RuntimeException("Unsupported task tag: "
                        + task.getCommand().getTaskAttribute().name());
            }
         }
      }
      finally
      {
         lock.unlock();
      }
   }

   /////////////////////////////////////////////////////////////////////////////

   /*
    * @see TaskSet#abortTask(Nexus)
    */
   public boolean abortTask(Nexus nexus) throws NoSuchElementException, InterruptedException,
         IllegalArgumentException
   {
      if (nexus.getTaskTag() == -1) // invalid Q
         throw new IllegalArgumentException("must provide an I_T_L_Q nexus for abort");

      lock.lockInterruptibly();
      try
      {
         TaskProxy task = this.tasks.remove(nexus.getTaskTag());
         if (task == null)
            throw new NoSuchElementException("Task with tag " + nexus.getTaskTag()
                  + " not in task set");

         return task.abort();
      }
      finally
      {
         lock.unlock();
      }
   }

   /*
    * @see org.jscsi.scsi.tasks.management.TaskSet#clearTaskSet()
    */
   public void clearTaskSet()
   {
      try
      {
         lock.lockInterruptibly();
      }
      catch (InterruptedException e)
      {
         // thread is shutting down, no reason to actually clear the task set.
         // TODO: Is the above statement okay?
         return;
      }

      // We have to copy the list ahead of time because task.abort() modifies this.task
      List<TaskProxy> taskList = new ArrayList<TaskProxy>(this.tasks.values());
      
      try
      {
         for (TaskProxy task : taskList)
         {
            task.abort();
         }
      }
      finally
      {
         lock.unlock();
      }
   }

   /*
    * @see org.jscsi.scsi.tasks.management.TaskSet#clearTaskSet(org.jscsi.scsi.transport.Nexus)
    */
   public void clearTaskSet(Nexus nexus) throws InterruptedException, IllegalArgumentException
   {
      // We don't check for I_T_L nexus because it makes no difference to this implementation.
	  // Normally, CLEAR_TASK_SET would need to check the CONTROL mode page for the contents of
	  // the TST field to make a judgement about whether the nexus is a required field.

      lock.lockInterruptibly();
      try
      {
         // We have to copy the list ahead of time because task.abort() modifies this.task
         List<TaskProxy> taskList = new ArrayList<TaskProxy>(this.tasks.values());
         
         for (TaskProxy task : taskList)
         {
            task.abort();
         }
      }
      finally
      {
         lock.unlock();
      }
   }

   /*
    * @see org.jscsi.scsi.tasks.management.TaskSet#abortTaskSet(org.jscsi.scsi.transport.Nexus)
    */
   public void abortTaskSet(Nexus nexus) throws InterruptedException, IllegalArgumentException
   {
	  // ABORT_TASK_SET in the case of this implementation is identical to
	  // CLEAR_TASK SET because task sets are on a per-I_T_L nexus basis.  Otherwise
	  // CLEAR_TASK_SET would need to check the CONTROL mode page for the contents of
	  // the TST field.
      this.clearTaskSet(nexus);
   }

   /////////////////////////////////////////////////////////////////////////////

   /*
    * @see org.jscsi.scsi.tasks.management.TaskSet#getRemainingCapacity()
    */
   public int getRemainingCapacity()
   {
      // we don't lock here because remainingCapacity() is not guaranteed to be correct
      return this.capacity;
   }

   /*
    * @see org.jscsi.scsi.tasks.management.TaskSet#isEmpty()
    */
   public boolean isEmpty()
   {
      lock.lock();
      try
      {
         return this.dormant.size() != 0;
      }
      finally
      {
         lock.unlock();
      }
   }

   /*
    * @see org.jscsi.scsi.tasks.management.TaskSet#getDormantTaskListSize()
    */
   public int getDormantTaskListSize()
   {
      return this.dormant.size();
   }
   
   public Task takeTask() throws InterruptedException
   {
      Task task = null;
      while (task == null)
      {
         if (_logger.isTraceEnabled())
            _logger.trace("Polling for next task; timeout in 30 seconds");
         task = this.poll(30, TimeUnit.SECONDS);
         if (_logger.isTraceEnabled())
            _logger.trace("returning command for execution: "
                  + (task == null ? "null" : task.getCommand()));
      }
      return task;
   }

   public Object[] toArray()
   {
      lock.lock();
      try
      {
         Object[] objs = new Object[this.dormant.size()];
         for (int i = 0; i < objs.length; i++)
         {
            objs[i] = this.dormant.get(i);
         }
         return objs;
      }
      finally
      {
         lock.unlock();
      }
   }

   @SuppressWarnings("unchecked")
   public <T> T[] toArray(T[] a)
   {
      if (!(a instanceof Task[]))
         throw new ArrayStoreException();

      Task[] dst = null;

      if (a.length < this.dormant.size())
         dst = new Task[this.dormant.size()];
      else
         dst = (Task[]) a;

      for (int i = 0; i < this.dormant.size(); i++)
      {
         dst[i] = this.dormant.get(i);
      }

      if (dst.length > this.dormant.size() + 1)
      {
         dst[this.dormant.size()] = null;
      }

      return (T[]) dst;
   }
   
   @Override
   public String toString()
   {
	   StringBuilder builder = new StringBuilder();
	   builder.append("Full task set: " + toNexusList(this.tasks.values()) + System.getProperty("line.separator"));
       builder.append("Enabled task set: " + toNexusList(this.enabled) + System.getProperty("line.separator"));
       builder.append("Dormant task set: " + toNexusList(this.dormant) + System.getProperty("line.separator"));
       builder.append("Full task set (CRN): " + toCommandReferenceNumberList(this.tasks.values()) + System.getProperty("line.separator"));
       builder.append("Enabled task set (CRN): " + toCommandReferenceNumberList(this.enabled) + System.getProperty("line.separator"));
       builder.append("Dormant task set (CRN): " + toCommandReferenceNumberList(this.dormant) + System.getProperty("line.separator"));
       builder.append("Detailed dormant listing: " + this.dormant + System.getProperty("line.separator"));
       
       return builder.toString();
   }
}
