//Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
//Cleversafe Dispersed Storage(TM) is software for secure, private and
//reliable storage of the world's data using information dispersal.
//
//Copyright (C) 2005-2007 Cleversafe, Inc.
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
//USA.
//
//Contact Information: 
// Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email: licensing@cleversafe.org
//
//END-OF-HEADER
//-----------------------
//@author: John Quigley <jquigley@cleversafe.com>
//@date: January 1, 2008
//---------------------

package org.jscsi.scsi.tasks.management;

import java.util.NoSuchElementException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;

import org.jscsi.scsi.tasks.Status;
import org.jscsi.scsi.tasks.Task;
import org.jscsi.scsi.transport.Nexus;

/**
 * A SAM-2 task set. The task set implements the BlockingQueue interface and provides additional
 * methods for removing a single task identified by a task tag or all tasks.
 * <p>
 * Some implementations may provide one task set for all I_T nexuses. In such cases task management
 * commands will affect outstanding commands for all initiators. 
 */
public interface TaskSet //extends BlockingQueue<Task>
{
	/**
	 * Attempts to insert the given task into the set. When an insertion failure is indicated the
	 * caller must not attempt to take any additional action. I.e., the originating target transport
	 * port is notified of a failure within the method. Thus, failure notifications are for
	 * information purposes only.
	 * <p>
	 * The following conditions can cause an insertion failure:
	 * <ol>
	 * <li>An additional untagged task added before the first untagged task has been cleared.</li>
	 * <li>A tagged task with a duplicate tag added (an overlapping command condition).</li>
	 * <li>The task set is full.</li>
	 * </ol>
	 * <p>
	 * Normally if the task set if full the thread will block until the timeout has expired. If
	 * timeout is set to zero or the after the timeout period the set is still full a
	 * {@link Status#TASK_SET_FULL} status will be sent to the target transport port and
	 * <code>false</code> will be returned.
	 * 
	 * @param task
	 *           The task to insert into the set.
	 * @param timeout
	 *           How long to wait before giving up, in units of <code>unit</code>.
	 * @param unit
	 *           A <code>TimeUnit</code> determining how to interpret the <code>timeout</code>
	 *           parameter.
	 */
   public boolean addTask(Task task, long timeout, TimeUnit unit) throws InterruptedException;
   
   /**
    * Attempts to insert the given task into the set. Equivalent to
    * <code>offer(task, 0, TimeUnit.SECONDS)</code>. In other words, a <code>TASK SET FULL</code>
    * status will be immediately returned to the target transport port when the set is full.
    * <p>
    * 
    * @param task
    *           The task to insert into the set.
    * @returns True if the task was insertion to the queue; False if insertion failed or the thread
    *          was interrupted.
    */
   public boolean addTask(Task task);
   
   /**
    * Retrieves and removes the task at the head of the queue. Blocks on both an empty set and all
    * blocking boundaries specified in SAM-2.
    * <p>
    * The maximum wait time is twice the timeout. This occurs because first we wait for the set to
    * be not empty, then we wait for the task to be unblocked.
    */
   public Task poll(long timeout, TimeUnit unit) throws InterruptedException;
   
   /**
    * Get the next task from the head of the queue.  Blocks until a task is in the queue.
    * 
    * @return The task from the head of the queue.
    */
   public Task takeTask() throws InterruptedException;
	
   /**
    * Aborts and removes the task specified by the given I_T_L_Q nexus. The caller obtains the
    * results of {@link Task#abort()}.
    * <p>
    * This method should be used when removing a task outside the normal execution flow. One
    * such usage is in processing the SAM-2 <code>ABORT TASK</code> task management function.
    * 
    * @param nexus An I_T_L_Q nexus identifying a given task.
    * @return The results of {@link Task#abort()}.
    * @throws NoSuchElementException If a task with the given task tag does not exist.
    * @throws InterruptedException If the thread is interrupted during the abort attempt.
    * @throws IllegalArgumentException If an I_T_L_Q nexus is not provided.
    */
   public boolean abortTask(Nexus nexus) throws NoSuchElementException, InterruptedException,
         IllegalArgumentException;
   
   /**
    * Aborts all tasks in progress and clears the task queue regardless of I_T_L nexus.
    * 
    * @throws InterruptedException If the thread is interrupted during the abort attempt.
    * @throws IllegalArgumentException If an I_T_L nexus is not provided.
    */
   public void clearTaskSet();
   
   /**
    * Aborts all in-progress tasks and clears the task set for the given I_T_L nexus. Succeeds even
    * if some tasks fail to abort. 
    * <p>
    * Implementations which maintain a separate task set per I_T nexus will abort only those tasks
    * which match the indicated initiator. Implementations which maintain a shared task set will
    * abort all tasks for all initiators.
    * 
    * @param nexus An I_T_L nexus identifying a given task set.
    * @throws InterruptedException If the thread is interrupted during the abort attempt.
    * @throws IllegalArgumentException If an I_T_L nexus is not provided.
    */
   public void clearTaskSet(Nexus nexus) throws InterruptedException, IllegalArgumentException;

   /**
    * Aborts in-progress tasks and clears the task set of tasks that were created by the SCSI
    * initiator port and routed through the SCSI target port indicated by the given I_T_L nexus.
    * Succeeds even if some tasks fail to abort.
    * <p>
    * Some simple implementations may call {@link #clearTaskSet(Nexus)} from this method. This will
    * result in less fine-grain task management and should not be used for logical units which
    * aim to properly support multiple simultaneous initiator connections.
    * 
    * @param nexus An I_T_L nexus identifying a source initiator and intermediate port.
    * @throws InterruptedException If the thread is interrupted during the abort attempt.
    * @throws IllegalArgumentException If an I_T_L nexus is not provided.
    */
   public void abortTaskSet(Nexus nexus) throws InterruptedException, IllegalArgumentException;
   
   /**
    * Returns the number of tasks in this task set which are marked as dormant.
    * 
    * @return The size of the dormant task queue.
    */
   public int getDormantTaskListSize();
   
   /**
    * Returns the number of available tasks that this task set can accommodate.
    * 
    * @return The remaining capacity of the task set.
    */
   public int getRemainingCapacity();
   
   /**
    * Determines if the dormant task queue for this task set is empty.
    * 
    * @return true If the task set contains no dormant tasks.
    *         false If the task set contains at least one dormant task.
    */
   public boolean isEmpty();
}
