//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: jquigley
//
// Date: Sep 17, 2007
//---------------------

package org.cleversafe.storage.ss;

import java.util.Map;

import org.cleversafe.authentication.AuthenticationFacilitatorConfiguration;
import org.cleversafe.layer.protocol.ErrorResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.VersionedProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.BadProtocolVersionException;
import org.cleversafe.layer.slicestore.SliceStoreDescriptorManager;
import org.cleversafe.layer.slicestore.SliceStoreManager;
import org.cleversafe.serialization.GridProtocolMessageFactory;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.ServerApplication;
import org.cleversafe.server.handlers.authentication.PasswordAuthenticationHandler;
import org.cleversafe.server.handlers.misc.NoopHandler;
import org.cleversafe.storage.ss.handlers.session.BeginSessionHandler;
import org.cleversafe.storage.ss.handlers.session.EndSessionHandler;
import org.cleversafe.storage.ss.handlers.slice.ExistsHandler;
import org.cleversafe.storage.ss.handlers.slice.IntegrityVerificationHandler;
import org.cleversafe.storage.ss.handlers.slice.ListBeginHandler;
import org.cleversafe.storage.ss.handlers.slice.ListContinueHandler;
import org.cleversafe.storage.ss.handlers.slice.ListInProgressHandler;
import org.cleversafe.storage.ss.handlers.slice.ListStopHandler;
import org.cleversafe.storage.ss.handlers.slice.MultipleReadHandler;
import org.cleversafe.storage.ss.handlers.slice.MultipleRemoveHandler;
import org.cleversafe.storage.ss.handlers.slice.MultipleWriteHandler;
import org.cleversafe.storage.ss.handlers.store.CreateStoreHandler;
import org.cleversafe.storage.ss.handlers.store.RemoveStoreHandler;
import org.cleversafe.storage.ss.handlers.transaction.BeginTransactionHandler;
import org.cleversafe.storage.ss.handlers.transaction.CommitTransactionHandler;
import org.cleversafe.storage.ss.handlers.transaction.RollbackTransactionHandler;
import org.cleversafe.storage.ss.handlers.vault.VaultBindHandler;

/**
 * SliceServerApplication implements the ServerApplication interface,
 * which is responsible for registering all handlers supported by this
 * application.
 * 
 * @see GridServerApplication
 * @see ServerApplication
 * @author Dusty Hendrickson
 * 
 */
public class SliceServerApplication extends ServerApplication
{
   private boolean isProtocolVersioned;
   
   /**
    * Constructs SliceServerApplication taking a SliceServerConfiguration
    * 
    * @param config Configuration used to setup server
    */
   public SliceServerApplication(SliceServerConfiguration config)
   {
      this.setupHandlers(config);
      
      this.isProtocolVersioned = true;
   }

   /**
    * Registers all handlers for the recognized protocol messages
    * 
    * @param config 
    */
   private void setupHandlers(SliceServerConfiguration config)
   {
      // Load configuration objects
      AuthenticationFacilitatorConfiguration authConfiguration = config.getAuthenticationFacilitatorConfiguration();
      SliceStoreDescriptorManager descriptorManager = config.getSliceStoreDescriptorManager();
      Map<String,SliceStoreManager> sliceStoreManagers = config.getVaultSliceStoreManagerMap();
      
      // Authentication Handlers
      // TODO: Do we need both parameters?
      addHandler(new PasswordAuthenticationHandler(authConfiguration, authConfiguration.getDefaultMethod()));
     
      // Store Creation and Removal Handlers
      addHandler(new CreateStoreHandler(descriptorManager, sliceStoreManagers));
      addHandler(new RemoveStoreHandler(descriptorManager, sliceStoreManagers));

      // Vault Binding Handler
      addHandler(new VaultBindHandler(descriptorManager, sliceStoreManagers));

      // SliceStore Session Handlers
      addHandler(new BeginSessionHandler(descriptorManager, sliceStoreManagers));
      addHandler(new EndSessionHandler());

      // Transaction Handlers
      addHandler(new BeginTransactionHandler());
      addHandler(new CommitTransactionHandler());
      addHandler(new RollbackTransactionHandler());

      // Slice Operation Handlers
      addHandler(new ExistsHandler());
      
      addHandler(new ListBeginHandler());
      addHandler(new ListContinueHandler());
      addHandler(new ListInProgressHandler());
      addHandler(new ListStopHandler());
      
      // Multiple Read/Write/Remove
      addHandler(new MultipleReadHandler());
      addHandler(new MultipleWriteHandler());
      addHandler(new MultipleRemoveHandler());
      
      // Miscellaneous handlers
      addHandler(new NoopHandler());
      
      addHandler(new IntegrityVerificationHandler());
   }
   
   /**
    * Verify that a protocol message is compatible with the Slice Server as defined in its
    * configuration.
    * 
    * @see SliceServerApplicationImpl.verifyProtocolVersionInMessage
    */
   public Response service(Request request, ClientSession session)
   {
      if(this.isProtocolVersioned)
      {
         if (request instanceof VersionedProtocolMessage)
         {
            VersionedProtocolMessage req = (VersionedProtocolMessage) request;
            
            if(req.getProtocolVersion() != GridProtocolMessageFactory.getGridProtocolVersion())
            {
               Response response = new ErrorResponse();
               response.setException(new BadProtocolVersionException("Incompatible protocol version detected"));
               return response;
            }
         }
         else
         {
            Response response = new ErrorResponse();
            response.setException(new BadProtocolVersionException("Message does not contain protocol information"));
            return response;
         }
      }
      
      return super.service(request, session);
   }
}
