//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: Apr 26, 2007
//---------------------

package org.cleversafe.storage.ss;

import java.util.Date;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.daemon.Daemonizer;
import org.cleversafe.layer.communication.Acceptor;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;
import org.cleversafe.server.exceptions.ServerIOException;
import org.cleversafe.storage.ss.configuration.ConfigurationLoader;
import org.cleversafe.storage.ss.configuration.XMLConfigurationLoader;
import org.cleversafe.util.Log4jReloader;

/**
 * Class implements a Slice Server Daemon which is responsible for instantiating individual
 * SliceServer objects.
 * 
 */
public class SliceServerDaemon extends Daemonizer implements Runnable
{
   private static Logger _logger = Logger.getLogger(SliceServerDaemon.class);

   private final Acceptor acceptor;

   final Lock lock = new ReentrantLock();
   final Condition shutdown = this.lock.newCondition();
   private final CountDownLatch startupLatch;

   public SliceServerDaemon(SliceServerConfiguration configuration)
   {
      super();

      this.acceptor = configuration.getAcceptor();

      this.acceptor.setServerApplication(new SliceServerApplication(configuration));
      this.acceptor.setProtocolMessageFactory(configuration.getProtocolMessageFactory());

      this.startupLatch = new CountDownLatch(1);
   }

   /**
    * Starts all listening acceptors and waits until the shutdown method is called.
    * 
    * throws ServerIOException If acceptor is unable to start listening. throws InterruptedException
    * If thread is interrupted during shutdown wait.
    */
   public void run()
   {
      _logger.info("Daemon starting");

      try
      {
         this.lock.lock();

         this.acceptor.start();
         this.startupLatch.countDown();

         this.shutdown.await();
      }
      catch (ServerIOException ex)
      {
         _logger.fatal("Acceptor failed to start", ex);
      }
      catch (InterruptedException ex)
      {
         _logger.fatal("Thread interrupted prematurely", ex);
      }
      catch (Exception ex)
      {
         _logger.fatal("Unexpected fatal exception in SliceServerDaemon: " + ex);
      }
      finally
      {
         this.startupLatch.countDown();
         if (this.acceptor.isRunning())
            this.acceptor.stop();
         this.lock.unlock();
      }

      _logger.info("Daemon exiting");
   }

   /**
    * Stops all listening acceptors and notifies run() method that it should stop.
    */
   @Override
   public void shutdown()
   {
      _logger.info("Shutdown requested at: " + new Date());

      this.lock.lock();
      try
      {
         this.acceptor.stop();

         this.shutdown.signalAll();
      }
      finally
      {
         this.lock.unlock();
      }

      _logger.info("Shutdown succeeded at: " + new Date());
   }

   /**
    * Blocks until daemon has been started or a startup error occurs
    * 
    * @throws InterruptedException
    */
   public void awaitStart() throws InterruptedException
   {
      this.startupLatch.await();
   }

   /**
    * Begins a daemon process. Checks to ensure that configuration system properties are set. Daemon
    * will fail to start if configuration system properties are not set, the configuration cannot be
    * loaded, or the listening acceptor is unable to start.
    * 
    * @param args
    *           Command line arguments. Currently unused.
    */
   public static void main(String args[])
   {
      // Check required system properties
      // TODO: Rename daemon property
      String serverConfiguration =
            System.getProperty(XMLConfigurationLoader.SLICE_SERVER_CONFIGURATION_PROPERTY);
      String bindingsConfiguration =
            System.getProperty(ConfigurationFactory.XML_BINDINGS_CONFIG_PROPERTY);
      String loggingConfiguration = System.getProperty("log4j.configuration");
      String daemonCountProperty = System.getProperty("org.cleversafe.storage.ss.daemons");

      // TODO: Reconsider use of logger for these error messages
      if (serverConfiguration == null)
      {
         _logger.fatal("Server configuration file not specified");
         System.exit(1);
      }

      if (bindingsConfiguration == null)
      {
         _logger.fatal("Bindings configuration file not specified");
         System.exit(1);
      }

      if (loggingConfiguration == null)
      {
         _logger.fatal("Logging configuration file not specified");
         System.exit(1);
      }

      int daemonCount = 1;
      if (daemonCountProperty != null)
      {
         daemonCount = Integer.parseInt(daemonCountProperty);
      }

      // Initialize logging
      DOMConfigurator.configure(loggingConfiguration);
      Log4jReloader.launch();

      runDaemon(serverConfiguration, daemonCount);
      _logger.info("Exiting application");
   }

   public static void runDaemon(String serverConfiguration, int daemonCount)
   {
      // Launch daemon
      try
      {
         // Create daemons with generated configurations
         SliceServerDaemon[] daemons = new SliceServerDaemon[daemonCount];
         for (int daemonID = 1; daemonID <= daemons.length; daemonID++)
         {
            // TODO: Figure out how to deal with ports with ID over 10
            Properties properties = new Properties();
            properties.setProperty("DAEMON.ID", Integer.toString(daemonID));

            ConfigurationLoader configLoader = new XMLConfigurationLoader(serverConfiguration);
            SliceServerConfiguration configuration = configLoader.getConfiguration(properties);

            daemons[daemonID - 1] = new SliceServerDaemon(configuration);
         }

         // Create threads for non-primary daemons
         // TODO: What do we want to do with thread exceptions
         Thread[] daemonThreads = new Thread[daemons.length - 1];
         for (int a = 1; a < daemons.length; a++)
         {
            daemonThreads[a - 1] = new Thread(daemons[a], "daemon-" + (a + 1));
            daemonThreads[a - 1].start();
         }

         // Run primary daemon until shutdown
         daemons[0].run();

         // Wait for non-primary daemons to finish
         // TODO: Should we call shutdown on the threads?
         for (int a = 0; a < daemonThreads.length; a++)
         {
            daemonThreads[a].join();
         }
      }
      catch (ServerConfigurationLoadException ex)
      {
         _logger.fatal("Unable to load configuration", ex);
         System.exit(1);
      }
      catch (InterruptedException ex)
      {
         _logger.fatal("Daemon thread died prematurely", ex);
         System.exit(1);
      }
   }
}
