//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: vthornton
//
// Date: Aug 21, 2007
//---------------------

package org.cleversafe.storage.ss.handlers.slice;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.exceptions.ProtocolLayerException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.server.exceptions.IllegalSessionStateException;
import org.cleversafe.server.exceptions.ServerException;
import org.cleversafe.server.exceptions.UnauthorizedRequestException;
import org.cleversafe.vault.exceptions.VaultException;

/**
 * Abstract class which can be extended to perform any operation that depends on a slice store's
 * list permission.
 * 
 * @author Vance Thornton
 */
public abstract class SliceStoreOperationHandler implements RequestHandler
{
   private final List<ApplicationType> _allowedApplications;

   /**
    * Constructs object and creates allowed applications access control list
    */
   public SliceStoreOperationHandler()
   {
      this._allowedApplications = new ArrayList<ApplicationType>();
      this._allowedApplications.add(ApplicationType.TYPE_AUTHENTICATED);
   }

   /**
    * Returns a list of application types that can access this handler
    * 
    * @see RequestHandler.getAllowedApplications()
    */
   public List<ApplicationType> getAllowedApplications()
   {
      return this._allowedApplications;
   }

   /**
    * Services the request if the account has the list permission for the vault
    * 
    * @see RequestHandler.service(Request request, ClientSession session)
    */
   public Response service(Request request, ClientSession session) throws ProtocolLayerException,
         SliceStoreLayerException, ServerException, VaultException
   {
      // Check preconditions
      if (!session.containsKey(ClientSession.SLICE_STORE))
      {
         throw new IllegalSessionStateException(
               "A slice store has not been bound to this session.", request);
      }

      // Check permissions
      if (!session.containsKey(ClientSession.LIST_PERMISSION)
            || (Boolean) session.get(ClientSession.LIST_PERMISSION) == false)
      {
         // Account lacks necessary permissions
         UUID vaultID = (UUID) session.get(ClientSession.VAULT_UUID);
         throw new UnauthorizedRequestException("Invalid list access to vault:" + vaultID, request);
      }

      // Process request
      SliceStore sliceStore = (SliceStore) session.get(ClientSession.SLICE_STORE);

      return performOperation(request, sliceStore);
   }

   protected abstract Response performOperation(Request request, SliceStore sliceStore)
         throws ProtocolLayerException, SliceStoreLayerException, ServerException, VaultException;
}
