//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: June 18, 2007
//---------------------

package org.cleversafe.storage.ss.handlers.store;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.cleversafe.layer.protocol.CreateStoreRequest;
import org.cleversafe.layer.protocol.CreateStoreResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.slicestore.SliceStoreDescriptor;
import org.cleversafe.layer.slicestore.SliceStoreDescriptorManager;
import org.cleversafe.layer.slicestore.SliceStoreManager;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedSliceStoreTypeException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedVaultTypeException;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.exceptions.VaultConfigurationException;
import org.cleversafe.vault.exceptions.VaultIOException;

/**
 * This handler responds to the CreateStoreRequest message, which is sent by a RemoteSliceStore to
 * create a SliceStore of a specified type on a server.
 * 
 * @author Jason
 */
public class CreateStoreHandler implements RequestHandler
{
   private final List<ApplicationType> _allowedApplications;
   private final Map<String, SliceStoreManager> vaultSliceStoreManagerMap;
   private final SliceStoreDescriptorManager descriptorManager;

   /**
    * Constructs object, creates allowed applications access control list and saves reference to the
    * SliceServerConfiguration object
    * 
    * @param config
    */
   public CreateStoreHandler(
         SliceStoreDescriptorManager descriptorManager,
         Map<String, SliceStoreManager> vaultSliceStoreManagerMap)
   {
      this._allowedApplications = new ArrayList<ApplicationType>();
      this._allowedApplications.add(ApplicationType.TYPE_AUTHENTICATED);
      this.vaultSliceStoreManagerMap = vaultSliceStoreManagerMap;
      this.descriptorManager = descriptorManager;
   }

   /**
    * Returns class for the type of message this handler services
    * 
    * @see RequestHandler.getRequestClass()
    */
   public Class<CreateStoreRequest> getRequestClass()
   {
      return CreateStoreRequest.class;
   }

   /**
    * Returns a list of application types that can access this handler
    * 
    * @see RequestHandler.getAllowedApplications()
    */
   public List<ApplicationType> getAllowedApplications()
   {
      return this._allowedApplications;
   }

   /**
    * Calls SliceStoreConfig.createSliceStore()
    * 
    * @see RequestHandler.service(Request request, ClientSession session)
    */
   public Response service(Request request, ClientSession session) throws VaultIOException,
         SliceStoreExistsException, SliceStoreIOException, UnsupportedVaultTypeException,
         UnsupportedSliceStoreTypeException
   {
      // Process request
      CreateStoreRequest createStoreRequest = (CreateStoreRequest) request;

      // Get requested vault type
      String vaultType = createStoreRequest.getVaultType();

      // Get requested max slice size
      long maxSliceSize = createStoreRequest.getMaxSliceSize();

      // Get requested slice store size
      long sliceStoreSize = createStoreRequest.getSliceStoreSize();

      // Get Store Options
      Map<String, String> options = createStoreRequest.getOptions();

      // Get requested Vault ID
      UUID vaultID = createStoreRequest.getVaultID();

      // Get the vault descriptor bytes
      byte[] vaultDescriptorBytes = createStoreRequest.getVaultDescriptorBytes();

      // Get Access Control List
      VaultACL vaultACL;
      try
      {
         vaultACL = createStoreRequest.getVaultACL();
      }
      catch (VaultConfigurationException ex)
      {
         throw new VaultIOException("Could not retrieve Vault ACL", ex);
      }

      // Perform Vault Creation
      CreateStoreResponse response = new CreateStoreResponse();

      try
      {
         SliceStoreManager manager = this.vaultSliceStoreManagerMap.get(vaultType);
         if (manager == null)
         {
            throw new SliceStoreIOException("No SliceStoreManager found for vault type "
                  + vaultType);
         }

         manager.createSliceStore(vaultType, maxSliceSize, sliceStoreSize, vaultID, vaultACL,
               vaultDescriptorBytes, options);

         SliceStoreDescriptor descriptor = new SliceStoreDescriptor(vaultType);
         this.descriptorManager.saveDescriptor(vaultID, descriptor);
      }
      catch (SliceStoreLayerException ex)
      {
         response.setException(ex);
      }

      return response;
   }
}
