//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: June 18, 2007
//---------------------

package org.cleversafe.storage.ss.handlers.store;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.cleversafe.layer.protocol.RemoveStoreRequest;
import org.cleversafe.layer.protocol.RemoveStoreResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.slicestore.SliceStoreDescriptor;
import org.cleversafe.layer.slicestore.SliceStoreDescriptorManager;
import org.cleversafe.layer.slicestore.SliceStoreManager;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedSliceStoreTypeException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedVaultTypeException;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.server.exceptions.IllegalSessionStateException;
import org.cleversafe.server.exceptions.UnauthorizedRequestException;
import org.cleversafe.vault.VaultPermission;

/**
 * Class implements handler for the RemoveStoreRequest, which is sent by a RemoteSliceStore to
 * delete a slice store on a server.
 * 
 * @author Jason
 */
public class RemoveStoreHandler implements RequestHandler
{
   private final List<ApplicationType> _allowedApplications;
   private final SliceStoreDescriptorManager sliceStoreDescriptorManager;
   private final Map<String, SliceStoreManager> vaultSliceStoreManagerMap;

   /**
    * Constructs object, creates allowed applications access control list and saves reference to the
    * SliceServerConfiguration object
    * 
    * @param config
    */
   public RemoveStoreHandler(
         SliceStoreDescriptorManager sliceStoreDescriptorManager,
         Map<String, SliceStoreManager> vaultSliceStoreManagerMap)
   {
      this.sliceStoreDescriptorManager = sliceStoreDescriptorManager;
      this.vaultSliceStoreManagerMap = vaultSliceStoreManagerMap;

      this._allowedApplications = new ArrayList<ApplicationType>();
      this._allowedApplications.add(ApplicationType.TYPE_AUTHENTICATED);
   }

   /**
    * Returns class for the type of message this handler services
    * 
    * @see RequestHandler.getRequestClass()
    */
   public Class<RemoveStoreRequest> getRequestClass()
   {
      return RemoveStoreRequest.class;
   }

   /**
    * Returns a list of application types that can access this handler
    * 
    * @see RequestHandler.getAllowedApplications()
    */
   public List<ApplicationType> getAllowedApplications()
   {
      return this._allowedApplications;
   }

   /**
    * Calls SliceStoreConfig.deleteSliceStore() after validating the client has the delete store
    * permission
    * 
    * @see RequestHandler.service(Request request, ClientSession session)
    */
   public Response service(Request request, ClientSession session)
         throws IllegalSessionStateException, UnauthorizedRequestException, SliceStoreIOException,
         SliceStoreNotFoundException, UnsupportedVaultTypeException,
         UnsupportedSliceStoreTypeException
   {
      // Check pre-conditions
      if (session.containsKey(ClientSession.SLICE_STORE))
      {
         throw new IllegalSessionStateException(
               "The slice store to be removed must not be active in this session.", request);
      }
      if (!session.containsKey(ClientSession.VAULT_UUID))
      {
         throw new IllegalSessionStateException(
               "A vault identifier was not found on this session.", request);
      }
      if (!session.containsKey(ClientSession.VAULT_PERMISSION))
      {
         throw new IllegalSessionStateException(
               "A permission object was not found on this session.", request);
      }

      // Check permissions
      UUID vaultID = (UUID) session.get(ClientSession.VAULT_UUID);
      VaultPermission permission = (VaultPermission) session.get(ClientSession.VAULT_PERMISSION);
      VaultPermission deleteVaultPermission =
            new VaultPermission(vaultID.toString(), VaultPermission.DELETE_VAULT);

      if (permission == null || !permission.implies(deleteVaultPermission))
      {
         throw new UnauthorizedRequestException("No permission to access vault:" + vaultID, request);
      }

      RemoveStoreRequest removeStoreRequest = (RemoveStoreRequest) request;

      // Ensure the VaultID in the request matches the vault the session is bound to
      if (!vaultID.equals(removeStoreRequest.getVaultIdentifier()))
      {
         throw new IllegalSessionStateException(
               "The vault to be removed was not bound to this session", request);
      }

      // Process request
      RemoveStoreResponse response = new RemoveStoreResponse();

      try
      {
         SliceStoreDescriptor descriptor = this.sliceStoreDescriptorManager.loadDescriptor(vaultID);

         SliceStoreManager manager = this.vaultSliceStoreManagerMap.get(descriptor.getVaultType());
         if (manager == null)
         {
            throw new SliceStoreIOException("No SliceStoreManager found for vault type "
                  + descriptor.getVaultType());
         }

         Map<String, String> options = new HashMap<String, String>();

         manager.deleteSliceStore(vaultID, options);

         this.sliceStoreDescriptorManager.deleteDescriptor(vaultID);
      }
      catch (SliceStoreLayerException ex)
      {
         response.setException(ex);
      }

      return response;

   }

}
