//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason
//
// Date: May 15, 2007
//---------------------

package org.cleversafe.storage.ss.handlers.transaction;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.server.exceptions.IllegalSessionStateException;

/**
 * Handler for CommitTransactionRequest message
 * 
 * @author Jason Resch
 */
public class CommitTransactionHandler implements RequestHandler
{
   private static Logger _logger = Logger.getLogger(CommitTransactionHandler.class);

   private final List<ApplicationType> _allowedApplications;

   /**
    * Constructs object and creates allowed applications access control list
    */
   public CommitTransactionHandler()
   {
      this._allowedApplications = new ArrayList<ApplicationType>();
      this._allowedApplications.add(ApplicationType.TYPE_AUTHENTICATED);
   }

   /**
    * Returns class for the type of message this handler services
    * 
    * @see RequestHandler.getRequestClass()
    */
   public Class<CommitTransactionRequest> getRequestClass()
   {
      return CommitTransactionRequest.class;
   }

   /**
    * Returns a list of application types that can access this handler
    * 
    * @see RequestHandler.getAllowedApplications()
    */
   public List<ApplicationType> getAllowedApplications()
   {
      return this._allowedApplications;
   }

   /**
    * Calls SliceStore.commitTransaction() on the SliceStore saved in the client session.
    * 
    * @see RequestHandler.service(Request request, ClientSession session)
    */
   public Response service(Request request, ClientSession session)
         throws IllegalSessionStateException, SliceStoreTransactionException, SliceStoreIOException
   {
      if (_logger.isTraceEnabled())
         _logger.trace("Servicing commit transaction request");

      // Check pre-conditions
      if (!session.containsKey(ClientSession.SLICE_STORE))
      {
         throw new IllegalSessionStateException(
               "A slice store has not been bound to this session.", request);
      }

      // Process request
      SliceStore sliceStore = (SliceStore) session.get(ClientSession.SLICE_STORE);

      CommitTransactionRequest commitTransactionRequest = (CommitTransactionRequest) request;

      // Perform Create Transaction
      CommitTransactionResponse response = new CommitTransactionResponse();
      long id = -1;
      try
      {
         id = commitTransactionRequest.getTransactionID();
         SliceStoreTransaction transaction = sliceStore.getTransaction(id);
         _logger.trace("Committing transaction " + id);
         sliceStore.commitTransaction(transaction);
         _logger.debug("Transaction committed with ID " + id);
      }
      catch (SliceStoreLayerException ex)
      {
         _logger.error("Commit txid=" + id + "failed" + ex.getMessage());
         response.setException(ex);
      }

      return response;
   }

}
