//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 12, 2007
//---------------------

package org.cleversafe.storage.ss.handlers.vault;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.VaultBindRequest;
import org.cleversafe.layer.protocol.VaultBindResponse;
import org.cleversafe.layer.slicestore.SliceStoreDescriptor;
import org.cleversafe.layer.slicestore.SliceStoreDescriptorManager;
import org.cleversafe.layer.slicestore.SliceStoreManager;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedSliceStoreTypeException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedVaultTypeException;
import org.cleversafe.policy.exceptions.AccessException;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.server.exceptions.IllegalSessionStateException;
import org.cleversafe.server.exceptions.UnauthorizedRequestException;
import org.cleversafe.vault.VaultPermission;
import org.cleversafe.vault.exceptions.VaultConfigurationException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * Handler for VaultBindRequest message
 * 
 * @author Jason Resch
 */
public class VaultBindHandler implements RequestHandler
{
   private static Logger _logger = Logger.getLogger(VaultBindHandler.class);

   // tracks vault usage, important for enforcing policy for scenarios such
   // as when we wish to only allow one binding to a particular vault
   // FIXME: Should this vector be static? It causes issues when starting multiple servers
   // within the same application.
   private static Vector<UUID> boundVaults = new Vector<UUID>();

   private final List<ApplicationType> applicationTypes;

   private final SliceStoreDescriptorManager sliceStoreDescriptorManager;
   private final Map<String, SliceStoreManager> vaultSliceStoreManagerMap;

   // TODO: this should become part of a large policy tracking framework
   private static final boolean allowConcurrentBind = true;

   /**
    * Constructs object, creates allowed applications access control list and saves reference to the
    * SliceServerConfiguration object
    * 
    * @param config
    */
   public VaultBindHandler(
         SliceStoreDescriptorManager sliceStoreDescriptorManager,
         Map<String, SliceStoreManager> vaultSliceStoreManagerMap)
   {
      this.sliceStoreDescriptorManager = sliceStoreDescriptorManager;
      this.vaultSliceStoreManagerMap = vaultSliceStoreManagerMap;

      this.applicationTypes = new ArrayList<ApplicationType>(1);
      this.applicationTypes.add(ApplicationType.TYPE_AUTHENTICATED);
   }

   /**
    * Returns a list of application types that can access this handler
    * 
    * @see RequestHandler.getAllowedApplications()
    */
   public List<ApplicationType> getAllowedApplications()
   {
      return this.applicationTypes;
   }

   /**
    * Returns class for the type of message this handler services
    * 
    * @see RequestHandler.getRequestClass()
    */
   public Class<VaultBindRequest> getRequestClass()
   {
      return VaultBindRequest.class;
   }

   /**
    * Verifies that the account has the appropriate permission to bind to a vault, also checks that
    * the vault is not already bound by another client. Stores the vault identifier in the session
    * if the request is granted.
    * 
    * @see RequestHandler.service(Request request, ClientSession session)
    */
   public Response service(Request request, ClientSession session)
         throws IllegalSessionStateException,
         SliceStoreIOException, SliceStoreNotFoundException,
         UnsupportedVaultTypeException, UnsupportedSliceStoreTypeException
   {
      // Check pre-conditions
      if (!session.containsKey(ClientSession.GRID_ACCOUNT_UUID))
      {
         throw new IllegalSessionStateException("Account UUID has not been set in the session",
               request);
      }
      if (session.containsKey(ClientSession.SLICE_STORE))
      {
         // IMPORTANT: This check is imperative to guard against an attack whereby a user
         // gains access to one vault they have limited permissions on and then bind to a
         // different vault on which they have greater permission. This would overwrite
         // their permission object in their session giving them access they shouldn't have - JKR
         throw new IllegalSessionStateException(
               "Cannot bind to a vault while the slice store is active in the session.", request);
      }

      // Create request
      VaultBindRequest bindRequest = (VaultBindRequest) request;

      // create response
      VaultBindResponse bindResponse = new VaultBindResponse();

      UUID vaultID = bindRequest.getVaultIdentifier();

      UUID gridAccountID = (UUID) session.get(ClientSession.GRID_ACCOUNT_UUID);

      // Check the ACL Permissions

      SliceStoreDescriptor descriptor = this.sliceStoreDescriptorManager.loadDescriptor(vaultID);

      SliceStoreManager manager = this.vaultSliceStoreManagerMap.get(descriptor.getVaultType());
      if (manager == null)
      {
         throw new SliceStoreIOException("No SliceStoreManager found for vault type "
               + descriptor.getVaultType());
      }

      VaultPermission permission = manager.loadVaultPermissions(vaultID, gridAccountID);

      // Store permissions in session

      VaultPermission listPermission =
            new VaultPermission(vaultID.toString(), VaultPermission.LIST);
      session.put(ClientSession.LIST_PERMISSION,
            ((permission != null) && (permission.implies(listPermission))));

      VaultPermission readPermission =
            new VaultPermission(vaultID.toString(), VaultPermission.READ);
      session.put(ClientSession.READ_PERMISSION,
            ((permission != null) && (permission.implies(readPermission))));

      VaultPermission writePermission =
            new VaultPermission(vaultID.toString(), VaultPermission.WRITE);
      session.put(ClientSession.WRITE_PERMISSION,
            ((permission != null) && (permission.implies(writePermission))));

      if ((Boolean) session.get(ClientSession.LIST_PERMISSION) == false)
      {
         // Account lacks permissions for the vault
         bindResponse.setException(new UnauthorizedRequestException("Invald list access to vault: "
               + vaultID.toString(), request));
      }
      else if (boundVaults.contains(vaultID) && (!allowConcurrentBind))
      {
         _logger.error("Server is configured to disallow concurrent vault binds");
         bindResponse.setException(new AccessException(vaultID,
               "Concurrent vault binds are disallowed"));
      }
      else
      {
         // Store the Vault ID and Permission object in the session
         session.put(ClientSession.VAULT_UUID, vaultID);
         session.put(ClientSession.VAULT_PERMISSION, permission);

         boundVaults.add(vaultID);

         _logger.info("Allowing account " + gridAccountID.toString() + " access to vault "
               + vaultID.toString());
      }

      return bindResponse;
   }
}
