package org.cleversafe.storage.ss.handlers;


import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.UUID;

import org.apache.log4j.BasicConfigurator;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.GridController;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.block.BlockFileSliceStore;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.storage.ss.SliceServerApplication;
import org.cleversafe.storage.ss.SliceServerConfiguration;
import org.cleversafe.storage.ss.configuration.ConfigurationLoader;
import org.cleversafe.storage.ss.configuration.XMLConfigurationLoader;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.VaultPermission;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.junit.BeforeClass;

public class SliceStorePerformanceBase
{

	// How long test should be run. This is mutually exclusive with testIterations
	// Only one of them is used.
	public static int testDuration;
	// I/O Request Size in KB
	public static int requestSize;
	// Number of concurrent requests
	public static int numConcurrentRequests;

	// how many writes of size requestSize (K) in each write in MultpleWriteHandler call.
	protected static int numRequestsInOper;
	
	// how many operations to be performed. This is mutually exclusive with testDuration.
	// Only one of them is used.
	public static int testIterations;
	public static final int txnID = 1;
	
	public static final String TEST_PATH =
         "test-data/" + GridController.class.getName().replace('.', '/');

	public static final String TEST_OUTPUT_PATH = "slice-data";
   
	public static UUID accountUUID;
	public static UUID vaultIdentifier;
	
	public SliceServerApplication sliceServer = null;

	public static ClientSession session = null;

	public static byte[] data;
	public static DataSlice[] dataSlices = null;
	public static SliceName[] sliceNames = null;
	public static String[] sliceStrings = null;
	public static String storeType;
	
	@BeforeClass
	public static void initializeTests()
	{
		// Only one of testIterations or testDuration is used.
		testIterations = Integer.getInteger("test.operations", 1); //
		testDuration = Integer.getInteger("test.duration", 1); // 1 minutes
		numConcurrentRequests = Integer.getInteger("test.concurrency", 1); //
		requestSize = Integer.getInteger("test.requestsize", 4); // 4K

		numRequestsInOper = Integer.getInteger("test.numrequestsinop", 10);
		
		storeType = System.getProperty("test.storetype", "blockfile");
		// prepare dataSlices[]
		sliceStrings = new String[testIterations];
		dataSlices = new DataSlice[testIterations];
		sliceNames = new SliceName[testIterations];

		data = new byte[ requestSize ];
		randomizeBuffer(data);
	
		for (int i=0; i<testIterations; i++){
			sliceStrings[i] = Integer.toString(i);
			sliceNames[i] =  new SliceName(sliceStrings[i], 0);
			dataSlices[i] =  new DataSlice(sliceNames[i], txnID, data);		
		}
		// Setup basic logging
		BasicConfigurator.configure();
		
		//  delete the data dir
		deleteDirectory(TEST_OUTPUT_PATH);
		System.out.println("test.operations=" + testIterations + " numConcurrentRequests=" + numConcurrentRequests + " requestSize=" + requestSize
				+ " storeType=" + storeType);

	}
	
	/**
	 * Attempts to create a SliceStore in the session and checks that it is
	 * operational
	 * 
	 */
	protected ClientSession createSession() throws Exception {
	   
	   System.setProperty("org.cleversafe.xml.properties", "conf/dev/block-device/properties.xml");
	   
	   System.setProperty("org.cleversafe.xml.configuration", "blockdevice-org/test-input/org/cleversafe/block/BlockDeviceUDCTest/core-org-bindings.xml;blockdevice-org/test-input/org/cleversafe/block/BlockDeviceUDCTest/blockdevice-org-bindings.xml");
	   
	   // Create Server Application
      String serverConfiguration = "blockdevice-org/test-input/org/cleversafe/block/BlockDeviceUDCTest/server-config.xml"; //System.getProperty("org.cleversafe.storage.ss.xml.configuration");
      Properties properties = new Properties();
      properties.setProperty("DAEMON.ID", "1");     
      ConfigurationLoader configLoader = new XMLConfigurationLoader(serverConfiguration);
      SliceServerConfiguration configuration = configLoader.getConfiguration(properties);
      

	   // Create permission object with full permissions
	   vaultIdentifier = UUID.randomUUID();

		sliceServer = null;
		try {
			sliceServer = new SliceServerApplication(configuration);
		} catch (Exception ex) {
		   ex.printStackTrace();
			fail("Unable to construct sliceServer");
		}
	   
	   // Setup the session appropriately
	   ClientSession session = new ClientSession();
		
	   session.put(ClientSession.GRID_ACCOUNT_UUID, accountUUID);
	   session.put(ClientSession.AUTHENTICATED, true);
	   session.put(ClientSession.APPLICATION_TYPE,
				ApplicationType.TYPE_AUTHENTICATED);
	   session.put(ClientSession.WRITE_PERMISSION, true);
	   session.put(ClientSession.READ_PERMISSION, true);
	   session.put(ClientSession.LIST_PERMISSION, true);

	   // Create the Store
	   File sliceDataDirectory = new File("slice-data");
	   sliceDataDirectory.mkdirs();
	   SliceStore store = null;
	   if (storeType.equals("bdb")) {
		   System.out.println("Creating BDB sliceStore");
		   // TODO: Fix SliceStore
		   store = null; //new BDBSliceStore(sliceDataDirectory);
	   } else if (storeType.equals("blockfile")){
		   System.out.println("Creating blockFile sliceStore");
		   store = new BlockFileSliceStore( vaultIdentifier, TEST_OUTPUT_PATH);
		   store.createStore("block", 8192, SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED, createACL(), null);
		   //(vaultType, maxSliceSize, sliceStoreSize, accessControlList, vaultDescriptorBytes)
	   }
	   
	   
	   if (!store.isOperational()) {
		   store.startSession();
	   }
	   // Save store in the session
	   session.put(ClientSession.SLICE_STORE, store);
		
	   VaultPermission permission = VaultPermission.getFullPermissions(vaultIdentifier);
		
	   // Store vault permission and vault identifier in session
	   session.put(ClientSession.VAULT_PERMISSION, permission);
	   session.put(ClientSession.VAULT_UUID, vaultIdentifier);
		
	   return session;
	}

	/**
	 * Creates a connection to a SliceServer, optionally authenticates
	 * 
	 * @param authenticate
	 *            Flag specifying if the connection should be authenticated
	 * @return A connection to a slice server
	 */
	private static VaultACL createACL() throws Exception
	{
		String vaultFileName = TEST_OUTPUT_PATH + vaultIdentifier.toString() + ".der";
		FileOutputStream out = new FileOutputStream(vaultFileName);
		
		PasswordCredentials credentials = new PasswordCredentials();
		credentials.setUsername("vaultacltest");
		credentials.setPassword("password");
		
		KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
		keygen.initialize(512);
		KeyPair keypair = keygen.generateKeyPair();
		
		List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();
		
		VaultACLFactory fact = new VaultACLFactory();
		fact.create(out, vaultIdentifier, credentials, keypair.getPublic(), keyInfoList);
		out.close();
		
		FileInputStream in = new FileInputStream(vaultFileName);
		
		VaultACL vaultACL = fact.getInstance(in);
		in.close();
		
		return vaultACL;
	}
	
	protected static void randomizeBuffer(byte data[]) {
		SecureRandom generator = new SecureRandom();
		generator.nextBytes(data);
	}

	public static boolean deleteDirectory(String dir)
	{
		return deleteDirectory(new File(dir));
	}
	public static boolean deleteDirectory(File dir)
	{
		if (dir.isDirectory())
		{
			String[] children = dir.list();
			for (int i = 0; i < children.length; i++)
			{
				boolean success = deleteDirectory(new File(dir, children[i]));

	            if (!success)
	            {
	               return false;
	            }
			}
		}

		// The directory is now empty so delete it
		return dir.delete();
	}
}
