package org.cleversafe.storage.ss.handlers.slices;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.BeginTransactionResponse;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionResponse;
import org.cleversafe.layer.protocol.ErrorResponse;
import org.cleversafe.layer.protocol.GridProtocolResponse;
import org.cleversafe.layer.protocol.MultipleReadRequest;
import org.cleversafe.layer.protocol.MultipleReadResponse;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.MultipleWriteResponse;
import org.cleversafe.server.ClientSession;
import org.cleversafe.storage.ss.handlers.SliceStorePerformanceBase;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.junit.Test;
public class ReadHandlerPerformanceTest extends SliceStorePerformanceBase
{


	
	//@Test
	public void readHandlerPerformance()
	{
		// Create a session in which a slices have been written.
		session = writeSlices();
		
		// Start a new transaction for removing.
		try {
			// Create Transaction
			BeginTransactionResponse beginTransactionResponse = (BeginTransactionResponse) this.sliceServer
					.service(new BeginTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer",
					beginTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
		
		int sliceNum;
		long beginTest, endTest;
		beginTest = System.currentTimeMillis();
		System.out.println("Begin test." + System.currentTimeMillis());
		for (sliceNum = 1; sliceNum < testIterations; sliceNum++) {
			try {
				// Create Request
			   MultipleReadResponse response = (MultipleReadResponse) this.sliceServer
				.service(new MultipleReadRequest(new SliceName( Integer.toString(sliceNum), 0)), session);
				
				// Ensure no exception flag was set
				assertTrue("An exception was thrown by the SliceServer",
						response.getExceptionFlag() == false);
			} catch (ClassCastException ex) {
				fail("SliceServer returned an inproper Response type");
			}
		}

		endTest = System.currentTimeMillis();
		
		try {
			// Commit Transaction
			CommitTransactionResponse commitTransactionResponse = (CommitTransactionResponse) this.sliceServer
					.service(new CommitTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer",
					commitTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
	
		PerformanceResults results =  new PerformanceResults("ReadHandler", testIterations, requestSize, beginTest, endTest);
		results.print(true);
		
		long duration = endTest - beginTest;
		System.out.println("End test" + System.currentTimeMillis());
		System.out.println("Threads : 1 Operations + " + testIterations + " : Duration(ms) " + duration +
							" : Ops/s " + (testIterations * 1000.0)/ duration + " : Rate (MB/s)" 
							+ (testIterations * 1000.0 * requestSize)/ (1024 * duration));

	}

	/*
	 * Writes number of slices equal to the number of testIterations.
	 * 
	 */
	private ClientSession writeSlices()
	{
		
		session = null;
		// Create a session
		if (session == null) {
			try {
				// Create a session with a valid SliceStore
				session = createSession();
			} catch (Exception ex) {
				ex.printStackTrace();
				fail("Unable to start an operational slice store");
			}
		}
		
		// Create a transaction
		try {
			// Create Transaction
			BeginTransactionResponse beginTransactionResponse = (BeginTransactionResponse) this.sliceServer
					.service(new BeginTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer",
					beginTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
		/* 
		 * Create data once and write the same stuff over and over again.
		 * 
		 */
	
		int i;
		for (i=1; i< testIterations; i++) {
			writeSlice(i, data);
		}
		
		// Commit the transaction.
		try {
			// Commit Transaction
			CommitTransactionResponse commitTransactionResponse = (CommitTransactionResponse) this.sliceServer
					.service(new CommitTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer",
					commitTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}

		return session;
	}
	
	
	@Test
	public void readHandlerPerformance_n()
	{
	
		// Create a session in which a slices have been written. No need to do in threads.
		session = writeSlices();
	
		class WorkItem implements Runnable
		{
			int sliceNum;
			public WorkItem(int slice)
			{
				sliceNum = slice;
			}
			public void run()
			{
				ReadHandlerPerformanceTest test = ReadHandlerPerformanceTest.this;;
				try {
					// Create Request and ask server to service it.
					GridProtocolResponse response = (GridProtocolResponse) test.sliceServer
					.service(new MultipleReadRequest( sliceNames[sliceNum]), session);
					
					if (response instanceof ErrorResponse) 
					{
						System.out.println(response.getException());
					} else {				
					   MultipleReadResponse rs = (MultipleReadResponse)response;
					}
					// Ensure no exception flag was set
					assertTrue("An exception was thrown by the SliceServer",
							response.getExceptionFlag() == false);
				} catch (ClassCastException ex) {
					fail("SliceServer returned an inproper Response type");
				}
				
			}
		}
		
		ExecutorService executor = new BoundedThreadPoolExecutor("Read handler Test", numConcurrentRequests);

		// Create a transaction
		try {
			// Create Transaction
			BeginTransactionResponse beginTransactionResponse = (BeginTransactionResponse) this.sliceServer
					.service(new BeginTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer",
					beginTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
		
		// Start Operations.
		int operations;
		long beginTest, endTest;
		beginTest = System.currentTimeMillis();
		System.out.println("Begin test.");
		for (operations = 1; operations < testIterations; operations++) {
			// execute in an available thread.
			executor.execute( new WorkItem(operations) );
		}
		      
		executor.shutdown();
		System.out.println("Waiting for threads to stop.");
		boolean finished = false;
		do {
			try {
				finished = executor.awaitTermination( 10, TimeUnit.SECONDS );
			} catch (InterruptedException e){
				e.printStackTrace();
			}
		} while ( !finished );

		endTest = System.currentTimeMillis();
		
		try {
			// Create Transaction
			CommitTransactionResponse commitTransactionResponse = (CommitTransactionResponse) this.sliceServer
					.service(new CommitTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			if (commitTransactionResponse.getExceptionFlag()) {
				commitTransactionResponse.getException().printStackTrace();
			}
			assertTrue("An exception was thrown by the SliceServer",
					commitTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
		
		PerformanceResults results =  new PerformanceResults("ReadHandler", testIterations, requestSize, beginTest, endTest);
		results.print(true);
		
	}
	
	/**
	 * Sends an Write request to the server and verifies that the slice is
	 * stored
	 * 
	 */
	private void writeSlice(int sliceNum, byte[] data) {

		// Create a begin session request
	   MultipleWriteRequest request = new MultipleWriteRequest( dataSlices[sliceNum]);

		try {
			// Process the request on our custom ClientSession object
		   MultipleWriteResponse response = (MultipleWriteResponse) this.sliceServer.service(
					request, session);

			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer", response
					.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
	}
}
