//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: dhendrickson
//
// Date: Nov 21, 2007
//---------------------

package org.cleversafe.storage.ss;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;

import org.cleversafe.authentication.JAASConfiguration;
import org.cleversafe.layer.protocol.BeginSessionRequest;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CreateStoreRequest;
import org.cleversafe.layer.protocol.EndSessionRequest;
import org.cleversafe.layer.protocol.ExistsRequest;
import org.cleversafe.layer.protocol.IntegrityVerificationRequest;
import org.cleversafe.layer.protocol.ListBeginRequest;
import org.cleversafe.layer.protocol.ListContinueRequest;
import org.cleversafe.layer.protocol.ListInProgressRequest;
import org.cleversafe.layer.protocol.ListStopRequest;
import org.cleversafe.layer.protocol.MultipleReadRequest;
import org.cleversafe.layer.protocol.MultipleRemoveRequest;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.layer.protocol.PasswordAuthenticationRequest;
import org.cleversafe.layer.protocol.ProtocolOperation;
import org.cleversafe.layer.protocol.RemoveStoreRequest;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.RollbackTransactionRequest;
import org.cleversafe.layer.protocol.VaultBindRequest;
import org.cleversafe.layer.protocol.VersionedProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.ProtocolLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.serialization.GridProtocolMessageFactory;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.server.exceptions.ServerException;
import org.cleversafe.server.exceptions.ServerRequestException;
import org.cleversafe.server.exceptions.UnauthorizedRequestException;
import org.cleversafe.server.exceptions.UnrecognizedRequestException;
import org.cleversafe.test.BaseTest;
import org.cleversafe.vault.exceptions.VaultException;
import org.junit.Test;

public class SliceServerApplicationTest extends BaseTest
{
   protected static class CustomHandlerProtocolOperation implements ProtocolOperation
   {   
      public int getOperationCode()
      {
         return 12345;     
      }         
   };
   
   protected static class CustomHandlerRequest implements Request, VersionedProtocolMessage
   {
      public ProtocolOperation getOperation()
      {
         return new CustomHandlerProtocolOperation();
      }

      public boolean isRequest()
      {
         return true;
      }

      public boolean isResponse()
      {
         return false;
      }

      public boolean isUnsolicited()
      {
         return false;
      }

      public short getProtocolIdentifier()
      {
         return 0;
      }

      public short getProtocolVersion()
      {
         return GridProtocolMessageFactory.getGridProtocolVersion();
      }

      public void setProtocolIdentifier(short identifier)
      {         
      }

      public void setProtocolVersion(short version)
      {
      }      
   }
   
   protected static class CustomHandlerResponse implements Response
   {
      public Exception getException()
      {
         return null;
      }

      public boolean getExceptionFlag()
      {
         return false;
      }

      public void setException(Exception exception)
      {
      }

      public ProtocolOperation getOperation()
      {
         return null;
      }

      public boolean isRequest()
      {
         return false;
      }

      public boolean isResponse()
      {
         return true;
      }

      public boolean isUnsolicited()
      {
         return false;
      }
   }
   
   protected static class CustomHandler implements RequestHandler
   {
      private boolean doAssert;
      private boolean doException;
      private boolean requireAuth;
      
      public CustomHandler(boolean doAssert, boolean doException, boolean requireAuth)
      {
         this.doAssert = doAssert;
         this.doException = doException;
         this.requireAuth = requireAuth;
      }
      
      public List<ApplicationType> getAllowedApplications()
      {
         List<ApplicationType> types = new ArrayList<ApplicationType>();
         
         if( requireAuth )
         {
            types.add(ApplicationType.TYPE_AUTHENTICATED);
         }
         else
         {
            types.add(ApplicationType.TYPE_ANY);
         }
         
         return types;
      }

      public Class<? extends Request> getRequestClass()
      {
         return CustomHandlerRequest.class;
      }
      
      public <T extends Request> Response service(T request, ClientSession session)
            throws ProtocolLayerException, SliceStoreLayerException, ServerException,
            VaultException
      {
         if( doAssert )
         {
            assert(true == false) : "True == False !!!";
         }
         
         if( doException )
         {
            throw new ServerRequestException("I am throwing an exception.");
         }
         
         return new CustomHandlerResponse();
      }
      
   }
   
   @Test
   public void testHandlers()
   {
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      
      SliceServerApplication server = new SliceServerApplication(configuration);
      
      assertEquals("Invalid number of registered handlers", 19, server.getHandlerCount());
      
      assertTrue(server.isRequestHandled(PasswordAuthenticationRequest.class));
      assertTrue(server.isRequestHandled(CreateStoreRequest.class));
      assertTrue(server.isRequestHandled(RemoveStoreRequest.class));
      assertTrue(server.isRequestHandled(VaultBindRequest.class));
      assertTrue(server.isRequestHandled(BeginSessionRequest.class));
      assertTrue(server.isRequestHandled(EndSessionRequest.class));
      assertTrue(server.isRequestHandled(BeginTransactionRequest.class));
      assertTrue(server.isRequestHandled(CommitTransactionRequest.class));
      assertTrue(server.isRequestHandled(RollbackTransactionRequest.class));
      assertTrue(server.isRequestHandled(ExistsRequest.class));
      assertTrue(server.isRequestHandled(ListBeginRequest.class));
      assertTrue(server.isRequestHandled(ListContinueRequest.class));
      assertTrue(server.isRequestHandled(ListInProgressRequest.class));
      assertTrue(server.isRequestHandled(ListStopRequest.class));
      assertTrue(server.isRequestHandled(MultipleReadRequest.class));
      assertTrue(server.isRequestHandled(MultipleWriteRequest.class));
      assertTrue(server.isRequestHandled(MultipleRemoveRequest.class));
      assertTrue(server.isRequestHandled(NoopRequest.class));
      assertTrue(server.isRequestHandled(IntegrityVerificationRequest.class));
   }
   
   @Test 
   public void testServiceWithAssertError() throws Exception
   {

      
      try
      {
         assert(true == false) : "Assertions are enabled";
         
         throw new Exception("Assertions must be enabled to run this test");
      } catch(AssertionError ex)
      {
      }
      
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      
      SliceServerApplication server = new SliceServerApplication(configuration);
     
      server.addHandler(new CustomHandler(true, false, false));
      Request request = new CustomHandlerRequest();      
      Response response = server.service(request, null);
      assertTrue(response.getExceptionFlag());
      assertTrue(response.getException().getMessage().contains("assertion"));
   }
   
   @Test
   public void testGetHandler()
   {
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      SliceServerApplication server = new SliceServerApplication(configuration);
      
      assertTrue(server.getHandler(CustomHandlerRequest.class) == null);
      
      RequestHandler handler = new CustomHandler(false, false, false);
      server.addHandler(handler);
      assertEquals(handler, server.getHandler(CustomHandlerRequest.class));
   }
   
   @Test
   public void testRemoveHandler()
   {
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      SliceServerApplication server = new SliceServerApplication(configuration);
      
      assertTrue(server.getHandler(CustomHandlerRequest.class) == null);
      
      RequestHandler handler = new CustomHandler(false, false, false);
      server.addHandler(handler);
      assertEquals(handler, server.getHandler(CustomHandlerRequest.class));
      
      server.removeHandler(handler);
      assertTrue(server.getHandler(CustomHandlerRequest.class) == null);
   }
   
   @Test
   public void testServiceWithoutError()
   {
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      SliceServerApplication server = new SliceServerApplication(configuration);
      
      RequestHandler handler = new CustomHandler(false, false, false);
      server.addHandler(handler);
      
      ClientSession session = new ClientSession();
      CustomHandlerRequest request = new CustomHandlerRequest();
      
      Response response = server.service(request, session);
      assertTrue(!response.getExceptionFlag());
   }
   
   @Test
   public void testServiceWithException()
   {
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      SliceServerApplication server = new SliceServerApplication(configuration);
      
      RequestHandler handler = new CustomHandler(false, true, false);
      server.addHandler(handler);
      
      ClientSession session = new ClientSession();
      CustomHandlerRequest request = new CustomHandlerRequest();
      
      Response response = server.service(request, session);
      assertTrue(response.getExceptionFlag());
   }
   
   @Test
   public void testServiceWithAuthenticationError()
   {
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      SliceServerApplication server = new SliceServerApplication(configuration);
      
      RequestHandler handler = new CustomHandler(false, false, true);
      server.addHandler(handler);
      
      ClientSession session = new ClientSession();
      CustomHandlerRequest request = new CustomHandlerRequest();
      
      Response response = server.service(request, session);
      assertTrue(response.getExceptionFlag());
      assertEquals(UnauthorizedRequestException.class, response.getException().getClass());
   }
   
   @Test
   public void testServiceWithAuthenticationSuccess()
   {
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      SliceServerApplication server = new SliceServerApplication(configuration);
      
      RequestHandler handler = new CustomHandler(false, false, true);
      server.addHandler(handler);
      
      ClientSession session = new ClientSession();
      session.put(ClientSession.AUTHENTICATED, true);
      CustomHandlerRequest request = new CustomHandlerRequest();
      
      Response response = server.service(request, session);
      assertTrue(!response.getExceptionFlag());
   }
   
   @Test
   public void testServiceWithoutHandler()
   {
      SliceServerConfiguration configuration = new SliceServerConfiguration();
      configuration.setAuthenticationFacilitatorConfiguration(new JAASConfiguration());
      SliceServerApplication server = new SliceServerApplication(configuration);
      
      assertTrue(server.getHandler(CustomHandlerRequest.class) == null);
      
      ClientSession session = new ClientSession();
      CustomHandlerRequest request = new CustomHandlerRequest();

      Response response = server.service(request, session);
      assertTrue(response.getExceptionFlag());
      assertEquals(UnrecognizedRequestException.class, response.getException().getClass());
   }
}


