
package org.cleversafe.storage.ss.handlers;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.UUID;

import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.BeginSessionRequest;
import org.cleversafe.layer.protocol.CreateStoreRequest;
import org.cleversafe.layer.protocol.ErrorResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.exceptions.ProtocolLayerException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;
import org.cleversafe.server.exceptions.ServerException;
import org.cleversafe.server.exceptions.UnauthorizedRequestException;
import org.cleversafe.storage.ss.SliceServerApplication;
import org.cleversafe.storage.ss.SliceServerConfiguration;
import org.cleversafe.storage.ss.configuration.ConfigurationLoader;
import org.cleversafe.storage.ss.configuration.XMLConfigurationLoader;
import org.cleversafe.test.BaseTest;
import org.cleversafe.test.TestException;
import org.cleversafe.util.UUIDGen;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.VaultPermission;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public abstract class HandlerTestBase extends BaseTest
{
   public static String USERNAME = "handlertest";
   public static int SLICE_SIZE = 512;
   public static String STORE_TYPE = "block";

   protected RequestHandler handler = null;
   protected SliceServerApplication sliceServerApp = null;

   @Before
   public void setUp() throws Exception
   {
      clean();
      createServerApplication();
      this.handler = this.sliceServerApp.getHandler(createRequest().getClass());
   }

   public void createServerApplication() throws ServerConfigurationLoadException, TestException
   {
      // Create Server Application
      String serverConfiguration = System.getProperty("org.cleversafe.storage.ss.xml.configuration");
      Properties properties = new Properties();
      properties.setProperty("DAEMON.ID", "1");

      ConfigurationLoader configLoader = new XMLConfigurationLoader(serverConfiguration);
      SliceServerConfiguration configuration = configLoader.getConfiguration(properties);

      this.sliceServerApp = new SliceServerApplication(configuration);
   }

   // Creates an initialized request for the handler under test to process
   protected abstract Request createRequest();

   @After
   public void tearDown() throws Exception
   {
      clean();
      this.handler = null;
   }

   public ClientSession createClientSession()
   {
      ClientSession session = new ClientSession();
      return session;
   }

   public ClientSession authenticateSession(ClientSession session)
   {
      UUID accountId = UUIDGen.getUUIDFromUsername(USERNAME);

      session.put(ClientSession.GRID_ACCOUNT_UUID, accountId);
      session.put(ClientSession.AUTHENTICATED, true);
      session.put(ClientSession.APPLICATION_TYPE, ApplicationType.TYPE_AUTHENTICATED);

      return session;
   }

   public ClientSession bindVault(ClientSession session, UUID vaultId)
   {
      session.put(ClientSession.VAULT_UUID, vaultId);
      session.put(ClientSession.VAULT_PERMISSION, VaultPermission.getFullPermissions(vaultId));
      session.put(ClientSession.LIST_PERMISSION, true);
      session.put(ClientSession.READ_PERMISSION, true);
      session.put(ClientSession.WRITE_PERMISSION, true);

      return session;
   }
   
   public ClientSession unAuthenticateSession(ClientSession session)
   {
      session.remove(ClientSession.GRID_ACCOUNT_UUID);
      session.remove(ClientSession.AUTHENTICATED);
      session.remove(ClientSession.APPLICATION_TYPE);
      return session;
   }

   public ClientSession addSliceStore(ClientSession session) throws SliceStoreExistsException,
         SliceStoreIOException, Exception
   {
      return addSliceStore(session, UUID.randomUUID());
   }
   
   public ClientSession addSliceStore(ClientSession session, UUID vaultId) throws SliceStoreExistsException,
         SliceStoreIOException, Exception
   {      
      // Create the Store using a CreateStoreRequest
      ClientSession tempSession = this.createClientSession();
      tempSession = this.authenticateSession(tempSession);

      Request request = this.makeCreateStoreRequest(vaultId);
      
      Response response = this.sliceServerApp.service(request, tempSession);

      assertEquals(false, response.getExceptionFlag());
      
      this.bindVault(tempSession, vaultId);
      
      request = new BeginSessionRequest();
      
      response = this.sliceServerApp.service(request, tempSession);
      
      assertEquals(false, response.getExceptionFlag());
      
      SliceStore sliceStore = getSliceStore(tempSession);
      sliceStore.endSession();
      
      // Save store in the session
      session.put(ClientSession.SLICE_STORE, sliceStore);
      

      VaultPermission permission = VaultPermission.getFullPermissions(vaultId);

      // Store vault permission and vault identifier in session
      session.put(ClientSession.VAULT_PERMISSION, permission);
      session.put(ClientSession.VAULT_UUID, vaultId);

      return session;
   }
   
   protected CreateStoreRequest makeCreateStoreRequest(UUID vaultId) throws IOException, Exception
   {
      Map<String, String> options = new HashMap<String, String>();
      
      CreateStoreRequest request = new CreateStoreRequest(
                                 "block",
                                 SLICE_SIZE,
                                 -1 /* max store size */,
                                 STORE_TYPE,
                                 vaultId,
                                 createACL(vaultId),
                                 loadVaultDescriptor(),
                                 options);
      
      return request;
   }
   
   private byte[] loadVaultDescriptor() throws IOException
   {
      String vaultDescriptorPath = System.getProperty("org.cleversafe.storage.ss.handlers.HandlerTestBase.vaultDescriptor");
      File vdFile = new File(vaultDescriptorPath);
      FileInputStream in = new FileInputStream(vdFile);
      
      byte[] vdBytes = new byte[(int)vdFile.length()];
      in.read(vdBytes);
      
      return vdBytes;
   }

   public SliceStore getSliceStore(ClientSession session)
   {
      return (SliceStore) session.get(ClientSession.SLICE_STORE);
   }

   public ClientSession addWritePermission(ClientSession session)
   {
      session.put(ClientSession.WRITE_PERMISSION, true);
      return session;
   }

   public ClientSession addListPermission(ClientSession session)
   {
      session.put(ClientSession.LIST_PERMISSION, true);
      return session;
   }

   public ClientSession addReadPermission(ClientSession session)
   {
      session.put(ClientSession.READ_PERMISSION, true);
      return session;
   }

   public ClientSession clearAllPermissions(ClientSession session)
   {
      session.remove(ClientSession.WRITE_PERMISSION);
      session.remove(ClientSession.READ_PERMISSION);
      session.remove(ClientSession.LIST_PERMISSION);
      return session;
   }

   public List<SliceName> generateSliceNames(int offset, int count)
   {
      List<SliceName> sliceNames = new ArrayList<SliceName>(count);
      for (int itr = offset; itr < count + offset; itr++)
      {
         sliceNames.add(new SliceName(Integer.toString(itr), 0));
      }
      return sliceNames;
   }

   protected static byte[] generatedRandomData(int length)
   {
      Random generator = new Random();
      byte[] data = new byte[length];
      generator.nextBytes(data);
      return data;
   }

   public List<DataSlice> createDataSlices(List<SliceName> sliceNames)
   {
      List<DataSlice> dataSlices = new ArrayList<DataSlice>(sliceNames.size());
      for (SliceName sliceName : sliceNames)
      {
         byte[] data = generatedRandomData(SLICE_SIZE);
         dataSlices.add(new DataSlice(sliceName, 0, data));
      }
      return dataSlices;
   }

   public List<DataSlice> writeSlices(SliceStore ss, List<DataSlice> dataSlices)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      SliceStoreTransaction transaction = ss.createTransaction(0);
      ss.beginTransaction(transaction);
      ss.write(dataSlices);
      ss.commitTransaction(transaction);

      return dataSlices;
   }

   public boolean slicesExist(SliceStore ss, List<SliceName> sliceNames)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreNotFoundException
   {
      boolean retVal = true;

      for (SliceName sliceName : sliceNames)
      {
         boolean exists = ss.exists(sliceName);
         if (exists == false)
         {
            retVal = false;
            break;
         }
      }

      return retVal;
   }
   
   public boolean slicesDontExist(SliceStore ss, List<SliceName> sliceNames)
      throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException
   {
      boolean retVal = true;
      
      for (SliceName sliceName : sliceNames)
      {
         boolean exists = ss.exists(sliceName);
         if (exists == true)
         {
            retVal = false;
            break;
         }
      }
      
      return retVal;
   }

   private VaultACL createACL(UUID vaultId) throws Exception
   {
      String vaultFileName = getOutputDirectory().getPath() + vaultId.toString() + ".der";
      FileOutputStream out = new FileOutputStream(vaultFileName);

      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername(USERNAME);
      credentials.setPassword("password");

      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize(512);
      KeyPair keypair = keygen.generateKeyPair();

      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();

      VaultACLFactory fact = new VaultACLFactory();
      fact.create(out, vaultId, credentials, keypair.getPublic(), keyInfoList);
      out.close();

      FileInputStream in = new FileInputStream(vaultFileName);

      VaultACL vaultACL = fact.getInstance(in);
      in.close();

      return vaultACL;
   }

   @Test
   public void testGetAllowedApplications()
   {
      // Ensure that handler requires authentication
      assertTrue(this.handler.getAllowedApplications().contains(ApplicationType.TYPE_AUTHENTICATED));
   }

   @Test
   public final void testGetRequestClass()
   {
      // Ensure the handler supports the right request type, and was registered properly with the server
      assertEquals(this.handler.getRequestClass(), createRequest().getClass());
   }

   public final void checkRequiresSliceStore() throws ProtocolLayerException,
         SliceStoreLayerException, ServerException, VaultException
   {
      ClientSession session = createClientSession();
      authenticateSession(session);

      this.handler.service(createRequest(), session);
   }

   public final void checkRequiresPermission() throws Exception
   {
      ClientSession session = createClientSession();
      authenticateSession(session);
      addSliceStore(session);

      this.handler.service(createRequest(), session);
   }

   @Test
   public final void testRequiresAuthentication() throws Exception
   {
      ClientSession session = createClientSession();

      Response response = this.sliceServerApp.service(createRequest(), session);
      assertTrue(response instanceof ErrorResponse);
      assertTrue(response.getExceptionFlag());
      assertTrue(response.getException() instanceof UnauthorizedRequestException);
   }

}
