
package org.cleversafe.storage.ss.handlers.slices;

import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.List;

import org.cleversafe.codec.Codec;
import org.cleversafe.codec.Encoder;
import org.cleversafe.codec.exceptions.CodecEncodeException;
import org.cleversafe.codec.exceptions.CodecInvalidDataFormatException;
import org.cleversafe.codec.exceptions.CodecInvalidParametersException;
import org.cleversafe.codec.exceptions.CodecNotInitializedException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.protocol.IntegrityVerificationRequest;
import org.cleversafe.layer.protocol.IntegrityVerificationResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.exceptions.ProtocolLayerException;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.exceptions.IllegalSessionStateException;
import org.cleversafe.server.exceptions.ServerException;
import org.cleversafe.server.exceptions.UnauthorizedRequestException;
import org.cleversafe.storage.ss.handlers.HandlerTestBase;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.junit.Test;

public class IntegrityVerificationHandlerTest extends HandlerTestBase
{
   public static String sliceName = "0";
   public static long transactionId = 0;

   // Creates a request to be used by the handler
   @Override
   protected Request createRequest()
   {
      SliceInfo sliceInfo = new SliceInfo(new SourceName(sliceName), transactionId);

      List<SliceInfo> sliceInfoList = new ArrayList<SliceInfo>();
      sliceInfoList.add(sliceInfo);

      return new IntegrityVerificationRequest(sliceInfoList);
   }

   @Test(expected = IllegalSessionStateException.class)
   public final void testRequiresSliceStore() throws ProtocolLayerException,
         SliceStoreLayerException, ServerException, VaultException
   {
      checkRequiresSliceStore();
   }

   @Test(expected = UnauthorizedRequestException.class)
   public final void testRequiresListPermission() throws SliceStoreExistsException,
         SliceStoreIOException, Exception
   {
      checkRequiresPermission();
   }

   @Test
   public final void testService() throws Exception
   {
      ClientSession session = createClientSession();
      session = authenticateSession(session);
      session = addSliceStore(session);
      session = addListPermission(session);

      // Begin session on slice store
      getSliceStore(session).startSession();

      // Write the Slice
      byte[] data = createEncodedData(getSliceStore(session), 256);
      DataSlice dataSlice = new DataSlice(new SliceName(sliceName, 0), transactionId, data);
      List<DataSlice> dataSlices = new ArrayList<DataSlice>();
      dataSlices.add(dataSlice);
      this.writeSlices(getSliceStore(session), dataSlices);

      // Verify slices exist      
      List<SliceName> sliceNames = new ArrayList<SliceName>();
      sliceNames.add(new SliceName(sliceName, 0));
      assertEquals(true, slicesExist(getSliceStore(session), sliceNames));

      // Do Integrity Verification
      Response response = this.handler.service(createRequest(), session);
      assertEquals(false, response.getExceptionFlag());

      IntegrityVerificationResponse ivResponse = (IntegrityVerificationResponse) response;
      assertEquals(0, ivResponse.getCorruptedSlices().size());

      // End session on slice store
      getSliceStore(session).endSession();
   }

   private byte[] createEncodedData(SliceStore ss, int length) throws VaultDescriptorException,
         SliceStoreIOException, VaultIOException, VaultSecurityException,
         CodecEncodeException, CodecInvalidDataFormatException, CodecNotInitializedException,
         CodecInvalidParametersException, SliceStoreNotFoundException
   {
      byte[] data = generatedRandomData(length);

      List<Codec> codecs;
      try
      {
         codecs = ss.getVaultDescriptor().getSliceCodecs();
      }
      catch (VaultACLException e)
      {
         throw new VaultSecurityException("Error accessing vault ACL", e);
      }
      for (Codec codec : codecs)
      {
         Encoder encoder = codec.getEncoder();
         encoder.reset(new SourceName(sliceName), transactionId);
         data = encoder.finish(data);
      }

      return data;
   }

}
