
package org.cleversafe.storage.ss.handlers.slices;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.util.List;

import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.ListContinueRequest;
import org.cleversafe.layer.protocol.ListContinueResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.exceptions.ProtocolLayerException;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.exceptions.IllegalSessionStateException;
import org.cleversafe.server.exceptions.ServerException;
import org.cleversafe.server.exceptions.UnauthorizedRequestException;
import org.cleversafe.storage.ss.handlers.HandlerTestBase;
import org.cleversafe.vault.exceptions.VaultException;
import org.junit.Test;

public class ListContinueHandlerTest extends HandlerTestBase
{
   @Test(expected = IllegalSessionStateException.class)
   public final void testRequiresSliceStore() throws ProtocolLayerException,
         SliceStoreLayerException, ServerException, VaultException
   {
      checkRequiresSliceStore();
   }

   @Test(expected = UnauthorizedRequestException.class)
   public final void testRequiresListPermission() throws Exception
   {
      checkRequiresPermission();
   }

   @Test
   public final void testEmptySliceStore() throws Exception
   {
      ClientSession session = createClientSession();
      authenticateSession(session);
      addSliceStore(session);
      addListPermission(session);

      SliceStore sliceStore = getSliceStore(session);

      sliceStore.startSession();

      sliceStore.listBegin();

      Response response = this.handler.service(createRequest(), session);

      sliceStore.endSession();

      assertFalse(response.getExceptionFlag());

      assertTrue(response instanceof ListContinueResponse);
      ListContinueResponse listContinueResponse = (ListContinueResponse) response;

      assertTrue(listContinueResponse.getSlices().size() == 0);
   }

   @Test
   public final void testSmallSliceStore() throws Exception
   {
      ClientSession session = createClientSession();
      authenticateSession(session);
      addSliceStore(session);
      addListPermission(session);

      SliceStore sliceStore = getSliceStore(session);

      sliceStore.startSession();

      // write 50,000 slices
      int MAX_SLICES = 50000;
      List<SliceName> inputSliceNames = generateSliceNames(0, MAX_SLICES);
      List<DataSlice> inputDataSlices = createDataSlices(inputSliceNames);
      writeSlices(sliceStore, inputDataSlices);

      sliceStore.listBegin();

      Response response = this.handler.service(createRequest(), session);
      assertFalse(response.getExceptionFlag());

      sliceStore.endSession();

      assertFalse(response.getExceptionFlag());

      assertTrue(response instanceof ListContinueResponse);
      ListContinueResponse listContinueResponse = (ListContinueResponse) response;

      List<SliceInfo> returnedSliceInfos = listContinueResponse.getSlices();
      assertTrue(returnedSliceInfos.size() == MAX_SLICES);

      for (int i = 0; i < MAX_SLICES; ++i)
      {
         assertEquals(returnedSliceInfos.get(i).getSourceName(),
               inputDataSlices.get(i).getSliceName().getSourceName());
      }
   }

   @Override
   protected Request createRequest()
   {
      return new ListContinueRequest();
   }

}
