/*
 * Copyright (c) 1999-2005 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.analysis;

import java.util.Collection;
import java.util.Iterator;

import antlr.ASTFactory;
import antlr.RecognitionException;
import antlr.collections.AST;

import com.evelopers.unimod.analysis.executors.FormulaGenerator;
import com.evelopers.unimod.analysis.executors.LongSet;
import com.evelopers.unimod.analysis.executors.Simplifier;
import com.evelopers.unimod.analysis.executors.Substitution;
import com.evelopers.unimod.analysis.executors.SubstitutionBuilder;
import com.evelopers.unimod.parser.ExprTokenTypes;

public class FormulaResolver {
    final private AST formula;
    final private TypeInfoProvider provider;
    private LongSet trueSubstitutions;
    private LongSet falseSubsisutions;
    private Substitution substitution;
    private FormulaCalculator calculator;

    public FormulaResolver(AST formula, TypeInfoProvider provider) {
        this.formula = formula;
        this.provider = provider;
        this.calculator = new FormulaCalculator();
    }

    /**
     * @deprecated Use {@link #FormulaResolver(AST, TypeInfoProvider)} with
     * correct TypeInfoProvider
     */
    public FormulaResolver(AST formula) throws RecognitionException {
        this(formula, OldTypeInfoProviderBuilder.buildProvider(formula));
    }

    public Substitution getSubstitution() throws RecognitionException {
        if (substitution == null) {
            Gatherer gatherer = new Gatherer(provider);
            gatherer.gather(formula);        
            SubstitutionBuilder builder = gatherer.getBuilder();
            substitution = builder.buildSubstitution();
        }
        return substitution;
    }
    
    public LongSet getSubstitutions(boolean value) throws RecognitionException {
        if (value) {            
            if (trueSubstitutions == null) {
                trueSubstitutions = findSubstitutions(value);
            }
            return trueSubstitutions;
        } else {
            if (falseSubsisutions == null) {
                falseSubsisutions = findSubstitutions(value);
            }
            return falseSubsisutions;
        }
    }

    /**
     * Turns formula to DNF and converts it to String
     * 
     * @param value if true then simplified formula is returned otherwise neglection of
     * simplified formula is returned. 
     * @return string presentation of simplified DNF notation.
     * @throws RecognitionException
     */
    public String getSimplifiedDNF(boolean value) throws RecognitionException {
        LongSet substitutions = getSubstitutions(value);
        if (substitutions.isEmpty()) {
            return "false";
        }
        Simplifier simplifier = new Simplifier(substitution, substitutions);
        simplifier.simplify();
        substitutions = simplifier.getSubstitutions();
        if (substitutions.isEmpty()) {
            return "true";
        }
        long[] mask = simplifier.getMask();
        FormulaGenerator generator = new FormulaGenerator(
                provider, substitution, substitutions, mask);
        return generator.generateExpression();
    }
    
    public boolean isTautology() throws RecognitionException {
        return getSubstitutions(false).isEmpty();
    }

    public boolean isUnsatisfiable() throws RecognitionException {
        return getSubstitutions(true).isEmpty();
    }
    
    public static FormulaResolver createConjunction(AST a, AST b, TypeInfoProvider provider) {
        AST conjunction = and(a, b);
        return new FormulaResolver(conjunction, provider);
    }
    
    /**  
     * @deprecated Use {@link #createConjunction(AST, AST, TypeInfoProvider)} 
     */
    public static FormulaResolver createConjunction(AST a, AST b) throws RecognitionException {
        AST conjunction = and(a, b);
        return new FormulaResolver(conjunction);
    }
    
    public static FormulaResolver createDisjunction(Collection formulas, TypeInfoProvider provider) {
        AST disjunction = or(formulas);
        return new FormulaResolver(disjunction, provider);
    }
    
    /**
     * @deprecated Use {@link #createDisjunction(Collection, TypeInfoProvider)}
     */
    public static FormulaResolver createDisjunction(Collection formulas) throws RecognitionException {
        AST disjunction = or(formulas);
        return new FormulaResolver(disjunction);
    }
    
    /**
     * Returns conjunction of formula <code>a</code> and 
     * formula <code>b</code>.
     * @return conjunction of <code>a</code> and <code>b</code>.
     */
    public static AST and(AST a, AST b) {
        ASTFactory astFactory = new ASTFactory();
        AST parent = astFactory.create(ExprTokenTypes.AND, "&&");
        parent.addChild(astFactory.dupTree(a));
        parent.addChild(astFactory.dupTree(b));
        return parent;
    }

    /**
     * Returns disjunction of <code>formulas</code>.
     * @param formulas collection of formulas to disjunct
     * @return disjunction of <code>formulas</code>.
     */
    public static AST or(Collection formulas) {
        ASTFactory astFactory = new ASTFactory();
        AST parent = astFactory.create(ExprTokenTypes.OR, "||");
        for (Iterator i = formulas.iterator(); i.hasNext();) {
            AST ast = (AST) i.next();
            parent.addChild(astFactory.dupTree(ast));
        }
        return parent;
    }

    private LongSet findSubstitutions(boolean formulaValue) throws RecognitionException {
        getSubstitution();
        LongSet substitutions = new LongSet();
        for (substitution.reset();
             substitution.hasMoreSubstitutions();
             substitution.nextSubstitution()) {
            boolean value = calculator.calculate(formula, substitution);
            if (formulaValue == value) {
                substitutions.add(substitution.getSubstitution());
            }
        }
        return substitutions;
    }
    
}
