/*
 * Copyright (c) 1999-2005 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.analysis.executors;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import com.evelopers.unimod.analysis.TypeInfo;
import com.evelopers.unimod.parser.ExprParser;
import com.evelopers.unimod.parser.ExprTokenTypes;

/**
 * @uml.dependency   supplier="com.evelopers.unimod.analysis.executors.RangePredicateExecutor" stereotypes="Basic::Create"
 * @uml.dependency   supplier="com.evelopers.unimod.analysis.executors.Neighborhood"
 */
class RangePredicateExecutorBuilder implements PredicateExecutorBuilder {
    private TypeInfo typeInfo;
    private Map usedNeighborhoods = new HashMap();

    /**
     * @param variableType one of 
     * <ul>
     *   <li>Byte.class</li>
     *   <li>Short.class</li>
     *   <li>Integer.class</li>
     *   <li>Long.class</li>
     *   <li>Character.class</li>
     * </ul>
     */
    public RangePredicateExecutorBuilder(TypeInfo typeInfo) {
        this.typeInfo = typeInfo;
    }

    public void addPredicate(int predicateType, Object value) {
        if (! typeInfo.getValueClass().isAssignableFrom(value.getClass())) {
            throw new IllegalArgumentException("Value ["+value+"] is not of type [" + typeInfo.getValueClass() + "]");
        }
        
        Neighborhood neighborhood = (Neighborhood) usedNeighborhoods.get(value);
        if (neighborhood == null) {
            Comparable point = (Comparable) value;
            neighborhood = new Neighborhood(point);
            usedNeighborhoods.put(point, neighborhood);
        }
        
        switch (predicateType) {
            case ExprTokenTypes.EQUAL:
            case ExprTokenTypes.NEQUAL:
                neighborhood.left = false;
                neighborhood.right = false;
                break;
            case ExprTokenTypes.GE:
            case ExprTokenTypes.LT:
                neighborhood.left = false;
                break;    
            case ExprTokenTypes.LE:
            case ExprTokenTypes.GT:
                neighborhood.right = false;
                break;
            default:
                throw new IllegalArgumentException("Bad predicate type: " + 
                        ExprParser._tokenNames[predicateType]);
        }
    }
    
    public PredicateExecutor createPredicateExecutor() {
        Neighborhood[] neighborhoods = (Neighborhood[]) usedNeighborhoods.
                values().toArray(new Neighborhood[usedNeighborhoods.values().size()]);
        Arrays.sort(neighborhoods);
        List intervals = new LinkedList();
        for (int i = 0; i < neighborhoods.length; i ++) {
    
            if (i > 0) {
                Interval interval = new Interval(
                        neighborhoods[i - 1].point, neighborhoods[i - 1].right, 
                        neighborhoods[i].point, neighborhoods[i].left);
                intervals.add(interval);
            } else {
                Interval interval = new Interval(
                        typeInfo.getMin(), false, 
                        neighborhoods[i].point, neighborhoods[i].left);
                intervals.add(interval);
            }
            
            if (! neighborhoods[i].left && ! neighborhoods[i].right) {
                Interval interval = new Interval(
                        neighborhoods[i].point, true, 
                        neighborhoods[i].point, true);
                intervals.add(interval);
            }
            
            if (i == neighborhoods.length - 1) {
                Interval interval = new Interval(
                        neighborhoods[i].point, neighborhoods[i].right, 
                        typeInfo.getMax(), false);
                intervals.add(interval);
            }
        }

        filterIntervals(intervals);
        
        return new RangePredicateExecutor((Interval[]) intervals.toArray(
                new Interval[intervals.size()]));
    }

    protected void filterIntervals(List intervals) {
        /* Does nothing by default */
    }
    
    protected TypeInfo getTypeInfo() {
        return typeInfo;
    }
}
