/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import java.util.Set;

import antlr.collections.AST;

import com.evelopers.unimod.core.ModelElement;

/**
 * <p>
 * Contains guard condition. Parses given logical expression into
 * Abstract Syntax Tree for futher interpreting.
 * </p>
 * <p>
 * Guard condition expression must sutisfy the following gramma:
 * <pre><code>
 *	S  -> S '||' I1
 *	S  -> I1
 *	I1 -> I1 '&&' I2
 *	I1 -> I2
 *	I2 -> '!' I3
 *	I2 -> I3
 *	I3 -> '(' S ')'
 *	I3 -> I4
 *	I4 -> I5 rel I5
 *	I4 -> ident_bool
 *	I4 -> const_bool
 *	I5 -> ident_number
 *	I5 -> const_number
 * </code></pre>
 * Where <b>rel</b> is one of
 * <code>'&gt;', '&lt;', '&gt;=', '&lt;=', '!=', '=='</code>
 * </p>
 * 
 * @see com.evelopers.unimod.parser.ExprParser
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public class Guard extends ModelElement {
    public static final String EXPR_PROPERTY = "EXPR_PROPERTY";

    /**
     * Guard with empty expression will be interpreted
     * as "true".
     */
    public static final String TRUE_EXPR = "true";
    
    /**
     * True Guard
     */
    public static final Guard TRUE = new Guard(TRUE_EXPR);
    
    /**
     * Expression for else guard
     */
    public static final String ELSE_EXPR = "else";
   
    /**
     * Else Guard
     */
    public static final Guard ELSE = new Guard(ELSE_EXPR);

    /**
     * @clientCardinality 1
     * @supplierCardinality 1 
     */
	private AST ast;
	
	private Set inputActions = null;
	
	private String expr;

    /**
	 * Creates gruard condition based on given logical expression.
	 * Treats <b>null</b> expression as <b>true</b>.
	 *
	 * @param expr guard condition logical expression.
     * Null will be converted to true.
	 */
	protected Guard(String expr) {
		setExpr(expr);
	}

	/**
	 * Returns Abstract Syntax Tree (AST)
	 * 
	 * @return Abstract Syntax Tree
	 */
	public AST getAST() {
		return ast;
	}

    /**
     * Sets abstract syntax tree. This method should be used only
     * by {@link com.evelopers.unimod.compilation.StateMachineCompiler compiler}
     * @param ast abstract syntax tree
     */
	public void setAST(AST ast) {
		this.ast = ast;
	}

	/**
	 * Return guard condition logical expression.
	 * 
	 * @return guard condition logical expression
	 */
	public String getExpr() {
		return expr;
	}
	
	/**
     * @param inputActions The inputActions to set.
     */
    public void setInputActions(Set inputActions) {
        this.inputActions = inputActions;
    }
	
	/**
	 * Returns set of input actions, that is being used in guard condiiton.
	 * Returns null, if Guard was not yet coimpiled.
	 * 
	 * @return
	 */
	public Set getInputActions() {
	    return inputActions;
	}

	/**
	 * Sets guard condition logical expression
	 * 
	 * @param expr new logical expression to set
	 */
	public void setExpr(String expr) {
        String oldExpr = getExpr();
		if (expr == null) {
			expr = TRUE_EXPR;
		}
		
		this.expr = expr;
		this.ast = null;

        if (! expr.equals(oldExpr)) {
            firePropertyChange(EXPR_PROPERTY, oldExpr, expr);
        }
	}

	/**
	 * Returns guard condition string representation.
	 * 
	 * @return guard condition string representation
	 */
	public String toString() {
		return expr;
	}
	
	/**
	 * Two guards equal if their expr equal
	 * 
	 * @param o guard to compare with this
	 * @return whether guards are equal
	 */
	public boolean equals(Object o) {
		if (!(o instanceof Guard)) {
			return false;
		}
		
		return ((Guard)o).getExpr().equals(expr);
	}
	
	/**
	 * Takes hashcode from expr
	 * 
	 * @return hash code
	 */
	public int hashCode() {
		return expr.hashCode();
	}
}