/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import java.io.*;
import com.evelopers.common.util.helper.StringHelper;
import com.evelopers.unimod.resources.Messages;
import com.evelopers.unimod.runtime.EventProcessorException;

/**
 * Provide facilities to handle implementation cass name in design-time and to
 * instanciate implementation class in run-time
 * 
 * @author vgurov
 * @version $Revision: 1$
 */
class ImplementationHandler implements Serializable {
    /**
     * Implementation class name.
     */
    protected String implName = null;

    protected Class handledType = null;

    /**
     * Implementation object. Exists only during run-time. Inited by getObject()
     * using lazy initialization.
     */
    protected transient Object implObject = null;

    /**
     * Creates object with given name and implementation class.
     * 
     * @param implementationClassName
     * @param handledType
     *            Class of handled object
     */
    protected ImplementationHandler(String implementationClassName,
            Class handledType) {
        this.implName = implementationClassName;
        this.handledType = handledType;
    }

    /**
     * Returns implementation class name
     * 
     * @return implementation class name
     */
    public String getImplName() {
        return implName;
    }

    /**
     * Sets new implementation class name
     * 
     * @param implName
     *            new implementation class name
     */
    public void setImplName(String implName) {
        if (implObject != null) {
            throw new IllegalStateException("Object already instantiated");
        }
        this.implName = implName;
    }

    /**
     * Instantiates target implementation object.
     * 
     * @return Controllable Object
     * @throws EventProcessorException
     */
    public Object getObject() throws EventProcessorException {
        if (implObject == null) {
            synchronized (this) {
                if (implObject == null) {
                    try {
                        Class implementationClass = this.getClass()
                                .getClassLoader().loadClass(implName);

                        if (!handledType
                                .isAssignableFrom(implementationClass)) {
                            throw new EventProcessorException(
                                    Messages
                                            .getMessages()
                                            .getMessage(
                                                    Messages.INCORRECT_OBJECT,
                                                    new Object[] {
                                                            implName,
                                                            handledType
                                                                    .getName() }));
                        }

                        implObject = implementationClass.newInstance();
                    } catch (EventProcessorException e) {
                        throw e;
                    } catch (Exception e) {
                        throw new EventProcessorException(
                                Messages
                                        .getMessages()
                                        .getMessage(
                                                Messages.CREATE_OBJECT_ERROR,
                                                new Object[] {
                                                        implName,
                                                        e.getMessage(),
                                                        StringHelper
                                                                .stackTraceToString(e) }));
                    }
                }
            }
        }

        return implObject;
    }

}