/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import java.util.*;
import com.evelopers.unimod.core.*;
import com.evelopers.unimod.resources.Messages;

/**
 * State abstraction.
 * 
 * @author Vadim Gurov
 * @author Maxim Mazin
 * @version $Revision: 1$
 */
public class State extends ModelElement {

    public static final String STATE_TYPE_PROPERTY = "STATE_TYPE_PROPERTY";

    public static final String OUTGOING_TRANSITIONS_PROPERTY = "OUTGOING_TRANSITIONS_PROPERTY";

    public static final String INCOMING_TRANSITIONS_PROPERTY = "INCOMING_TRANSITIONS_PROPERTY";

    public static final String SUBSTATES_PROPERTY = "SUBSTATES_PROPERTY";

    public static final String SUBMACHINES_PROPERTY = "SUBMACHINES_PROPERTY";

    public static final String ON_ENTER_ACTIONS_PROPERTY = "ON_ENTER_ACTIONS_PROPERTY";

    /***************************************************************************
     * @supplierCardinality 1
     * @clientCardinality 0..
     **************************************************************************/
    protected StateType type;

    /**
     * Superstate. Null for top state.
     * 
     * @supplierCardinality 0..1
     * @clientCardinality 0..*
     * @supplierRole superstate
     * @clientRole substate
     * @undirected
     */
    protected State superstate;

    /**
     * Outgoing transitions.
     * 
     * @associates <{Transition}>
     * @clientCardinality 1
     * @supplierCardinality 0..*
     * @undirected
     * @link aggregation
     * @supplierRole outgoing
     * @clientRole source
     */
    protected List outgoingTransitions = new ArrayList();

    /**
     * Incoming transitions.
     * 
     * @associates <{Transition}>
     * @clientCardinality 1
     * @supplierCardinality 0..*
     * @undirected
     * @link aggregation
     * @supplierRole incoming
     * @clientRole target
     */
    protected List incomingTransitions = new ArrayList();

    /** Substates. Empty for simple state. If not empty then state is composite. */
    protected List substates = new ArrayList();

    /**
     * Included state machines to be executed on-enter if current state is
     * active.
     * 
     * @associates <{StateMachine}>
     * @clientCardinality 0..*
     * @supplierCardinality 0..*
     * @directed
     * @label subMachines
     */
    protected List submachines = new ArrayList();

    /**
     * On-enter actions to be executed on-enter
     * 
     * @associates <{Action}>
     * @supplierCardinality 0..*
     * @label on-enter
     * @link aggregation
     * @directed
     */
    protected List onEnterActions = new ArrayList();

    /**
     * Creates states with given name.
     * 
     * @param name
     *            state name
     */
    protected State(String name, StateType type) {
        super(name);

        if (type == null) {
            throw new IllegalArgumentException("State type can not be null.");
        }

        this.type = type;
    }

    /**
     * Returns superstate. Null for top state.
     * 
     * @return superstate
     */
    public State getSuperstate() {
        return superstate;
    }

    /**
     * Sets superstate
     * 
     * @param superstate
     *            superstate
     */
    public void setSuperstate(State superstate) {
        this.superstate = superstate;

        if (superstate != null) {
            superstate.addSubstate(this);
        }
    }

    /**
     * Returns outgoing transitions.
     * 
     * @return outgoing transitions
     */
    public List getOutgoingTransitions() {
        return outgoingTransitions;
    }

    public List getIncomingTransitions() {
        return incomingTransitions;
    }

    /**
     * Returs filtered list of outgoing transitions. <p/>Usage: <code><pre>
     * 
     *    State state = ...
     *    Event e1 = new Event(&quot;e1&quot;);
     *    List transitions;
     * 
     *    // Transitions trigged on event [e1] with not else guard condition
     *    transitions = state.getFilteredOutgoingTransitions(e1, false);
     *    ...
     *    // Transitions trigged on event [e1] with else guard condition
     *    transitions = state.getFilteredOutgoingTransitions(e1, true);
     *    ...
     *    // Transitions trigged on any event with not else guard condition
     *    transitions = state.getFilteredOutgoingTransitions(Event.ANY, false);
     *    ...
     *    // Transitions trigged on any event with else guard condition
     *    transitions = state.getFilteredOutgoingTransitions(Event.ANY, true);
     *  
     * </pre></code> <p/>If <code>event</code> is not equal to {@link Event#ANY}
     * then transitions trigged on eny event won't be returned.
     * 
     * @param event
     *            event trigging the transitions in result list
     * @param elseGuard
     *            whether transitions in result list have else guard condition
     * @return filtered list of transitions
     */
    public List getFilteredOutgoingTransitions(Event event, boolean elseGuard) {
        List transitions = new LinkedList();
        for (Iterator i = getOutgoingTransitions().iterator(); i.hasNext();) {
            Transition transition = (Transition) i.next();
            Event _event = transition.getEvent();
            boolean _elseGuard = transition.getGuard().equals(Guard.ELSE);
            if (_event.equals(event) && (_elseGuard == elseGuard)) {
                transitions.add(transition);
            }
        }
        return transitions;
    }

    /**
     * Adds outgoing transition. For internal pusposes only, called by
     * Transition.
     * 
     * @param transition
     *            transition to add
     */
    void addOutgoingTransition(Transition transition) {
        if (transition == null) {
            throw new IllegalArgumentException(Messages.getMessages()
                    .getMessage(Messages.TRANSITION_NULL));
        }

        outgoingTransitions.add(transition);

        firePropertyChange(OUTGOING_TRANSITIONS_PROPERTY, null, transition);
    }

    void removeOutgoingTransition(Transition transition) {
        if (transition == null) {
            throw new IllegalArgumentException(Messages.getMessages()
                    .getMessage(Messages.TRANSITION_NULL));
        }

        outgoingTransitions.remove(transition);

        firePropertyChange(OUTGOING_TRANSITIONS_PROPERTY, transition, null);
    }

    /**
     * Adds incoming transition. For internal pusposes only, called by
     * Transition.
     * 
     * @param transition
     *            transition to add
     */
    void addIncomingTransition(Transition transition) {
        if (transition == null) {
            throw new IllegalArgumentException(Messages.getMessages()
                    .getMessage(Messages.TRANSITION_NULL));
        }

        incomingTransitions.add(transition);

        firePropertyChange(INCOMING_TRANSITIONS_PROPERTY, null, transition);
    }

    void removeIncomingTransition(Transition transition) {
        if (transition == null) {
            throw new IllegalArgumentException(Messages.getMessages()
                    .getMessage(Messages.TRANSITION_NULL));
        }

        incomingTransitions.remove(transition);

        firePropertyChange(INCOMING_TRANSITIONS_PROPERTY, transition, null);
    }

    /**
     * Returns state type.
     * 
     * @return state type
     */
    public StateType getType() {
        return type;
    }

    /**
     * Adds substate.
     * 
     * @param substate
     *            state
     */
    public void addSubstate(State substate) {

        substates.add(substate);

        if (substate.getSuperstate() != this) {
            substate.superstate = this;
        }

        firePropertyChange(SUBSTATES_PROPERTY, null, substate);
    }

    /**
     * Removes given substate.
     * 
     * @param substate
     *            state to remove
     * @throws IllegalStateException
     *             if given state is not substate of state is being called.
     */
    public void removeSubstate(State substate) {

        if (substate.getSuperstate() == this) {
            substates.remove(substate);

            substate.setSuperstate(null);

            firePropertyChange(SUBSTATES_PROPERTY, substate, null);
        } else {
            throw new IllegalStateException(
                    "Given state is not substate of state is being called.");
        }

    }

    /**
     * @return
     * <li>Initial substate if <code>State</code> is composite and has
     * initial state;
     * <li><code>null</code> elsewise.
     */
    public State getInitialSubstate() {
        if (isComposite()) {
            // try to find Initial state among substates
            for (Iterator i = substates.iterator(); i.hasNext();) {
                State current = (State) i.next();
                if (current.getType().equals(StateType.INITIAL)) {
                    return current;
                }
            }
        }
        return null;
    }

    /**
     * Returns substates.
     * 
     * @return substates
     */
    public List getSubstates() {
        return substates;
    }

    /**
     * Returns true if state has substates.
     * 
     * @return whether state has substates
     */
    public boolean isComposite() {
        return (substates.size() != 0);
    }

    /**
     * Returns true if state is top state
     * 
     * @return whether state is top state
     */
    public boolean isTop() {
        return getSuperstate() == null;
    }

    /**
     * Returns true if state has superstate. False only for top state.
     * 
     * @return whether state has superstate
     */
    public boolean hasSuperstate() {
        return getSuperstate() != null;
    }

    /**
     * Returns all events that are located on outgoing transitions
     * It's guaranteed that if ay event is present, it will be last in list
     * 
     * @param any include any events or not
     * @return list of events
     */
    public List getEvents(boolean any) {

        Set events = new HashSet();
        Iterator trans = outgoingTransitions.iterator();
        boolean wasAny = false;

        while (trans.hasNext()) {
            Event e = ((Transition) trans.next()).getEvent();

            if (e != null && !e.equals(Event.ANY)) {
                events.add(e);
            }
            
            if (e.equals(Event.ANY)) {
                wasAny = true;
            }
        }

        List e = new ArrayList(events);
        
        if (wasAny && any) {
            e.add(Event.ANY);
        }
        
        return e;
    }
    
    /**
     * Returns list of included state machines
     * 
     * @return list of included state machines
     */
    public List getSubmachines() {
        return submachines;
    }

    /**
     * Adds included state machine
     * 
     * @param sm
     *            state machine to include
     */
    public void addSubmachine(StateMachine sm) {
        submachines.add(sm);

        firePropertyChange(SUBMACHINES_PROPERTY, null, sm);
    }

    /**
     * Removes given included state machine from list of submachines
     * 
     * @param sm
     *            state machine to remove
     */
    public void removeSubmachine(StateMachine sm) {
        submachines.remove(sm);

        firePropertyChange(SUBMACHINES_PROPERTY, sm, null);
    }

    /**
     * Returns list of on-enter actions
     * 
     * @return list of on-enter actions
     */
    public List getOnEnterActions() {
        return onEnterActions;
    }

    /**
     * Adds on-enter action
     * 
     * @param action
     *            action to add
     */
    public void addOnEnterAction(Action action) {
        onEnterActions.add(action);

        firePropertyChange(ON_ENTER_ACTIONS_PROPERTY, null, action);
    }

    /**
     * Removes on-enter action
     * 
     * @param action
     *            on-enter action to remove
     */
    public void removeOnEnterAction(Action action) {
        onEnterActions.remove(action);

        firePropertyChange(ON_ENTER_ACTIONS_PROPERTY, action, null);
    }

    public int hashCode() {
        return name == null ? 0 : name.hashCode();
    }

    public boolean equals(Object o) {
        if (!(o instanceof State)) {
            return false;
        }

        return name != null && name.equals(((State) o).getName());
    }

    /**
     * Accepts given visitor by substates if {@link ModelVisitor#visit}returns
     * true.
     * 
     * @see ModelElement#accept
     */
    public void accept(ModelVisitor v) throws VisitorException {
        if (v.visit(this)) {
            for (Iterator i = substates.iterator(); i.hasNext();) {
                ((State) i.next()).accept(v);
            }
        }
    }

    private static final long serialVersionUID = 4061653607833892876L;
}