/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import com.evelopers.unimod.core.ModelElement;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * Transition abstraction. Has source and target states, trigged event, guard
 * condition and output actions to be executed in a case of this transition
 * chosen as transition to go.
 * </p>
 * <p>
 * Transition has several internal states that the following table illustrates:
 * <pre>
 * +---------------------------------------------------------------------------------+
 * | state | source state set | source attached | target state set | target attached |
 * +---------------------------------------------------------------------------------+
 * |     1 | no               | no              | no               | no              |
 * |     2 | yes              | no              | no               | no              |
 * |     3 | yes              | yes             | no               | no              |
 * |     4 | yes              | yes             | yes              | no              |
 * |     5 | yes              | yes             | yes              | yes             |
 * |     6 | no               | no              | yes              | no              |
 * |     7 | no               | no              | yes              | yes             |
 * |     8 | yes              | no              | yes              | yes             |
 * |     9 | yes              | no              | yes              | no              |
 * +---------------------------------------------------------------------------------+
 * </pre>
 *
 * <p>
 * To set source and target state and attach them, use method {@link Transition#reconnect}.
 * </p>
 * <p>
 * There are 4 different types of Transition because of different types of {@link Event} and {@link Guard}:
 * <table border="1">
 * <tr><td>Type</td><td>{@link Event}</td><td>{@link Guard}</td><td>Semantics</td></tr>
 * <tr><td>1</td><td>E1</td><td>O1.X1</td><td>E1 &amp;&amp; O1.X1</td></tr>
 * <tr><td>2</td><td>{@link Event#ANY}</td><td>O1.X1</td><td>E1 || E2 || .. || Ex &amp;&amp; O1.X1 where E1..Ex are all possible event</td></tr>
 * <tr><td>3</td><td>E1</td><td>{@link Guard#ELSE}</td><td>E1 && !(guard1 || guard2 || ... || guardx) where guard 1..guardx are guard conditions on all outgoing transitions with event E1</td></tr>
 * <tr><td>4</td><td>{@link Event#ANY}</td><td>{@link Guard#ELSE}</td><td>(E1 && !(guard11 || guard12 || ... || guard1x)) || (E2 && !(guard21 || guard22 || ... || guard2x)) || ... || (Ey && !(guardy1 || guardy2 || ... || guardyx))</td></tr>
 * </table>
 * </p>
 * 
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public class Transition extends ModelElement {

    public static final String OUTPUT_ACTIONS_PROPERTY = "OUTPUT_ACTIONS_PROPERTY";
	public static final String EVENT_PROPERTY = "EVENT_PROPERTY";
	public static final String GUARD_PROPERTY = "GUARD_PROPERTY";

	/**
	 * Source state 
	 */
	protected State sourceState = null;
	
	/**
	 * Target state 
	 */
	protected State targetState = null;
	
	/**
	 * Trigger event.
	 * @link aggregation
	 * @supplierCardinality 1
	 * @clientCardinality 1
	 */
	protected Event event = null;
	
	/**
	 * Guard condition
	 * @link aggregation
	 * @supplierCardinality 0..1
	 */
	protected Guard guard = null;

	/**
	 * List of output actions. List here, because sequence of execution makes
	 * sence.
	 *
     * @associates <{Action}>
     * @supplierCardinality 0..* 
     * @link aggregation
     * @directed
	 */
	protected List outputActions = new ArrayList();

	/**
	 * Creates transition. Assumes that if given Conditon is null - it's {@link Guard#TRUE_EXPR}
	 * and if given Event is null - it's {@link Event#NO_EVENT} 
	 *
	 * @param sourceState source state
	 * @param targetState target state
	 * @param guard guard guard
	 * @param event trigged event
	 */
	protected Transition(State sourceState, State targetState,
					  Guard guard, Event event) {
        if (guard == null) {
            guard = Guard.TRUE;
        }
        
        if (event == null) {
            event = Event.NO_EVENT;
        }
        
		this.event = event;
		this.guard = guard;

		reconnect(sourceState, targetState);
	}

	/**
	 * Reconnects transition to new source and target states. 
    	 * <li>Detachs from source and sets new source state
    	 * <li>Detachs from target and sets new target state
    	 * <li>Attaches to new source state
    	 * <li>Attaches to new target state
	 *  
	 * @param newSourceState new source state
	 * @param newTargetState
	 */
	public void reconnect(State newSourceState, State newTargetState) {

		setSourceState(newSourceState);
		setTargetState(newTargetState);
		
		attachSource();
		attachTarget();
	}
	
	/**
	 * Returns source state
	 *
	 * @return source state
	 */
	public State getSourceState() {
		return sourceState;
	}

	/**
	 * Sets new source state. If old source state was not null - removes this transition
	 * from outgoing transitions of source state.
	 *  
	 * @param newSourceState new source state. If null, then transition will only removes itself
	 * from old source state outgoing transitions list.
	 */
	public void setSourceState(State newSourceState) {
		detachSource();		
		sourceState = newSourceState;
	}

	public void attachSource() {
		if (sourceState == null) {
			return;
		}

		sourceState.addOutgoingTransition(this);
	}

	public void detachSource() {
		if (sourceState == null) {
			return;
		}

		sourceState.removeOutgoingTransition(this);
	}

	/**
	 * Returns target state
	 *
	 * @return target state
	 */
	public State getTargetState() {
		return targetState;
	}

	/**
	 * Sets new target state. If old target state was not null - removes this transition
	 * from incoming transitions of target state. 
     *
	 * @param newTargetState new target state. If null, then transition will only removes itlsef
	 * from target state incoming transitions set.
	 */
	public void setTargetState(State newTargetState) {
		detachTarget();		
		targetState = newTargetState;
	}

	public void attachTarget() {
		if (targetState == null) {
			return;
		}

		targetState.addIncomingTransition(this);
	}

	public void detachTarget() {
		if (targetState == null) {
			return;
		}

		targetState.removeIncomingTransition(this);
	}

    /**
     * Returns trigged event
     *
     * @return trigged event
     */
    public Event getEvent() {
        return event;
    }

    public void setEvent(Event newEvent) {
        Event oldEvent = event;
        event = newEvent;

        if (event == null) {
            event = Event.NO_EVENT;
        }

        firePropertyChange(EVENT_PROPERTY, oldEvent, newEvent);
    }

	/**
	 * Returns guard condition
	 *
	 * @return guard condition
	 */
	public Guard getGuard() {
		return guard;
	}

	public void setGuard(Guard guard) {
		Guard oldGuard = this.guard;
		this.guard = guard;
		
		firePropertyChange(GUARD_PROPERTY, oldGuard, guard);
	}

	/**
	 * Adds output action
	 *
	 * @param action action to add
	 */
	public void addOutputAction(Action action) {
		outputActions.add(action);
		
		firePropertyChange(OUTPUT_ACTIONS_PROPERTY, null, action);
	}

	/**
	 * Removes output action
	 * 
	 * @param action action to remove
	 */
	public void removeOutputAction(Action action) {
		outputActions.remove(action);

		firePropertyChange(OUTPUT_ACTIONS_PROPERTY, action, null);
	}	

	/**
	 * Returns output actions
	 *
	 * @return list of output actions
	 */
	public List getOutputActions() {
		return outputActions;
	}

	public boolean equals(Object o) {
		
		if (!(o instanceof Transition)) {
			return false;
		}
		
		Transition t = (Transition)o;
		
		boolean r = t.getSourceState().getName().equals(sourceState.getName()) &&
			t.getTargetState().getName().equals(targetState.getName());
			
		r = event == null ? r : r && event.equals(t.getEvent());
		
		return guard == null ? r : guard.equals(t.getGuard());
	}

	public int hashCode() {
		int h = 0;
		
		h = sourceState == null ? h : sourceState.hashCode();
		h = targetState == null ? h : h ^ targetState.hashCode();
		h = event ==       null ? h : h ^ event.hashCode();
		h = guard ==   null ? h : h ^ guard.hashCode();
		
		return h;
	}

	/**
	 * Returns transition string representation
	 *
	 * @return transition string representation
	 */
	public String toString() {
		return sourceState + "->" + targetState + " " +
			   ((event == null) ? "" : event + "") + 
			   "[" + (guard == null ? "" : guard + "") + "]" + "/" +
			   actions();
	}
	
	private String actions() {
		StringBuffer s = new StringBuffer();
		
		if (outputActions.size() == 0) {
			return "";
		}
		
		for (int i = 0; i < outputActions.size() - 1; i++) {
			s.append(((Action)outputActions.get(i)).getIdentifier()).append(",");
		}
		
		s.append(((Action)outputActions.get(outputActions.size() - 1)).getIdentifier());
	
		return s.toString();
	}
}
