/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.debug.app;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ServerSocket;
import java.net.Socket;

import com.evelopers.common.exception.SystemException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.debug.Params;
import com.evelopers.unimod.debug.protocol.CommandMessage;
import com.evelopers.unimod.debug.protocol.EventMessage;
import com.evelopers.unimod.debug.protocol.Message;
import com.evelopers.unimod.debug.protocol.MessageCoder;
import com.evelopers.unimod.debug.protocol.MessageCoderException;
import com.evelopers.unimod.debug.protocol.ThreadInfo;
import com.evelopers.unimod.debug.protocol.position.Position;
import com.evelopers.unimod.runtime.ControlledObject;
import com.evelopers.unimod.runtime.EventManager;
import com.evelopers.unimod.runtime.EventProvider;
import com.evelopers.unimod.runtime.ModelEngine;
import com.evelopers.unimod.runtime.context.Parameter;
import com.evelopers.unimod.runtime.context.StateMachineContext;
import com.evelopers.unimod.runtime.context.StateMachineContextImpl;

/**
 * @author vgurov
 */
public class AppConnector implements ControlledObject, EventProvider {

	/**
	* @unimod.event.descr debugger connected
	*/
	public static final String E1 = "e1";
    
	/**
	* @unimod.event.descr set break points
	*/
	public static final String E2 = "e2";

	/**
	* @unimod.event.descr clear break point
	*/
	public static final String E3 = "e3";

	/**
	* @unimod.event.descr step
	*/
	public static final String E4 = "e4";

	/**
	* @unimod.event.descr resume
	*/
	public static final String E5 = "e5";

	/**
	* @unimod.event.descr debugger disconnected
	*/
	public static final String E6 = "e6";
	
	/**
	* @unimod.event.descr unknown command received
	*/
	public static final String E7 = "e7";

	/**
	* @unimod.event.descr can't start server socket
	*/
	public static final String E8 = "e8";

	/**
	* @unimod.event.descr upload new model command
	*/
	public static final String E9 = "e9";
	
	private int port;
	private MessageCoder coder; 
    private ServerSocket socket;
    private OutputStream socketWriter;
    private InputStream socketReader;

    private EventManager handler;
    
	AppConnector(int port, MessageCoder coder) {
	    this.port = port;
	    this.coder = coder;
	}
	
	/* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.EventProvider#init(com.evelopers.unimod.runtime.ModelEngine)
     */
    public void init(ModelEngine engine) throws SystemException {
        handler = engine.getEventManager();
        
        try {
            socket = new ServerSocket(port);
        } catch (IOException e) {
            throw new SystemException(e);
        }
    }

    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.EventProvider#dispose()
     */
    public void dispose() {
        if (socketReader == null) {
            return;
        }
        
        try {
            InputStream is = socketReader;
            socketReader = null;
            is.close();
            
            //socketReader.close();
            socket.close();
        } catch (Exception e) {
            // should never occur
            e.printStackTrace();
        } 
    }
    
    
	/**
	*@unimod.action.descr start accepting debugger connection
	*/
	public void z0(StateMachineContext context) throws MessageCoderException, IOException {
        new SocketListenerThread().start();
	}
    
	/**
	*@unimod.action.descr send suspended on breakpoint hit
	*/
	public void z1(StateMachineContext context) throws MessageCoderException, IOException {
		Position b = (Position)context.getEventContext().getParameter(Params.Event.POSITION);
		ThreadInfo ti = (ThreadInfo)context.getEventContext().getParameter(Params.Event.THREAD_INFO);
		
		sendMessage(EventMessage.createSuspendedOnBreakpoint(b, ti));
	}

	/**
	*@unimod.action.descr send suspended on step
	*/
	public void z2(StateMachineContext context) throws MessageCoderException, IOException {
		Position p = (Position)context.getEventContext().getParameter(Params.Event.POSITION);
		ThreadInfo ti = (ThreadInfo)context.getEventContext().getParameter(Params.Event.THREAD_INFO);
		
		sendMessage(EventMessage.createSuspendedOnStep(p, ti));
	}

	/**
	*@unimod.action.descr send resumed
	*/
	public void z3(StateMachineContext context) throws MessageCoderException, IOException {
		ThreadInfo ti = (ThreadInfo)context.getEventContext().getParameter(Params.Event.THREAD_INFO);
		
		sendMessage(EventMessage.createResumed(ti));
	}
	
	/**
	*@unimod.action.descr send unknown command received
	*/
	public void z4(StateMachineContext context) throws MessageCoderException, IOException {
		MessageCoderException e = (MessageCoderException)context.getEventContext().getParameter(Params.Event.MESSAGE_CODER_EXCEPTION);
		
		sendMessage(EventMessage.createUnknownCommand(e.getChainedMessage() + "\n" + e.getChainedTrace()));
	}

	/**
	*@unimod.action.descr new thread created
	*/
	public void z5(StateMachineContext context) throws MessageCoderException, IOException {
		ThreadInfo ti = (ThreadInfo)context.getEventContext().getParameter(Params.Event.THREAD_INFO);
		
		if (ti != null) {
		    sendMessage(EventMessage.createCreated(ti));
		}
	}
	

	/**
	 * @unimod.action.descr send cant update model 
	 */
	public void z6(StateMachineContext context) throws MessageCoderException, IOException {
		ThreadInfo ti = (ThreadInfo)context.getEventContext().getParameter(Params.Event.THREAD_INFO);
		ThreadInfo[] suspenedThreads = (ThreadInfo[])context.getEventContext().getParameter(Params.Event.SUSPENED_THREADS);
		String info = (String)context.getEventContext().getParameter(Params.Event.INFO);
		
	    sendMessage(EventMessage.createCantUpdateModel(ti, info, suspenedThreads));
	}
	
	private void sendMessage(Message m) throws MessageCoderException, IOException {
	    coder.encode(m, socketWriter);
	    socketWriter.flush();
	}
	
	private class SocketListenerThread extends Thread {
	    
	    SocketListenerThread() {
	        super("AppDebugger.SocketListenerThread");
	        // set daemon to true, so this thread will die, even if app
	        // debugger doesn't come to final state, but for some reason, there is
	        // no another non-daemon threads	        
	        setDaemon(true);
	    }
	    
        public void run() {
            // wait for debugger 
            while (true) {
                Socket client = null;
                try {
                    client = socket.accept();

                    socketWriter = client.getOutputStream();
                    socketReader = client.getInputStream();

                    // debugger connected
                    handler.handle(new Event(E1), StateMachineContextImpl
                            .create());

                    while (true) {
                        CommandMessage m = null;

                        try {
                            m = (CommandMessage) coder.decode(socketReader);
                        } catch (MessageCoderException e) {
                            if (e.getParentException() instanceof IOException) {
                                throw (IOException) e.getParentException();
                            } else {
                                // not a communication problem
                                handler.handle(new Event(E7, new Parameter(Params.Event.MESSAGE_CODER_EXCEPTION, e)),
                                               StateMachineContextImpl.create());
                                continue;
                            }
                        }

                        switch (m.getType()) {
                        case CommandMessage.SET_BREAKPOINTS:
                            handler.handle(new Event(E2,
                                    new Parameter(Params.Event.BREAKPOINTS, m
                                            .getBreapoints())),
                                    StateMachineContextImpl.create());
                            break;

                        case CommandMessage.REMOVE_BREAKPOINTS:
                            handler.handle(new Event(E3,
                                    new Parameter(Params.Event.BREAKPOINTS, m
                                            .getBreapoints())),
                                    StateMachineContextImpl.create());
                            break;

                        case CommandMessage.STEP:
                            handler.handle(new Event(E4,
                                    new Parameter(Params.Event.THREAD_INFO, m
                                            .getThreadInfo())),
                                    StateMachineContextImpl.create());
                            break;

                        case CommandMessage.RESUME:
                            handler.handle(new Event(E5,
                                    new Parameter(Params.Event.THREAD_INFO, m
                                            .getThreadInfo())),
                                    StateMachineContextImpl.create());
                            break;
                            
                        case CommandMessage.UPLOAD_NEW_MODEL:
                        	handler.handle(new Event(E9,
                        			new Parameter(Params.Event.NEW_MODEL, m.getNewModel())),
                        			StateMachineContextImpl.create());
                            break;
                        }
                    }

                } catch (IOException e) {
                    // determine if client disconnected or server socker was closed
                    if (socketReader == null) {
                        // server socket was closed
                        return;
                    }

                    // if server socket is opened - debugger disconnected or
                    // some error occured during reading message from client
                    // force client to disconnect, if it still connected
                    if (client != null) {
                        try {
                            client.close();
                        } catch (Exception e1) {
                        }
                    }

                    handler.handle(new Event(E6), StateMachineContextImpl.create());
                }
            }
        }
	    
	}

}
