/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.debug.app;

import java.io.IOException;

import org.apache.commons.logging.LogFactory;
import org.xml.sax.XMLReader;

import com.evelopers.common.exception.CommonException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.debug.ExceptionHandlerImpl;
import com.evelopers.unimod.debug.ThreadConfigManager;
import com.evelopers.unimod.debug.protocol.MessageCoder;
import com.evelopers.unimod.runtime.AbstractEventProcessorListener;
import com.evelopers.unimod.runtime.ControlledObject;
import com.evelopers.unimod.runtime.ControlledObjectsMap;
import com.evelopers.unimod.runtime.ModelEngine;
import com.evelopers.unimod.runtime.StateMachineConfig;
import com.evelopers.unimod.runtime.StateMachinePath;
import com.evelopers.unimod.runtime.context.StateMachineContext;
import com.evelopers.unimod.runtime.interpretation.InterpretationHelper;
import com.evelopers.unimod.runtime.logger.ShortLogger;
import com.evelopers.unimod.transform.xml.XMLToModel;

/**
 * ModelEngine remote debugger. Client-side or application that is being debugged side.
 * To use it, create your ModelEngine, before starting engine, call 
 * new AppDebugger(port, engine.getEventProcessor(), new SomeMessageCoder()) and
 * then call {@link #start()}. After this you may start engine that is being debugged
 * 
 * @author vgurov
 */
public class AppDebugger {

    private AppConnector connector;
    private EventProcessorEventProvider processorListener;
    private ModelEngine engine;
    private ModelEngine debuggedEngine;
    
    /**
     * Create app-side debugger engine
     * 
     * @param port port to listen to incomming debugger connection 
     * @param mustSuspend it true, engine will be suspended till debugger connection 
     * @param coder message coder to use
     * @param debugged model engine that is being debugged
     * @throws CommonException
     * @throws IOException
     */
    public AppDebugger(int port, boolean mustSuspend, MessageCoder coder, ModelEngine debugged) throws CommonException, IOException {
        this(port, mustSuspend, coder, debugged, null);
    }
    
    /**
     * Create app-side debugger engine
     *
     * @param port port to listen to incomming debugger connection
     * @param mustSuspend it true, engine will be suspended till debugger connection
     * @param coder message coder to use
     * @param debugged model engine that is being debugged
     * @throws CommonException
     * @throws IOException
     */
    public AppDebugger(int port, boolean mustSuspend, MessageCoder coder, ModelEngine debugged, XMLReader xmlReader) throws CommonException, IOException {
        this.debuggedEngine = debugged;

        // setup statemachine engine
        // build model
        Model model = XMLToModel.loadAndCompile(getClass().getResourceAsStream("app.xml"), xmlReader);

        // create event providers and controlled objects
        connector = new AppConnector(port, coder);
        final ThreadManager threadManager = new ThreadManager();
        final ModelManager modelManager = new ModelManager();
        processorListener = new EventProcessorEventProvider(threadManager, modelManager, debuggedEngine);
        final BreakpointManager breakpointManager = new BreakpointManager(mustSuspend);

        // create runtime debugger engine
        engine = InterpretationHelper.getInstance().createBuildInModelEngine(
                model,
                new ControlledObjectsMap() {
                    public ControlledObject getControlledObject(String coName) {
                        if (coName.equals("o1")) {
                            return connector;
                        } else if (coName.equals("o2")) {
                            return breakpointManager;
                        } else if (coName.equals("o3")) {
                            return threadManager;
                        } else if (coName.equals("o4")) {
                            return modelManager;
                        }

                        throw new IllegalArgumentException("Unknown controlled object [" + coName + "]");
                    }
                },
                true);

        // set up thread config manager
        threadManager.setThreadConfigManager(
                (ThreadConfigManager)engine.getEventProcessor().getModelStructure().getConfigManager("A3"));
                
        
        // add exception handler
        engine.getEventProcessor().addExceptionHandler(new ExceptionHandlerImpl());

        // stop all event providers when debugger comes to final state
        // BTW, debugger comes to final state, when engine that is being debugged comes to final state :)
        // EventProcessorEventProvider throws e22, when engine that is being debugged comes to final state
        // and app.unimod shows that on e22 we come to final state also
        engine.getEventProcessor().addEventProcessorListener(new AbstractEventProcessorListener() {
            public void stateMachineCameToFinalState(StateMachineContext context, StateMachinePath path, StateMachineConfig config) {
                if (path.isRoot()) {
                    disposeEventProviders();
                }
            }
        });
        
        // add log 
        if (LogFactory.getLog(getClass()).isDebugEnabled()) {
            engine.getEventProcessor().addEventProcessorListener(new ShortLogger(LogFactory.getLog(getClass())));
        }
    }

    public void start() throws CommonException {
        // init event providers  
        connector.init(engine);
        processorListener.init(engine);
        // listern for events inside engine that is being debugged
        debuggedEngine.getEventProcessor().addEventProcessorListener(processorListener);
        
        // send startup event
        engine.getEventManager().handle(new Event("e0"), null);
    }
    
    protected void disposeEventProviders() {
        connector.dispose();
        debuggedEngine.getEventProcessor().removeEventProcessorListener(processorListener);
    }
    
    public void stop() {
        disposeEventProviders();
        
        // also we have to dipose queued event manager
        // because ModelEngine disposes it only if root 
        // state machine comes to final state, but this method stop()
        // may be called even if root state machine not in final state
        engine.getEventManager().dispose();
    }
}
