/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.debug.app;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.evelopers.common.util.helper.StringHelper;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.debug.Params;
import com.evelopers.unimod.debug.protocol.ThreadInfo;
import com.evelopers.unimod.debug.protocol.position.ComeToStatePosition;
import com.evelopers.unimod.debug.protocol.position.EventProcessingPosition;
import com.evelopers.unimod.debug.protocol.position.InputActionExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.OnEnterActionExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.OutputActionExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.SubmachinesExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.TransitionCandidatePosition;
import com.evelopers.unimod.runtime.AbstractEventProcessorListener;
import com.evelopers.unimod.runtime.EventManager;
import com.evelopers.unimod.runtime.EventProvider;
import com.evelopers.unimod.runtime.ModelEngine;
import com.evelopers.unimod.runtime.StateMachineConfig;
import com.evelopers.unimod.runtime.StateMachinePath;
import com.evelopers.unimod.runtime.context.Parameter;
import com.evelopers.unimod.runtime.context.StateMachineContext;
import com.evelopers.unimod.runtime.context.StateMachineContextImpl;
import com.evelopers.unimod.runtime.interpretation.InterpretationEventProcessor;

/**
 * EventProvider for app-side debugger engine.
 * Listens debuggable engine for event processing and notifies about these
 * event debugger engine. Reacts only on events types defined in {@link com.evelopers.unimod.debug.BreakpointPositions}.  
 */
public class EventProcessorEventProvider extends AbstractEventProcessorListener implements EventProvider {

	private static final Log log = LogFactory.getLog(EventProcessorEventProvider.class);
	
	/**
	* @unimod.event.descr some action done during event processing
	*/
	public static final String E21 = "e21";

	/**
	* @unimod.event.descr reach final state of root state machine
	*/
	public static final String E22 = "e22";

	/**
	* @unimod.event.descr can't update model
	*/
	public static final String E23 = "e23";
	
	private EventManager handler;
	
	private ModelEngine debuggedEngine;

	private ThreadManager threadManager; 
	
	private ModelManager modelManager;

	EventProcessorEventProvider(ThreadManager threadManager, ModelManager modelManager, ModelEngine debuggedEngine) {
		this.threadManager = threadManager;
		this.modelManager = modelManager;
    	this.debuggedEngine = debuggedEngine;
	}
	
    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.EventProvider#init(com.evelopers.unimod.runtime.ModelEngine)
     */
    public void init(ModelEngine engine) {
        handler = engine.getEventManager();
    }

    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.EventProvider#dispose()
     */
    public void dispose() {
    }

    /**
     * Suspends event dispatch thread of engine that is being debugged, if
     * ThreadManager wants it, until ThreadManager resumes thread.
     * Must be called at the end of every method of EventProcessorListener.
     */
    protected void suspendThread() {
        ThreadInfo ti = getThreadInfo();
        
        if (threadManager.needSuspend(ti)) {
            synchronized (threadManager.getLock(ti)) {
                try {
                	threadManager.getLock(ti).wait();
                } catch (InterruptedException e) {
                    // should never be catched
                    throw new RuntimeException("Unexpected InterruptedException inside suspendThread(): " + StringHelper.stackTraceToString(e));
                }
            }
        }
    }
    
    protected void updateModelInDebuggedEngine(StateMachinePath path) {
    	// check if debugged engine is executed with interpretator
    	if (!(debuggedEngine != null && debuggedEngine.getEventProcessor() instanceof InterpretationEventProcessor)) {
			log.info("Update model is possible during model interpretation only");
			
	        handler.handleAndWait(
	                new Event(E23, 
	                        new Parameter[]{
	                			new Parameter(Params.Event.INFO, "Update model is possible during model interpretation only"),
	                			new Parameter(Params.Event.THREAD_INFO, getThreadInfo())}),                        	
	                StateMachineContextImpl.create());
	        
	        return;
    	}
    	
    	// try to update model if it's root statemachine and there is new model
    	if (path.isRoot() && modelManager.getNewModel() != null) {
    		ThreadInfo[] ti = threadManager.getSuspenedThreads();
    		
    		// check if there is no suspened threads
    		if (ti != null && ti.length > 0) {
    			log.info("Can't update model, because there're suspended threads");
    			
    	        handler.handleAndWait(
    	                new Event(E23, 
    	                        new Parameter[]{
    	                			new Parameter(Params.Event.INFO, "Can't update model, because there're suspended threads"),
    	                			new Parameter(Params.Event.THREAD_INFO, getThreadInfo()),
    	                			new Parameter(Params.Event.SUSPENED_THREADS, ti)}),                        	
    	                StateMachineContextImpl.create());
    		} else {
    			// update model and reset modelin model manager 
    			log.info("===============");
    			log.info("Set new model");
    			log.info("===============");
    			
    			((InterpretationEventProcessor)debuggedEngine.getEventProcessor()).updateModelStructure(modelManager.getNewModel());
    			modelManager.setNewModel(null);
    		}
    	}
    	
	}
    
    protected ThreadInfo getThreadInfo() {
        return new ThreadInfo(Thread.currentThread().getName(), Thread.currentThread().hashCode());
    }
    
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.AbstractEventProcessorListener#eventProcessingStarted(com.evelopers.unimod.core.stateworks.Event, com.evelopers.unimod.runtime.StateMachinePath, com.evelopers.unimod.runtime.StateMachineConfig)
     */
    public void eventProcessingStarted(StateMachineContext context, Event e, StateMachinePath path,
            StateMachineConfig c) {

    	/*
    	 * update model in engine being debugged if new model version comes
    	 * from debugger. Note, that model update done only when event processing started 
    	 * by root state machine and there is no suspened threads!
    	 */
		updateModelInDebuggedEngine(path);
    	
        handler.handleAndWait(
                new Event(E21, 
                        new Parameter[]{
                        	new Parameter(Params.Event.POSITION, new EventProcessingPosition(context, path, e, c, true)),
                        	new Parameter(Params.Event.THREAD_INFO, getThreadInfo())}),                        	
                StateMachineContextImpl.create());
    
        suspendThread();
    }

	/**
     * Debuggable engine come to final state
     */
    public void stateMachineCameToFinalState(StateMachineContext context, StateMachinePath path,
            StateMachineConfig config) {
        
        if (path.isRoot()) {
            handler.handle(
                    new Event(E22),
                    StateMachineContextImpl.create());
        }
    }
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.AbstractEventProcessorListener#beforeInputActionExecution(com.evelopers.unimod.runtime.StateMachinePath, java.lang.String, java.lang.String)
     */
    public void beforeInputActionExecution(StateMachineContext context, StateMachinePath path,
            String transition, String inputAction) {
        
        handler.handleAndWait(
                new Event(E21, 
                        new Parameter[]{
                        	new Parameter(Params.Event.POSITION, new InputActionExecutionPosition(context, path, transition, inputAction, true, null)),
                        	new Parameter(Params.Event.THREAD_INFO, getThreadInfo())}),
                StateMachineContextImpl.create());
        
        suspendThread();
    }
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.AbstractEventProcessorListener#beforeOnEnterActionExecution(com.evelopers.unimod.runtime.StateMachinePath, java.lang.String, java.lang.String)
     */
    public void beforeOnEnterActionExecution(StateMachineContext context, StateMachinePath path,
            String state, String outputAction) {

        handler.handleAndWait(
                new Event(E21, 
                        new Parameter[]{
                        	new Parameter(Params.Event.POSITION, new OnEnterActionExecutionPosition(context, path, state, outputAction, true)),
                        	new Parameter(Params.Event.THREAD_INFO, getThreadInfo())}), 
                StateMachineContextImpl.create());
    
        suspendThread();
    }
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.AbstractEventProcessorListener#beforeOutputActionExecution(com.evelopers.unimod.runtime.StateMachinePath, java.lang.String, java.lang.String)
     */
    public void beforeOutputActionExecution(StateMachineContext context, StateMachinePath path,
            String transition, String outputAction) {
        
        handler.handleAndWait(
                new Event(E21, 
                        new Parameter[]{
                        	new Parameter(Params.Event.POSITION, new OutputActionExecutionPosition(context, path, transition, outputAction, true)),
                        	new Parameter(Params.Event.THREAD_INFO, getThreadInfo())}),                        	
                StateMachineContextImpl.create());

        suspendThread();
    }
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.AbstractEventProcessorListener#beforeSubmachinesExecution(com.evelopers.unimod.core.stateworks.Event, com.evelopers.unimod.runtime.StateMachinePath, java.lang.String)
     */
    public void beforeSubmachineExecution(StateMachineContext context, Event e, StateMachinePath path,
            String state, String subMachine) {

        handler.handleAndWait(
                new Event(E21, 
                        new Parameter[]{
                        	new Parameter(Params.Event.POSITION, new SubmachinesExecutionPosition(context, path, e, state, subMachine, true)),
                        	new Parameter(Params.Event.THREAD_INFO, getThreadInfo())}),                        	
                StateMachineContextImpl.create());

        suspendThread();
    }

    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.AbstractEventProcessorListener#transitionCandidate(com.evelopers.unimod.runtime.StateMachinePath, java.lang.String, com.evelopers.unimod.core.stateworks.Event, java.lang.String)
     */
    public void transitionCandidate(StateMachineContext context, StateMachinePath path, String state,
            Event event, String transition) {

        handler.handleAndWait(
                new Event(E21, 
                        new Parameter[]{
                        	new Parameter(Params.Event.POSITION, new TransitionCandidatePosition(context, path, event, state, transition)),
                        	new Parameter(Params.Event.THREAD_INFO, getThreadInfo())}),                        	
                StateMachineContextImpl.create());
    
        suspendThread();
    }
    
    public void comeToState(StateMachineContext context, StateMachinePath path, String state) {
        handler.handleAndWait(
                new Event(E21, 
                        new Parameter[]{
                        	new Parameter(Params.Event.POSITION, new ComeToStatePosition(context, path, state)),
                        	new Parameter(Params.Event.THREAD_INFO, getThreadInfo())}),                        	
                StateMachineContextImpl.create());
    
        suspendThread();
    }
    
}
