/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.debug.debugger;

import java.io.IOException;

import org.apache.commons.logging.LogFactory;

import com.evelopers.common.exception.CommonException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.debug.ExceptionHandlerImpl;
import com.evelopers.unimod.debug.protocol.MessageCoder;
import com.evelopers.unimod.runtime.AbstractEventProcessorListener;
import com.evelopers.unimod.runtime.ControlledObject;
import com.evelopers.unimod.runtime.ControlledObjectsMap;
import com.evelopers.unimod.runtime.ModelEngine;
import com.evelopers.unimod.runtime.StateMachineConfig;
import com.evelopers.unimod.runtime.StateMachinePath;
import com.evelopers.unimod.runtime.context.StateMachineContext;
import com.evelopers.unimod.runtime.context.StateMachineContextImpl;
import com.evelopers.unimod.runtime.interpretation.InterpretationHelper;
import com.evelopers.unimod.runtime.logger.ShortLogger;
import com.evelopers.unimod.transform.xml.XMLToModel;

/**
 * ModelEngine remote debugger. Debugger-side.
 * 
 * @author vgurov
 */
public class Debugger {

    public static final String RESUMED = "Resumed";
    public static final String SUSPENDED = "Suspended";
    public static final String TERMINATED = "s2";
    
    private ModelEngine engine;
    private Model model;
    private DebuggerConnector connector;
    private UIManager uiManager;
    private BreakpointManager breakpointManager;

    public Debugger(String hostName, int port, MessageCoder coder, BreakpointManager breakpoint, UIManager ui) throws CommonException, IOException {
        this.breakpointManager = breakpoint;
        this.uiManager = ui;
        
        // setup statemachine engine
        // build model
        model = XMLToModel.loadAndCompile(getClass().getResourceAsStream("debugger.xml"));
        
        // create event providers and controlled objects
        connector = new DebuggerConnector(hostName, port, coder);

        // create runtime debugger engine
        engine = InterpretationHelper.getInstance().createBuildInModelEngine(
                model,
                new ControlledObjectsMap() {
                    public ControlledObject getControlledObject(String coName) {
                        if (coName.equals("o1")) {
                            return connector;
                        } else if (coName.equals("o2")) {
                            return uiManager;
                        } else if (coName.equals("o3")) {
                            return breakpointManager;
                        }
                        
                        throw new IllegalArgumentException("Unknown controlled object [" + coName + "]");
                    }
                },
                true);

        // add exception handler
        engine.getEventProcessor().addExceptionHandler(new ExceptionHandlerImpl());
        
        // stop all event providers when debugger comes to final state
        engine.getEventProcessor().addEventProcessorListener(new AbstractEventProcessorListener() {
            public void stateMachineCameToFinalState(StateMachineContext context, StateMachinePath path, StateMachineConfig config) {
                if (path.isRoot()) {
                    disposeEventProviders();
                }
            }
        });
        
        // add logger if needed
        // add log 
        if (LogFactory.getLog(getClass()).isDebugEnabled()) {
            engine.getEventProcessor().addEventProcessorListener(new ShortLogger(LogFactory.getLog(getClass())));
        }
    }
    
    public void start() throws CommonException {
        // init event providers  
        connector.init(engine);
        uiManager.init(engine);
        breakpointManager.init(engine);
        
        // send startup event
        engine.getEventManager().handle(new Event("e0"), StateMachineContextImpl.create());
    }
    
    protected void disposeEventProviders() {
        connector.dispose();
        breakpointManager.dispose();
        uiManager.dispose();
    }
    
    public void stop() {
        disposeEventProviders();
        engine.getEventManager().dispose();
    }
    
}
