/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.debug.debugger.console;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.evelopers.common.exception.CommonException;
import com.evelopers.common.util.helper.StringHelper;
import com.evelopers.unimod.contract.CoreContract;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.Transition;
import com.evelopers.unimod.debug.Params;
import com.evelopers.unimod.debug.debugger.BreakpointManager;
import com.evelopers.unimod.debug.debugger.UIManager;
import com.evelopers.unimod.debug.protocol.position.Position;
import com.evelopers.unimod.debug.protocol.position.TransitionCandidatePosition;
import com.evelopers.unimod.runtime.EventManager;
import com.evelopers.unimod.runtime.ModelEngine;
import com.evelopers.unimod.runtime.StateMachinePath;
import com.evelopers.unimod.runtime.context.Parameter;
import com.evelopers.unimod.runtime.context.StateMachineContext;
import com.evelopers.unimod.runtime.context.StateMachineContextImpl;

/**
 * @author vgurov
 */
public class ConsoleManager implements UIManager {

    private boolean inited = false;
    private EventManager handler;
    private Thread kb;
    private Model model;
    private Position[] positions;

    ConsoleManager(Model model) {
        this.model = model;
        initPositions();
    }
    
    private void initPositions() {
        List p = new ArrayList();
        
        for (Iterator iter = model.getStateMachines().iterator(); iter.hasNext();) {
            StateMachine sm = (StateMachine) iter.next();
            
            for (Iterator iterator = sm.getAllTransition().iterator(); iterator.hasNext();) {
                Transition t = (Transition) iterator.next();
                
                p.add(new TransitionCandidatePosition(null, new StateMachinePath(CoreContract.encode(sm)), t.getEvent(), CoreContract.encode(t.getSourceState()), CoreContract.encode(t)));
            }
        }
        
        positions = (Position[])p.toArray(new Position[p.size()]);
    }
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.debug.debugger.BreakpointManager#z1(com.evelopers.unimod.runtime.context.StateMachineContext)
     */
    public void z1(StateMachineContext context) throws CommonException {
       Position p = (Position)context.getEventContext().getParameter(Params.Event.POSITION);
       
       System.out.println("Suspended on breakpoint: ");
       System.out.println(p);
    }    

    /* (non-Javadoc)
     * @see com.evelopers.unimod.debug.debugger.UIManager#z2(com.evelopers.unimod.runtime.context.StateMachineContext)
     */
    public void z2(StateMachineContext context) throws CommonException {
        Position p = (Position)context.getEventContext().getParameter(Params.Event.POSITION);
        
        System.out.println("Suspended on step: ");
        System.out.println(p);
    }

    /* (non-Javadoc)
     * @see com.evelopers.unimod.debug.debugger.UIManager#z3(com.evelopers.unimod.runtime.context.StateMachineContext)
     */
    public void z3(StateMachineContext context) throws CommonException {
        System.out.println("App resumed");
    }

    /* (non-Javadoc)
     * @see com.evelopers.unimod.debug.debugger.UIManager#z4(com.evelopers.unimod.runtime.context.StateMachineContext)
     */
    public void z4(StateMachineContext context) throws CommonException {
        System.out.println("App has closed connection");
    }

    /* (non-Javadoc)
     * @see com.evelopers.unimod.debug.debugger.UIManager#z5(com.evelopers.unimod.runtime.context.StateMachineContext)
     */
    public void z5(StateMachineContext context) throws CommonException {
        System.out.println("App has sent unknown message");
    }
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.debug.debugger.UIManager#z6(com.evelopers.unimod.runtime.context.StateMachineContext)
     */
    public void z6(StateMachineContext context) throws CommonException {
        System.out.println("App has received unknown command from debugger");        
    }
    
    public void z7(StateMachineContext context) throws CommonException {
        System.out.println("New thread created");        
    }    
    
    public void z8(StateMachineContext context) throws CommonException {
        System.out.println("Can't update model in debugged engine");        
    }
    
    
    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.EventProvider#init(com.evelopers.unimod.runtime.ModelEngine)
     */
    public void init(ModelEngine engine) throws CommonException {
        if (inited) {
            return;
        }
        inited = true;
        
        handler = engine.getEventManager();
        kb = new ConsoleThread();
        kb.start();
     }

    /* (non-Javadoc)
     * @see com.evelopers.unimod.runtime.EventProvider#dispose()
     */
    public void dispose() {
		kb.interrupt();
    }
    
    private static String[][] COMMANDS = {{"add", BreakpointManager.E21}, {"remove", BreakpointManager.E22}, {"step", E23}, {"resume", E24}, {"disconnect", E25}, {"close", E26}};
    
    private class ConsoleThread extends Thread {
        
        ConsoleThread() {
            super("ConsoleDebugger.ConsoleThread");
        }

        public void run() {
            BufferedReader br = new BufferedReader(new InputStreamReader(System.in));

			while (!isInterrupted()) {
				try {
				    System.out.print("unimod> ");

				    while (!br.ready()) {
				        Thread.sleep(10);
				    }
				    
				    if (isInterrupted()) {
				        return;
				    }
				    
					String s = br.readLine();
					String[] s1 = StringHelper.splitString(s, " ");
					String event = getEvent(s1[0]);
					
					if (event == null) {
					    printHelp();
					    printPosiiblePositions();
					    continue;
					}
					
					if (event.equals(BreakpointManager.E21) || event.equals(BreakpointManager.E22)) {
					    Position p = parsePosition(s1[1]);
					    
					    handler.handle(
					            new Event(
					                    event, 
					                    new Parameter(Params.Event.BREAKPOINTS, new Position[] {p})), StateMachineContextImpl.create());
					} else {
					    handler.handle(
					            new Event(event), StateMachineContextImpl.create());
					
					}
					
				} catch (Exception e) {
					return;
				}
			}
        }
    }

    
    private String getEvent(String command) {
        for (int i = 0; i < COMMANDS.length; i++) {
            if (COMMANDS[i][0].equalsIgnoreCase(command)) {
                return COMMANDS[i][1];
            }
        }
        
        return null;
    }
    
    private void printHelp() {
        System.out.println("Unknown command. Possible commands are:");

        for (int i = 0; i < COMMANDS.length; i++) {
            System.out.println(COMMANDS[i][0]);
        }
    }
    
    private void printPosiiblePositions() {
        for (int i = 0; i < positions.length; i++) {
            System.out.println(i + ":" + positions[i]);
        }
    }
    
    private Position parsePosition(String s) {
        return positions[Integer.parseInt(s)];
    }
    
}
