/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.debug.protocol;

import com.evelopers.unimod.debug.protocol.position.Position;

/**
 * Event messages are sent by engine being debugged to debugger host
 * Posiible events: suspended on breakpoint, suspended on step, resumed
 */
public class EventMessage extends Message {

    public static final int THREAD_CREATED = 0;
    public static final int SUSPENDED_ON_BREAKPOINT = 1;
    public static final int SUSPENDED_ON_STEP = 2;
    public static final int RESUMED = 3;
    public static final int UNKNOWN_COMMAND = 4;
    public static final int CANT_UPDATE_MODEL = 5;

    /** suspend position */
    private Position position;
    
    /** info */
    private String info;

    /** suspened threads */
    private ThreadInfo[] suspendedThreads;
    
    private EventMessage(int type, Position position, ThreadInfo thread) {
        super(type, thread);
        this.position = position;
    }

    private EventMessage(int type, Position position, ThreadInfo thread, String info) {
        this(type, position, thread);
        this.info = info;
    }

    private EventMessage(int type, Position position, ThreadInfo thread, String info, ThreadInfo[] suspendedThreads) {
        this(type, position, thread, info);
        this.suspendedThreads = suspendedThreads;
    }
    
    public static EventMessage createSuspendedOnBreakpoint(Position position, ThreadInfo thread) {
        return new EventMessage(SUSPENDED_ON_BREAKPOINT, position, thread);
    }

    public static EventMessage createSuspendedOnStep(Position position, ThreadInfo thread) {
        return new EventMessage(SUSPENDED_ON_STEP, position, thread);
    }

    public static EventMessage createCreated(ThreadInfo thread) {
        return new EventMessage(THREAD_CREATED, null, thread);
    }
    
    public static EventMessage createResumed(ThreadInfo thread) {
        return new EventMessage(RESUMED, null, thread);
    }

    public static EventMessage createUnknownCommand(String info) {
        return new EventMessage(UNKNOWN_COMMAND, null, null, info);
    }

    public static EventMessage createCantUpdateModel(ThreadInfo thread, String info, ThreadInfo[] suspenedThreads) {
        return new EventMessage(CANT_UPDATE_MODEL, null, null, info, suspenedThreads);
    }
    
    public Position getPosition() {
        return position;
    }
    
    /**
     * @return Returns the info.
     */
    public String getInfo() {
        return info;
    }

	public ThreadInfo[] getSuspendedThreads() {
		return suspendedThreads;
	}
}
