/*
 * Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.debug.protocol;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.NotSerializableException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.debug.protocol.position.EventProcessingPosition;
import com.evelopers.unimod.debug.protocol.position.SubmachinesExecutionPosition;
import com.evelopers.unimod.debug.protocol.position.TransitionSelectionPosition;
import com.evelopers.unimod.runtime.context.Parameter;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * Uses java serialization to save objects into Strings.
 * Use it only for debugging and monitoring of Java applications.
 */
public class JavaSpecificMessageCoder implements MessageCoder {

    public Message decode(InputStream is) throws MessageCoderException {
        try {
            ObjectInputStream ois = new ObjectInputStream(is);

            return (Message)ois.readObject();
        } catch (Exception e) {
            throw new MessageCoderException(e, "Can't read message from stream");
        }
    }

    public void encode(Message m, OutputStream os) throws MessageCoderException {
        try {
            if (m instanceof EventMessage) {
                EventMessage ms = (EventMessage)m;
                
                makeSerializable(ms);
                
                if (writeEventMessage(ms, os) != null) {
                    clearEventParameters(ms);
                    
                    if (writeEventMessage(ms, os) != null) {
                        clearContext(ms);
                        
                        Exception ex;
                        if ((ex = writeEventMessage(ms, os)) != null) {
                            throw new MessageCoderException(ex, "Can't write EventMessage to stream");
                        }
                    }
                }
            } else {
                    ObjectOutputStream ous = new ObjectOutputStream(os);
                    ous.writeObject(m);
                    ous.flush();
            }        
        } catch (Exception e) {
            throw new MessageCoderException(e, "Can't write message to stream");
        }
    }
    
    protected Exception writeEventMessage(EventMessage m, OutputStream os) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream ous = new ObjectOutputStream(bos);

        try {
            ous.writeObject(m);
        } catch (NotSerializableException e) {
            return e;
        }

        ous.flush();
        os.write(bos.toByteArray());
        
        return null;
    }
    
    protected void makeSerializable(EventMessage ms) {
        
        if (ms.getPosition() != null) {
            // transform context to serializable copy
            ms.getPosition().setContext(makeSerializableCopy(ms.getPosition().getContext()));
        }
        
        // transform event parameters to serializable copy
        if (ms.getPosition() instanceof EventProcessingPosition) {
            ((EventProcessingPosition)ms.getPosition()).setEvent(
                    makeSerializableCopy(((EventProcessingPosition)ms.getPosition()).getEvent())
            );
        } if (ms.getPosition() instanceof SubmachinesExecutionPosition) {
            ((SubmachinesExecutionPosition)ms.getPosition()).setEvent(
                    makeSerializableCopy(((SubmachinesExecutionPosition)ms.getPosition()).getEvent())
            );
        } if (ms.getPosition() instanceof TransitionSelectionPosition) {
            ((TransitionSelectionPosition)ms.getPosition()).setEvent(
                    makeSerializableCopy(((TransitionSelectionPosition)ms.getPosition()).getEvent())
            );
        } 
    }
    
    protected void clearContext(EventMessage ms) {
        if (ms.getPosition() != null && ms.getPosition().getContext() != null) {
            ms.getPosition().setContext(null);
        }
    }
    
    protected void clearEventParameters(EventMessage ms) {
        if (ms.getPosition() instanceof EventProcessingPosition) {
            ((EventProcessingPosition)ms.getPosition()).setEvent(
                    new Event((((EventProcessingPosition)ms.getPosition()).getEvent()).getName()));
        } if (ms.getPosition() instanceof SubmachinesExecutionPosition) {
            ((SubmachinesExecutionPosition)ms.getPosition()).setEvent(
                    new Event((((SubmachinesExecutionPosition)ms.getPosition()).getEvent()).getName()));
        } if (ms.getPosition() instanceof TransitionSelectionPosition) {
            ((TransitionSelectionPosition)ms.getPosition()).setEvent(
                    new Event((((TransitionSelectionPosition)ms.getPosition()).getEvent()).getName()));
        } 
    }
    
    protected Event makeSerializableCopy(Event e) {
        if (e.getParameters() == null) {
            return e;
        }
        
        List p = new ArrayList();
        for (int i = 0; i < e.getParameters().length; i++) {
            /**@TODO: workaround the following problem: eclipse-side debugger engine  
             * can't find classes serialized classes that are come from client-side debugger
             */
            //if (!(e.getParameters()[i] instanceof Serializable)) {
                p.add(new Parameter(e.getParameters()[i].getName(), new NotSerializableObjectHandlerImpl(e.getParameters()[i].getValue()))); 
            //} else {
            //    p.add(e.getParameters()[i]);
            //}
        }
        
        return new Event(e.getName(), (Parameter[])p.toArray(new Parameter[p.size()]));
    }
    
    protected StateMachineContext makeSerializableCopy(StateMachineContext c) {
        if (c == null) {
            return null;
        }
        
        StateMachineContext ctx = new StateMachineContextImpl1(c);
        
        makeSerializableCopy(c.getEventContext(), ctx.getEventContext());
        makeSerializableCopy(c.getUserContext(), ctx.getUserContext());
        makeSerializableCopy(c.getApplicationContext(), ctx.getApplicationContext());
        
        return ctx;
    }
    
    /**
     * Replace all not serializable objects with it's serializable handles
     * 
     * @param ctx
     */
    protected void makeSerializableCopy(StateMachineContext.Context oldCtx, StateMachineContext.Context newCtx) {
        if (oldCtx == null) {
            return;
        }
        
        for (Enumeration i = oldCtx.getParameterNames(); i.hasMoreElements(); ) {
            String name = (String)i.nextElement();
            
            Object p = oldCtx.getParameter(name);
            /**@TODO: workaround the following problem: eclipse-side debugger engine  
             * can't find classes serialized classes that are come from client-side debugger
             */
            //if (!(p instanceof Serializable)) {
                newCtx.setParameter(name, new NotSerializableObjectHandlerImpl(p));
            //} else {
            //    newCtx.setParameter(name, p);
            //}
        }
    }
    
    public static class StateMachineContextImpl1 implements StateMachineContext, Serializable {
    	private Context appContext;
    	private Context userContext;
    	private Context eventContext;
    	
    	private StateMachineContextImpl1(StateMachineContext ctx) {
    	    if (ctx.getApplicationContext() != null) {
    	        this.appContext = new ContextImpl(ctx.getApplicationContext());
    	    }

    	    if (ctx.getUserContext() != null) {
    	        this.userContext = new ContextImpl(ctx.getUserContext());
    	    }

    	    if (ctx.getEventContext() != null) {
    	        this.eventContext = new ContextImpl(ctx.getEventContext());
    	    }
    	}

    	public Context getApplicationContext() {
    		return appContext;
    	}
      
    	public Context getUserContext() {
    		return userContext;
    	}

    	public Context getEventContext() {
    		return eventContext;
    	}

    	private static class ContextImpl implements NotSerializableObjectHandler, Context, Serializable {
      	
    		private Map m = new HashMap(); 
    		private String toString;
    		private String className;
    		
    		private ContextImpl(StateMachineContext.Context ctx) {
    		    toString = ctx.toString();
    		    className = ctx.getClass().getName(); 
    		}
    		
            public String getOriginalClassName() {
                return className;
            }
            
            public String getOriginalToString() {
                return toString;
            }
    		
    		public void setParameter(String name, Object data) {
    			m.put(name, data);
    		}
        
    		public Object getParameter(String name) {
    			return m.get(name);
    		}
        
    		public Object[] getParameterValues(String name) {
    			return null;
    		}
        
    		public Enumeration getParameterNames() {
    		    return Collections.enumeration(m.keySet());			
    		}
    	}
    }
    
    public static class NotSerializableObjectHandlerImpl implements Serializable, NotSerializableObjectHandler {
        
        private String toString;
        private String className;
        
        private NotSerializableObjectHandlerImpl(Object o) {
            if (o == null) {
                return;
            }
            
            toString = o.toString();
            className = o.getClass().getName();
        }
        
        /* (non-Javadoc)
         * @see java.lang.Object#toString()
         */
        public String getOriginalToString() {
            return toString;
        }
        
        public String getOriginalClassName() {
            return className;
        }
        
    }
    
    public interface NotSerializableObjectHandler {
        
        public String getOriginalClassName();
        
        public String getOriginalToString();
        
    }

}
