/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime;

import com.evelopers.common.exception.SystemException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.runtime.context.StateMachineContext;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

public abstract class AbstractEventProcessor implements EventProcessor {
    /**
     * List of plugged
     * {@link com.evelopers.unimod.runtime.EventProcessorListener
     * EventProcessorListeners}
     *
     * @associates <{EventProcessorListener}>
     * @supplierCardinality 0..*
     * @directed
     * @label listeners
     * @link aggregation
     */
    private List eventProcessorListeners = new LinkedList();

	/**
	 * @associates <{ExceptionHandler}>
	 * @link aggregation
	 * @supplierCardinality 0..* 
	 * @label handlers
	 */
    private List exceptionHandlers = new LinkedList();

    /**
     * @link aggregation
     * @supplierCardinality 1
     */
    private ControlledObjectsMap controlledObjectsMap;

    /**
     * @link aggregation
     * @supplierCardinality 1
     * @supplierRole modelStructure
     * @directed */
    /*#private ModelStructure modelStructure;*/

    /**
     * @directed
     * @supplierCardinality 1..**/
    /*#StateMachineConfigManager lnkStateMachineConfigManager;*/

    /* Event processing */
    /**
     * Processes given event in the given context using root
     * state machine instance.
     * <p/>
     * This method is used by {@link ModelEngine} to notify event processor
     * about events.
     *
     * @param event triggered event to process
     * @param context state machine context of the event
     */
    final public void process(Event event, StateMachineContext context) {
        try {
            process(event, context, getModelStructure().getRootPath());
        } catch (SystemException e) {
            fireException(context, e);
        }
    }

    /**
     * Processes given event using state machine instance defined by the given path.
     * <p/>
     * This method should be invoked for processing of event by included state machines.
     *
     * @param event triggered event to process
     * @param context state machine context of the event
     * @param path path that defines the state machine instance
     */
    final protected void process(Event event, StateMachineContext context, StateMachinePath path) {
        try {
            /* Load config for given statemachine path */
            StateMachineConfig config = loadConfig(path, context);

            /* If config doesn't exist */
            if (! configExists(config)) {
                /* Init config for given path */
                config = initConfig(path, context);
            }
            
            /* Event Processing Started */
            fireEventProcessingStarted(context, event, path, config);

            /* If initial config */
            if (isTopConfig(path, config)) {
                /* Transite to stable state */
                config = transiteToStableState(context, path, config);
                /* Store config for given path */
                storeConfig(path, context, config);
            }

            /* Implementation specific (Compilation / Interpretation) event processing */
            config = process(event, context, path, config);

            /* If final config */
            if (isFinalConfig(path, config)) {
                /* State Machine Came To Final State */
                fireStateMachineCameToFinalState(context, path, config);
            }
            
            /* Event Processing Finished */
            fireEventProcessingFinished(context, event, path, config);

            /* Store config for given path */
            storeConfig(path, context, config);

            /* If final config */
            if (isFinalConfig(path, config)) {
                /* Dispose config for given path */
                disposeConfig(path, context, config);
            }
        } catch (SystemException ex) {
            fireException(context, ex);
        }
    }

    final protected StateMachineConfig initConfig(StateMachinePath path, StateMachineContext context) throws EventProcessorException {
        /* Get config manager */
        String stateMachine = path.getStateMachine();
        StateMachineConfigManager configManager =
                getModelStructure().getConfigManager(stateMachine);

        /* Get top config */
        StateMachineConfig config = getModelStructure().getTopConfig(stateMachine);

        /* Init config manager */
        configManager.init(path, context, config);
        return config;
    }

    final protected StateMachineConfig loadConfig(StateMachinePath path, StateMachineContext context) throws EventProcessorException {
        /* Get config manager */
        String stateMachine = path.getStateMachine();
        StateMachineConfigManager configManager =
                getModelStructure().getConfigManager(stateMachine);

        /* Load config */
        StateMachineConfig config = configManager.load(path, context);
        
        return config;
    }

    final protected void storeConfig(StateMachinePath path, StateMachineContext context, StateMachineConfig config) throws EventProcessorException {
        /* Get config manager */
        String stateMachine = path.getStateMachine();
        StateMachineConfigManager configManager =
                getModelStructure().getConfigManager(stateMachine);

        /* Store config */
        configManager.store(path, context, config);
    }

    final protected void disposeConfig(StateMachinePath path, StateMachineContext context, StateMachineConfig config) throws EventProcessorException {
        /* Get config manager */
        String stateMachine = path.getStateMachine();
        StateMachineConfigManager configManager =
                getModelStructure().getConfigManager(stateMachine);

        /* Dispose config */
        configManager.dispose(path, context, config);
    }
    
    final protected boolean configExists(StateMachineConfig config) {
        return config != null;
    }

    final protected boolean isTopConfig(StateMachinePath path, StateMachineConfig config) throws EventProcessorException {
        String stateMachine = path.getStateMachine();
        StateMachineConfig topConfig = getModelStructure().getTopConfig(stateMachine);
        return topConfig.equals(config);
    }

    final protected boolean isFinalConfig(StateMachinePath path, StateMachineConfig config) throws EventProcessorException {
        String stateMachine = path.getStateMachine();
        return getModelStructure().isFinal(stateMachine, config);
    }
        
    /**
     * Processes given event using state machine instance defined by
     * the given config.
     * <p/>
     * Implementation of this method depends on approach. For instance
     * it may be an interpretator of the in-memory state machine model or
     * a precompiled <code>switch</code>-operator.
     * <p/>
     * Implementations of this method should fire event processing events
     * during execution.
     *
     * @param event triggered event to process
     * @param context state machine context of the event
     * @param path path to state machine instance
     * @param config state machine config that defines the
     * state machine instance
     * @return state machine config after event processing
     * @throws SystemException if some input or output action throws
     * an exception. This exception will be handled by added ExceptionHandlers
     * @see #addExceptionHandler(ExceptionHandler)
     * @see #removeExceptionHandler(ExceptionHandler)
     * @see #addEventProcessorListener(EventProcessorListener)
     * @see #removeEventProcessorListener(EventProcessorListener)
     */
    abstract protected StateMachineConfig process(
            Event event, StateMachineContext context,
            StateMachinePath path, StateMachineConfig config) throws SystemException;

    /**
     * Transites from initial state <code>config</code> to stable 
     * state. This method should:
     * <ol>
     *   <li>Get transition from current initial state</li>
     *   <li>Transite using this transition</li>
     *   <li>
     *     If target state is composite then
     *     <ol>
     *       <li>{@link #fireCompositeTargetState}</li>
     *       <li>Get initial state from the composite state</li>
     *       <li>{@link #fireComeToState} for new initial state</li>
     *       <li>Repeat {@link #transiteToStableState} with new initial state</li>
     *     </ol> 
     *   </li>
     * </ol>
     *  
     * @param context state machine context of the event
     * @param path path that defines the state machine instance
     * @param config initial state to transite from
     * @return stable state reached from the given initial
     */
    abstract protected StateMachineConfig transiteToStableState(
            StateMachineContext context,
            StateMachinePath path, StateMachineConfig config) throws SystemException;
    
    /* Model structure */
    /**
     * Returns the structure of the model.
     * @return the structure of the model
     */
    public abstract ModelStructure getModelStructure();

    /* Controlled objects mapping */
    /**
     * Sets map of controlled objects. This method is invoked by
     * {@link ModelEngine} from either
     * {@link ModelEngine#createStandAlone(EventManager, EventProcessor,
     * ControlledObjectsManager, EventProvidersManager) stand alone} and
     * {@link ModelEngine#createBuildIn(EventManager, EventProcessor,
     * ControlledObjectsMap) build-in} factory methods.
     *
     * @param controlledObjectsMap map of controlled objects
     */
    public void setControlledObjectsMap(ControlledObjectsMap controlledObjectsMap) {
        this.controlledObjectsMap = controlledObjectsMap;
    }

    /**
     * Returns map of controlled objects.
     *
     * @return map of controlled objects
     */
    protected ControlledObjectsMap getControlledObjectsMap() {
        return controlledObjectsMap;
    }

    /* Exception handling */
    /**
     * Add handler for exceptions raised during event processing.
     *
     * @param handler handler for exceptions raised during event processing
     */
    public void addExceptionHandler(ExceptionHandler handler) {
        exceptionHandlers.add(handler);
    }

    /**
     * Remove handler for exceptions raised during event processing.
     *
     * @param handler handler for exceptions raised during event processing
     */
    public void removeExceptionHandler(ExceptionHandler handler) {
        exceptionHandlers.remove(handler);
    }

    /**
     * Sends notification about exception to exception handlers
     * @param context of event that led to exception
     * @param e raised exception
     */
    protected void fireException(StateMachineContext context, SystemException e) {
        for (Iterator i = exceptionHandlers.iterator(); i.hasNext();) {
            ExceptionHandler handler = (ExceptionHandler) i.next();
            handler.handleException(context, e);
        }
    }

    /* Event processor events */
    /**
     * Adds eventProcessorListener to listeners list.
     *
     * @param eventProcessorListener eventProcessorListener to add
     */
    public void addEventProcessorListener(EventProcessorListener eventProcessorListener) {
        eventProcessorListeners.add(eventProcessorListener);
    }

    /**
     * Removess eventProcessorListener from listeners list.
     *
     * @param eventProcessorListener eventProcessorListener to remove
     */
    public void removeEventProcessorListener(EventProcessorListener eventProcessorListener) {
        eventProcessorListeners.remove(eventProcessorListener);
    }

    protected void fireEventProcessingStarted(final StateMachineContext context, Event e, StateMachinePath path, StateMachineConfig c) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.eventProcessingStarted(context, e, path, c);
        }
    }

    protected void fireEventProcessingFinished(final StateMachineContext context, Event e, StateMachinePath path, StateMachineConfig c) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.eventProcessingFinished(context, e, path, c);
        }
    }

    protected void fireStateMachineCameToFinalState(final StateMachineContext context, StateMachinePath path, StateMachineConfig config) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.stateMachineCameToFinalState(context, path, config);
        }
    }

    protected void fireBeforeSubmachineExecution(final StateMachineContext context, Event e, StateMachinePath path, String state, String subMachine) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.beforeSubmachineExecution(context, e, path, state, subMachine);
        }
    }

    protected void fireAfterSubmachineExecution(final StateMachineContext context, Event e, StateMachinePath path, String state, String subMachine) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.afterSubmachineExecution(context, e, path, state, subMachine);
        }
    }

    protected void fireBeforeOnEnterActionExecution(final StateMachineContext context, StateMachinePath path, String stateName, String outputAction) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.beforeOnEnterActionExecution(context, path, stateName, outputAction);
        }
    }

    protected void fireAfterOnEnterActionExecution(final StateMachineContext context, StateMachinePath path, String stateName, String outputAction) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.afterOnEnterActionExecution(context, path, stateName, outputAction);
        }
    }

    protected void fireBeforeOutputActionExecution(final StateMachineContext context, StateMachinePath path, String transition, String outputAction) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.beforeOutputActionExecution(context, path, transition, outputAction);
        }
    }

    protected void fireAfterOutputActionExecution(final StateMachineContext context, StateMachinePath path, String transition, String outputAction) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.afterOutputActionExecution(context, path, transition, outputAction);
        }
    }

    protected void fireBeforeInputActionExecution(final StateMachineContext context, StateMachinePath path, String transition, String inputAction) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.beforeInputActionExecution(context, path, transition, inputAction);
        }
    }

    protected void fireAfterInputActionExecution(final StateMachineContext context, StateMachinePath path, String transition, String inputActionName, Object value) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.afterInputActionExecution(context, path, transition, inputActionName, value.toString());
        }
    }

    protected void fireTransitionCandidate(final StateMachineContext context, StateMachinePath path, String state, Event event, String transition) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.transitionCandidate(context, path, state, event, transition);
        }
    }

    protected void fireTransitionsOfSuperstate(final StateMachineContext context, StateMachinePath path, String superstate, Event event) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.transitionsOfSuperstate(context, path, superstate, event);
        }
    }

    protected void fireTransitionFound(final StateMachineContext context, StateMachinePath path, String state, Event event, String transition) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.transitionFound(context, path, state, event, transition);
        }
    }

    protected void fireTransitionNotFound(final StateMachineContext context, StateMachinePath path, String state, Event event) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.transitionNotFound(context, path, state, event);
        }
    }

    protected void fireEventSkipped(final StateMachineContext context, StateMachinePath path, String state, Event event) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.eventSkipped(context, path, state, event);
        }
    }

    protected void fireCompositeTargetState(final StateMachineContext context, StateMachinePath path, String state) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.compositeTargetState(context, path, state);
        }
    }

    protected void fireComeToState(final StateMachineContext context, StateMachinePath path, String state) {
        for (Iterator i = eventProcessorListeners.iterator(); i.hasNext();) {
            EventProcessorListener l = (EventProcessorListener) i.next();
            l.comeToState(context, path, state);
        }
    }
}
