/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime;

import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * <p>
 * Interface for pluggable runtime units.
 * </p>
 * <p><strong>Note, that all String parameters must sutisfy {@link com.evelopers.unimod.contract.CoreContract}, i.e.
 * there must be ability to convert them into real model objects using methods of class {@link com.evelopers.unimod.contract.CoreContract}</strong></p> 
 *
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public interface EventProcessorListener {

    /**
     * Event processor started processing of event <code>e</code>.
     * @param e event that will be processed
     * @param path
     * @param c config of state machine before event processing
     */
    void eventProcessingStarted(final StateMachineContext context, Event e, StateMachinePath path, StateMachineConfig c);

    /**
     * Event processor finished processing of event <code>e</code>.
     * @param e processed event
     * @param path
     * @param c config of state machine after event processing
     */
    void eventProcessingFinished(final StateMachineContext context, Event e, StateMachinePath path, StateMachineConfig c);

    /**
     * Root state machine came to final state.
     */
    void stateMachineCameToFinalState(final StateMachineContext context, StateMachinePath path, StateMachineConfig config);

    /**
     * Submachines of the active state will be executed with event
     * <code>e</code>.
     * @param e event that will be processed by submachines
     * @param path
     * @param stateName config of state machine before submachines execution
     * @param subMachine submachine that is about to execute
     */
    void beforeSubmachineExecution(final StateMachineContext context, Event e, StateMachinePath path, String state, String subMachine);

    /**
     * Submachines of the active state was executed with event
     * <code>e</code>.
     * @param event event that was processed by submachines
     * @param path
     * @param stateName config of state machine after submachines execution
     * @param subMachine submachine that was executed
     */
    void afterSubmachineExecution(final StateMachineContext context, Event event, StateMachinePath path, String state, String subMachine);

    /**
     * Output action will be executed
     * @param path
     * @param stateName
     * @param outputActionName name of output action that will be executed
     */
    void beforeOnEnterActionExecution(final StateMachineContext context, StateMachinePath path, String state, String outputAction);

    /**
     * Output action was executed
     * @param path
     * @param stateName
     * @param outputActionName name of output action that was executed
     */
    void afterOnEnterActionExecution(final StateMachineContext context, StateMachinePath path, String state, String outputAction);

    /**
     * Output action will be executed
     * @param path
     * @param transition
     * @param outputActionName name of output action that will be executed
     */
    void beforeOutputActionExecution(final StateMachineContext context, StateMachinePath path, String transition, String outputAction);

    /**
     * Output action was executed
     * @param path
     * @param transition
     * @param outputActionName name of output action that was executed
     */
    void afterOutputActionExecution(final StateMachineContext context, StateMachinePath path, String transition, String outputAction);

    /**
     * Input action will be executed
     * @param path
     * @param transition
     * @param inputActionName name of input action that will be executed
     */
    void beforeInputActionExecution(final StateMachineContext context, StateMachinePath path, String transition, String inputAction);

    /**
     * Input action was executed
     * @param path
     * @param transition
     * @param inputActionName name of input action that was executed
     * @param value calculated value of input action
     */
    void afterInputActionExecution(final StateMachineContext context,
            StateMachinePath path, String transition, String inputAction, String value);

    /**
     * Examine transition if it may be selected as transition to go
     * @param path
     * @param state source state's name
     * @param event trigged event
     * @param transition transition to examine
     */
    void transitionCandidate(final StateMachineContext context, StateMachinePath path, String state, Event event, String transition);

    /**
     * Transition to go was not found, try transitions of superstate
     * @param path
     * @param superstate super state's name
     * @param event trigged event
     */
    void transitionsOfSuperstate(final StateMachineContext context, StateMachinePath path, String superstate, Event event);

    /**
     * Transition to go was found
     * @param path
     * @param state source state's name
     * @param event trigged event
     * @param transition transition to go
     */
    void transitionFound(final StateMachineContext context, StateMachinePath path, String state, Event event, String transition);

    /**
     * Transition to go was not found. Bad model.
     * @param path
     * @param state source state's name
     * @param event trigged event
     */
    void transitionNotFound(final StateMachineContext context, StateMachinePath path, String state, Event event);

    /**
     * Transition to go was not found. Given event isn't handled in the state or its superstates. 
     * And transitions for ANY event outgoing from the state or its superstates. So transition is
     * internal and event is skipped. 
     * @param path path to state machine containing the state
     * @param state source state's name
     * @param event trigged event
     */
    void eventSkipped(final StateMachineContext context, StateMachinePath path, String state, Event event);

    /**
     * Target state is composite
     * @param path
     * @param state target state's name
     */
    void compositeTargetState(final StateMachineContext context, StateMachinePath path, String state);
    
    /**
     * Come to some state
     * 
     * @param path
     * @param state
     */
    void comeToState(final StateMachineContext context, StateMachinePath path, String state);
}
