/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime;

import com.evelopers.common.exception.CommonException;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * Entry point for state machine runtime engine.
 * 
 * @author Vadim Gurov
 * @author Maxim Mazin
 * @version $Revision: 1$
 */
public class ModelEngine {
    /**
     * @link aggregation
     * @label event source 
     */
	private EventManager eventManager;

    /**
     * @link aggregation 
     */
    private EventProcessor eventProcessor;

    /**
     * @link aggregation
     * @supplierCardinality 0..1 
     */
    private ControlledObjectsManager controlledObjectsManager;

    /**
     * @link aggregation
     * @supplierCardinality 0..1 
     */
    private EventProvidersManager eventProvidersManager = null;

    /**
     * After creating new stand alone engine you have to {@link #start()} it
     * 
     * @param eventManager
     * @param eventProcessor
     * @param coManager
     * @param epManager
     * @return
     * @throws CommonException
     */
    public static ModelEngine createStandAlone(
            EventManager eventManager,
            EventProcessor eventProcessor,
            ControlledObjectsManager coManager,
            EventProvidersManager epManager)  throws CommonException {
        final ModelEngine modelEngine = new ModelEngine(eventManager, eventProcessor, coManager, epManager);

        coManager.init(modelEngine);
        eventProcessor.setControlledObjectsMap(coManager);

        modelEngine.eventManager.init(modelEngine);

        // moved to method start()
        // modelEngine.eventProvidersManager.init(modelEngine);

        modelEngine.eventProcessor.addEventProcessorListener(new AbstractEventProcessorListener() {
            public void stateMachineCameToFinalState(StateMachineContext context, StateMachinePath path,
                    StateMachineConfig config) {
                if (path.isRoot()) {
                    modelEngine.eventProvidersManager.dispose();
                    modelEngine.eventManager.dispose();
                    modelEngine.controlledObjectsManager.dispose();
                }
            }
        });

        return modelEngine;
    }

    /**
     * For stand alone model inits all event providers, so they 
     * starts providing event. For build-in do nothing
     *
     */
    public void start() throws CommonException {
        if (eventProvidersManager != null) {
            eventProvidersManager.init(this);
        }
    }
    
    /**
     * After creating new build-in engine you have to start your event 
     * providers manually. Doesn't try to call {@link #start()} - it's actual 
     * for stand alone engine only.
     * 
     * @param eventManager
     * @param eventProcessor
     * @param controlledObjectsMap
     * @return
     * @throws CommonException
     */
    public static ModelEngine createBuildIn(
            EventManager eventManager,
            EventProcessor eventProcessor,
            ControlledObjectsMap controlledObjectsMap) throws CommonException {
        final ModelEngine modelEngine = new ModelEngine(eventManager, eventProcessor);

        eventManager.init(modelEngine);
        eventProcessor.setControlledObjectsMap(controlledObjectsMap);

        modelEngine.eventProcessor.addEventProcessorListener(new AbstractEventProcessorListener() {
            public void stateMachineCameToFinalState(StateMachineContext context, StateMachinePath path,
                    StateMachineConfig config) {
                if (path.isRoot()) {
                    modelEngine.eventManager.dispose();
                }
            }
        });

        return modelEngine;
    }
    
    public static ModelEngine createBuildIn(
            EventManager eventManager,
            EventProcessor eventProcessor) throws CommonException {
        final ModelEngine modelEngine = new ModelEngine(eventManager, eventProcessor);

        modelEngine.eventManager.init(modelEngine);

        modelEngine.eventProcessor.addEventProcessorListener(new AbstractEventProcessorListener() {
            public void stateMachineCameToFinalState(StateMachineContext context, StateMachinePath path,
                    StateMachineConfig config) {
                if (path.isRoot()) {
                    modelEngine.eventManager.dispose();
                }
            }
        });

        return modelEngine;
    }

    /**
     * Constructs standalone runtime model engine.
     *
     * @param eventManager manages events passing them to method {@link #process}
     * @param eventProcessor processes events using some state machine representation
     * @param controlledObjectsManager manages controlled objects life cycle
     * @param eventProvidersManager manages event providers life cycle
     */
    private ModelEngine(
            EventManager eventManager,
            EventProcessor eventProcessor,
            ControlledObjectsManager controlledObjectsManager,
            EventProvidersManager eventProvidersManager) {
        this.eventManager = eventManager;
        this.eventProcessor = eventProcessor;
        this.controlledObjectsManager = controlledObjectsManager;
        this.eventProvidersManager = eventProvidersManager;
    }

    /**
     * Constructs build-in runtime model engine.
     *
     * @param eventManager manages events passing them to method {@link #process}
     * @param eventProcessor processes events using some state machine representation
     */
	private ModelEngine(
            EventManager eventManager,
            EventProcessor eventProcessor) {
        this.eventManager = eventManager;
        this.eventProcessor = eventProcessor;
        this.controlledObjectsManager = null;
        this.eventProvidersManager = null;
    }

    /**
     * Delegates call to {@link AbstractEventProcessor#process}.
     * <p/>
     * This method should be used ONLY by {@link EventManager event manager}.
     *
     * @param event event to process
     * @param context current state machine context
     */
    void process(Event event, StateMachineContext context) {
        eventProcessor.process(event, context);
    }

    public EventManager getEventManager() {
        return eventManager;
    }
    
    /**
     * Returns event processor passed to factory method.
     * @return event processor passed to factory method
     */
    public EventProcessor getEventProcessor() {
        return eventProcessor;
    }
    
    /**
     * @return Returns the eventProvidersManager.
     */
    public EventProvidersManager getEventProvidersManager() {
        return eventProvidersManager;
    }
    
    /**
     * @return Returns the controlledObjectsManager.
     */
    public ControlledObjectsManager getControlledObjectsManager() {
        return controlledObjectsManager;
    }
}
