/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.runtime.interpretation;

import com.evelopers.common.util.helper.ClassHelper;
import com.evelopers.common.util.helper.StringHelper;
import com.evelopers.unimod.resources.Messages;
import com.evelopers.unimod.runtime.ControlledObject;
import com.evelopers.unimod.runtime.EventProcessorException;
import com.evelopers.unimod.runtime.EventProvider;
import com.evelopers.unimod.runtime.StateMachineConfigManager;

/**
 * Helps to instantiate {@link com.evelopers.unimod.runtime.ControlledObject ControlledObjects},
 * {@link com.evelopers.unimod.runtime.EventProvider EventProviders} and
 * {@link com.evelopers.unimod.runtime.StateMachineConfigManager StateMachineConfigManagers}
 *
 * @author Maxim Mazin
 */
public class InstantiateHelper {

    /**
     * Instantiates target ControlledObject implementation.
     *
     * @return instantiated object
     * @throws EventProcessorException if class couldn't be casted to ControlledObject or
     * object couldn't be created.
     */
    public static ControlledObject getControlledObject(String implementationClassName) throws EventProcessorException {
        return (ControlledObject) getObject(implementationClassName, ControlledObject.class);
    }

    /**
     * Instantiates target EventProvider implementation.
     *
     * @return instantiated object
     * @throws EventProcessorException if class couldn't be casted to EventProvider or
     * object couldn't be created.
     */
    public static EventProvider getEventProvider(String implementationClassName) throws EventProcessorException {
        return (EventProvider) getObject(implementationClassName, EventProvider.class);
    }

    /**
     * Instantiates target StateMachineConfigManager implementation.
     *
     * @return instantiated object
     * @throws EventProcessorException if class couldn't be casted to StateMachineConfigManager or
     * object couldn't be created.
     */
    public static StateMachineConfigManager getStateMachineConfigManager(String implementationClassName) throws EventProcessorException {
        return (StateMachineConfigManager) getObject(implementationClassName, StateMachineConfigManager.class);
    }

    /**
     * Instantiates target implementation object.
     *
     * @return instantiated object
     * @throws EventProcessorException if class couldn't be casted to generalClass or
     * object couldn't be created.
     */
    public static Object getObject(String implementationClassName, Class generalClass) throws EventProcessorException {
        try {
            Class implementationClass = ClassHelper.loadClass(implementationClassName);
            if (!generalClass.isAssignableFrom(implementationClass)) {
                String message = Messages.getMessages().getMessage(
                        Messages.INCORRECT_OBJECT,
                        new Object[] {
                            implementationClassName,
                            generalClass.getName()
                        });
                throw new EventProcessorException(message);
            }
            return implementationClass.newInstance();
        } catch (EventProcessorException e) {
            throw e;
        } catch (Exception e) {
            String message = Messages.getMessages().getMessage(
                    Messages.CREATE_OBJECT_ERROR,
                    new Object[] {
                        implementationClassName,
                        e.getMessage(),
                        StringHelper.stackTraceToString(e)
                    });
            throw new EventProcessorException(message);
        }
    }
}