package com.evelopers.unimod.runtime.logger;

import java.text.MessageFormat;

import org.apache.commons.logging.Log;

import com.evelopers.common.util.helper.StringHelper;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.runtime.AbstractEventProcessorListener;
import com.evelopers.unimod.runtime.StateMachineConfig;
import com.evelopers.unimod.runtime.StateMachinePath;
import com.evelopers.unimod.runtime.context.StateMachineContext;

/**
 * @author mmazin
 */
public class ShortLogger extends AbstractEventProcessorListener {
	private static final MessageFormat EPS = new MessageFormat("{0} -> {1}:{2} '{'");  
	private static final MessageFormat EPF = new MessageFormat("'}' {1}:{2}");  
	private static final MessageFormat BOEAE = new MessageFormat("   {2} '{'");  
	private static final MessageFormat AOEAE = new MessageFormat("   '}'");  
	private static final MessageFormat BOAE = new MessageFormat("   {2} '{'");  
	private static final MessageFormat AOAE = new MessageFormat("   '}'");  
	private static final MessageFormat BIAE = new MessageFormat("   {2} '{'");  
	private static final MessageFormat AIAE = new MessageFormat("   '}' = {3}");  
	private static final MessageFormat TF = new MessageFormat("   {3}");  
	private static final MessageFormat TNF = new MessageFormat("  {2} -> {0}:{1} No transition found!");  
	
	private Log logger;
	
	public ShortLogger(Log logger) {
		this.logger = logger;
	}

    public void eventProcessingStarted(
            StateMachineContext context, Event e, StateMachinePath path, StateMachineConfig c) {
        logger.info(getShift(path) + EPS.format(new Object[]{e, path, c}));
    }

    public void eventProcessingFinished(StateMachineContext context, Event e, StateMachinePath path, StateMachineConfig c) {
        logger.info(getShift(path) + EPF.format(new Object[]{e, path, c}));
        
        if (path.isRoot()) {
            logger.info("\n");
        }
    }

    public void beforeOnEnterActionExecution(
            StateMachineContext context, StateMachinePath path, String state, String outputAction) {
        logger.info(getShift(path) + BOEAE.format(new Object[]{path, state, outputAction}));
    }

	public void afterOnEnterActionExecution(StateMachineContext context, StateMachinePath path,
			String state, String outputAction) {
		logger.info(getShift(path) + AOEAE.format(new Object[]{path, state, outputAction}));
	}

	public void beforeOutputActionExecution(StateMachineContext context, StateMachinePath path,
			String transition, String outputAction) {
		transition = translateTransition(transition);
		logger.info(getShift(path) + BOAE.format(new Object[]{path, transition, outputAction}));
	}

	public void afterOutputActionExecution(StateMachineContext context, StateMachinePath path,
			String transition, String outputAction) {
		transition = translateTransition(transition);
		logger.info(getShift(path) + AOAE.format(new Object[]{path, transition, outputAction}));
	}

	public void beforeInputActionExecution(StateMachineContext context, StateMachinePath path,
			String transition, String inputAction) {
		transition = translateTransition(transition);
		logger.info(getShift(path) + BIAE.format(new Object[]{path, transition, inputAction}));
	}

	public void afterInputActionExecution(StateMachineContext context, StateMachinePath path,
			String transition, String inputAction, String value) {
		transition = translateTransition(transition);
		logger.info(getShift(path) + AIAE.format(new Object[]{path, transition, inputAction, value}));
	}

	public void transitionFound(StateMachineContext context, StateMachinePath path, String state,
			Event event, String transition) {
		transition = translateTransition(transition);		
		logger.info(getShift(path) + TF.format(new Object[]{path, state, event, transition}));
	}

	public void transitionNotFound(StateMachineContext context, StateMachinePath path, String state,
			Event event) {
		logger.info(getShift(path) + TNF.format(new Object[]{path, state, event}));
	}
	
	private static final String EMPTY = "                             ";
	
	protected String getShift(StateMachinePath path) {
	    int k = 0;
	    
	    while ((path = path.getSuperStateMachinePath()) != null) {k++;};
	    
	    return EMPTY.substring(0, k);
	}	

	protected String translateTransition(String transition) {
		String[] tokens = StringHelper.splitString(transition, "#");
		return tokens[0] + " -> " + tokens[1];
	}
}
